
#' Calculate errors for elliptical distributions
#'
#' @name elliptical_errors
#' @title Calculate errors for elliptical distributions
#' @description This function calculates errors for elliptical distributions, including Elliptical-Normal Distribution and Elliptical-t Distribution.
#' @usage elliptical_errors(data, distribution = "normal", df = NULL)
#' @param data Matrix of data following an elliptical distribution.
#' @param distribution Type of elliptical distribution ("normal" or "t").
#' @param df Degrees of freedom for Elliptical-t Distribution (required if distribution is "t").
#' @return A list containing error metrics for the specified elliptical distribution.
#' @export
#' @examples
#' set.seed(123)
#' n <- 100; p <- 5
#' data_normal <- MASS::mvrnorm(n, rep(0, p), diag(p))
#' errors_normal <- elliptical_errors(data_normal, distribution = "normal")
#'
#' data_t <- matrix(stats::rt(n * p, df = 5), nrow = n, ncol = p)
#' errors_t <- elliptical_errors(data_t, distribution = "t", df = 5)
#' print(errors_normal)
#' print(errors_t)
elliptical_errors <- function(data, distribution = "normal", df = NULL) {
  if (!(distribution %in% c("normal", "t"))) {
    stop("Distribution must be either 'normal' or 't'.")
  }
  
  if (distribution == "t" && is.null(df)) {
    stop("Degrees of freedom (df) must be provided for Elliptical-t Distribution.")
  }
  
  errors <- list()
  errors$mean_error <- mean(data)
  
  if (distribution == "normal") {
    errors$normal_specific_error <- sum(data^2) / length(data)
  } else if (distribution == "t") {
    errors$t_specific_error <- sum(abs(data)) / length(data)
  }
  
  return(errors)
}
