#' Plot Methods
#'
#' @description
#' These `plot` methods for different results, including:
#' \itemize{
#'  \item Hull plot for \code{\link[EFAfactors]{Hull}} results.
#'  \item Comparison Data plot for \code{\link[EFAfactors]{CD}} results.
#'  \item Parallel Analysis plot for \code{\link[EFAfactors]{PA}} results.
#'  \item Empirical Kaiser Criterion for \code{\link[EFAfactors]{EKC}} results.
#'  \item Kaiser-Guttman Criterion for \code{\link[EFAfactors]{KGC}} results.
#'  \item K-means plot for \code{\link[EFAfactors]{EFAkmeans}} results.
#'  \item Hierarchical Clustering plot for \code{\link[EFAfactors]{EFAhclust}} results.
#'  \item pre-trained Neural Networks plot for \code{\link[EFAfactors]{NN}} results.
#'  \item Factor Forest for \code{\link[EFAfactors]{FF}} results.
#'  \item Comparison Data Forest plot for \code{\link[EFAfactors]{CDF}} results.
#'  \item Voting Method plot for \code{\link[EFAfactors]{EFAvote}} results.
#'  \item Scree Plot for \code{\link[EFAfactors]{EFAscreet}} results.
#'  \item Minimum Average Partial Test plot for \code{\link[EFAfactors]{MAP}} results.
#'  \item Scree Test Optimal Coordinate plot for \code{\link[EFAfactors]{STOC}} results.
#' }
#'
#' @param x An object of class \code{Hull}, \code{CD}, \code{PA}, \code{EKC},
#'          \code{KGC}, \code{EFAkmeans}, \code{EFAhclust}, \code{NN}, \code{FF},
#'          \code{CDF}, \code{EFAvote}, \code{EFAscreet}, \code{MAP},  or \code{STOC}.
#' @param ... Additional arguments passed to the plotting functions.
#' @return None. Plots are produced as side effects.
#' @seealso \code{\link[EFAfactors]{Hull}}, \code{\link[EFAfactors]{CD}}, \code{\link[EFAfactors]{PA}},
#'          \code{\link[EFAfactors]{EKC}}, \code{\link[EFAfactors]{KGC}}, \code{\link[EFAfactors]{EFAkmeans}},
#'          \code{\link[EFAfactors]{EFAhclust}}, \code{\link[EFAfactors]{NN}}, \code{\link[EFAfactors]{FF}},
#'          \code{\link[EFAfactors]{CDF}}, \code{\link[EFAfactors]{EFAvote}}, \code{\link[EFAfactors]{EFAscreet}},
#'          \code{\link[EFAfactors]{MAP}}, \code{\link[EFAfactors]{STOC}}
#' @name plot
NULL

#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text title
#' @describeIn plot Plot method for Hull objects
#' @export
plot.Hull <- function(x, ...) {
  obj <- x

  nfact <- obj$nfact
  CFI <- obj$CFI
  df <- obj$df
  Hull.CFI <- obj$Hull.CFI
  Hull.df <- obj$Hull.df

  index <- obj$index
  if(nfact != 0){

    posi <- c()
    for(i in 1:length(Hull.CFI)){
      for(j in 1:length(CFI)){
        if(Hull.CFI[i] == CFI[j]){
          if(!is.null(unique(posi)) && any(rep(j, length(posi)) == posi)){
            next
          }else{
            posi <- c(posi, j)
            break
          }
        }
      }
    }

    par(mar = c(5, 4, 5, 2) + 0.1)  # c(bottom, left, top, right)

    plot(Hull.df, Hull.CFI,
         type = "o",
         xlab = "Degree of freedom (parameters)",
         ylab = "Comparative Fit Index",
         pch = 2,
         col = "blue")

    points(df[-posi], CFI[-posi], pch = 1, col = "black")
    if(nfact >= 1)
      points(df[posi[1:which(posi - 1 == nfact)]], CFI[posi[1:which(posi - 1 == nfact)]], pch = 17, col = "blue")

    axis(3, at = df, labels = c(0:(length(df)-1)), line = 0, tick = TRUE)
    title(main = "Hull Plot", line = 3.5)
    mtext("Number of Factors", side = 3, line = 2)
  }else{
    message("Hull exception !")
  }
}

#'
#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for CD objects
#' @export
plot.CD <- function(x, ...) {
  obj <- x

  Sig <- obj$Sig
  RMSE.Eigs <- obj$RMSE.Eigs
  nfact <- obj$nfact

  x.max <- ifelse(Sig, nfact, nfact + 1)
  ys <- apply(RMSE.Eigs[, 1:x.max], 2, mean)

  plot(x = 1:x.max, y = ys, ylim = c(0, max(ys)),
       xlab = "Number of Factors", ylab = "RMSE Eigenvalue", type = "o",
       main = "Comparison Data",
       pch = 2,
       col = "blue")

  if(nfact >= 1)
    points(1:nfact, ys[1:nfact], pch = 17, col = "blue")

}

#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for PA objects
#' @export
plot.PA <- function(x, ...) {
  obj <- x

  nfact <- obj$nfact
  eigen.value <- obj$eigen.value
  eigen.ref <- obj$eigen.ref
  type <- obj$type
  fa <- obj$fa
  if(fa == "pc"){
    fa <- "Principal Component Analysis"
  }else{
    fa <- "Factor Analysis"
  }

  if(type == "quant"){
    type = paste0(type, "=", obj$quant)
  }

  x_axis <- 1:length(eigen.value)
  plot(x_axis, eigen.value, type = "o", col = "blue",
       ylim = range(c(eigen.value, eigen.ref)),
       pch = 2,
       xlab = "Number of Factors", ylab = paste0("Eigenvalue of ", fa), main = paste("Parallel Analysis Scree Plots", paste0("(", type, ")")))

  lines(x_axis, eigen.ref, col = "red", lty = 2)
  if(nfact >= 1)
    points(1:nfact, eigen.value[1:nfact], pch = 17, col = "blue")

  legend("topright", legend = c("Actual Eigenvalue", "Reference Eigenvalue"),
         col = c("blue", "red"), pch = c(17, 46), lty = 1:2)
  abline(h = 1, col = "black", lty = 2)

}

#'
#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for EKC objects
#' @export
plot.EKC <- function(x, ...) {
  obj <- x

  nfact <- obj$nfact
  eigen.value <- obj$eigen.value
  eigen.ref <- obj$eigen.ref

  x_axis <- 1:length(eigen.value)
  plot(x_axis, eigen.value, type = "o", col = "blue",
       ylim = range(c(eigen.value, eigen.ref)),
       pch = 2,
       xlab = "Number of Factors", ylab = "Eigenvalue", main = "Empirical Kaiser Criterion")

  lines(x_axis, eigen.ref, col = "red", lty = 2)
  if(nfact >= 1)
    points(1:nfact, eigen.value[1:nfact], pch = 17, col = "blue")

  legend("topright", legend = c("Actual Eigenvalue", "Reference Eigenvalue"),
         col = c("blue", "red"), pch = c(17, 46), lty = 1:2)
  abline(h = 1, col = "black", lty = 2)

}

#'
#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for KGC objects
#' @export
plot.KGC <- function(x, ...) {
  obj <- x

  eigen.value <- obj$eigen.value
  nfact <- obj$nfact

  x_axis <- 1:length(eigen.value)
  plot(x_axis, eigen.value, type = "o", col = "blue",
       ylim = range(c(eigen.value)),
       pch = 2,
       xlab = "Number of Factors", ylab = "Eigenvalue", main = "Kaiser-Guttman Criterion")

  abline(h = 1, col = "black", lty = 2)

  if(nfact >= 1)
    points(1:nfact, eigen.value[1:nfact], pch = 17, col = "blue")

}

#'
#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for EFAkmeans objects
#' @export
plot.EFAkmeans <- function(x, ...) {
  obj <- x

  wss <- obj$wss
  x_axis <- 1:length(wss)
  plot(x_axis, wss, type = "o", col = "blue",
       ylim = range(c(wss)),
       pch = 17,
       xlab = "Number of Factors", ylab = "Within-cluster Sum of Squares", main = "EFAkmeans")

}

#'
#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for EFAhclust objects
#' @export
plot.EFAhclust <- function(x, ...) {
  obj <- x

  hc <- obj$hc
  plot(hc, main = "Clusters of Items", xlab = "Items", ylab = "Dissimilarity ")

}

#'
#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for NN objects
#' @export
plot.NN <- function(x, ...) {
  obj <- x

  probability <- obj$probability
  bp <- barplot(probability,
                beside = TRUE,
                col = gray.colors(ncol(probability)),
                xlab = "Number of Factor",
                ylab = "Probability",
                main = paste0(obj$model, "\nProbability Distribution of the Number of Factors"),
                ylim = c(0, max(probability) * 1.1),
                las = 2)

  text(x = bp, y = probability, label = round(probability, digits = 3),
       pos = 3, cex = 0.8, col = "black")
  axis(1, at = bp, labels = 1:ncol(probability))

}


#'
#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for FF objects
#' @export
plot.FF <- function(x, ...) {
  obj <- x

  probability <- obj$probability
  bp <- barplot(probability,
                beside = TRUE,
                col = gray.colors(ncol(probability)),
                xlab = "Number of Factor",
                ylab = "Probability",
                main = "Factor Forest\nProbability Distribution of the Number of Factors",
                ylim = c(0, max(probability) * 1.1),
                las = 2)

  text(x = bp, y = probability, label = round(probability, digits = 3),
       pos = 3, cex = 0.8, col = "black")
  axis(1, at = bp, labels = 1:length(probability))

}


#'
#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for CDF objects
#' @export
plot.CDF <- function(x, ...) {
  obj <- x

  probability <- obj$probability
  bp <- barplot(probability,
                beside = TRUE,
                col = gray.colors(ncol(probability)),
                xlab = "Number of Factor",
                ylab = "Probability",
                main = "Comparison Data Forest\nProbability Distribution of the Number of Factors",
                ylim = c(0, max(probability) * 1.1),
                las = 2)

  text(x = bp, y = probability, label = round(probability, digits = 3),
       pos = 3, cex = 0.8, col = "black")
  axis(1, at = bp, labels = 1:length(probability))

}

#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for EFAvote objects
#' @export
plot.EFAvote <- function(x, ...) {
  obj <- x

  votes <- obj$votes
  freq_table <- table(votes)
  proportions <- prop.table(freq_table)
  labels <- paste(round(proportions * 100, 2), "%")

  pie(freq_table,
      labels = labels,
      main = "Number of Factors",
      col = gray.colors(length(freq_table)))

  legend("topright",
         legend = paste("Factor ", names(freq_table)),
         fill = gray.colors(length(freq_table)))

}


#'
#' @importFrom graphics plot
#' @describeIn plot Plot method for EFAscreet objects
#' @export
plot.EFAscreet <- function(x, ...) {
  obj <- x

  eigen.value <- obj$eigen.value

  x_axis <- 1:length(eigen.value)
  plot(x_axis, eigen.value, type = "o", col = "blue",
       ylim = range(eigen.value),
       pch = 2,
       xlab = "Number of Factors", ylab = "Eigenvalue", main = "Scree Plot")
}

#'
#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for MAP objects
#' @export
plot.MAP <- function(x, ...) {
  obj <- x

  nfact <- obj$nfact
  MAP.values <- obj$MAP.values

  MAP.for.plot <- MAP.values[MAP.values != Inf]
  x_axis <- 1:length(MAP.values)
  plot(x_axis, c(MAP.for.plot, rep(NA, length(MAP.values) - length(MAP.for.plot))),
       type = "o", col = "blue",
       ylim = range(MAP.for.plot),
       pch = 2,
       xlab = "Number of Factors", ylab = "Average Squared Partial Correlation",
       main = "Minimum Average Partial")

  points(nfact, MAP.for.plot[nfact], pch = 17, col = "blue")
}

#' @importFrom grDevices gray.colors
#' @importFrom graphics plot abline axis barplot legend lines mtext par pie points text
#' @describeIn plot Plot method for STOC objects
#' @export
plot.STOC <- function(x, ...) {
  obj <- x

  nfact <- obj$nfact
  eigen.value <- obj$eigen.value
  eigen.ref <- obj$eigen.ref
  type <- obj$type
  fa <- obj$fa
  if(fa == "pc"){
    fa <- "Principal Component Analysis"
  }else{
    fa <- "Factor Analysis"
  }

  x_axis <- 1:length(eigen.value)
  plot(x_axis, eigen.value, type = "o", col = "blue",
       ylim = range(c(eigen.value, eigen.ref[1:(length(eigen.value)-2)])),
       pch = 2,
       xlab = "Number of Factors", ylab = paste0("Eigenvalue of ", fa), main = paste("STOC Plots"))

  lines(x_axis[1:(length(eigen.value)-2)], eigen.ref[1:(length(eigen.value)-2)], col = "red", lty = 2)
  if(nfact >= 1)
    points(1:nfact, eigen.value[1:nfact], pch = 17, col = "blue")

  legend("topright", legend = c("Actual Eigenvalue", "Reference Eigenvalue"),
         col = c("blue", "red"), pch = c(17, 46), lty = 1:2)
  abline(h = 1, col = "black", lty = 2)

}
