% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/KGC.R
\name{KGC}
\alias{KGC}
\title{Kaiser-Guttman Criterion}
\usage{
KGC(
  response,
  fa = "pc",
  nfact = 1,
  cor.type = "pearson",
  use = "pairwise.complete.obs",
  vis = TRUE,
  plot = TRUE
)
}
\arguments{
\item{response}{A required \code{N} × \code{I} matrix or data.frame consisting of the responses of \code{N} individuals
to \code{I} items.}

\item{fa}{A string that determines the method used to obtain eigenvalues. If 'pc', it represents
Principal Component Analysis (PCA); if 'fa', it represents Principal Axis Factoring (a widely
used Factor Analysis method; @seealso \code{\link[EFAfactors]{factor.analysis}};
Auerswald & Moshagen, 2019). (Default = 'pc')}

\item{nfact}{A numeric value that specifies the number of factors to extract, only effective when \code{fa = 'fa'}. (Default = 1)}

\item{cor.type}{A character string indicating which correlation coefficient (or covariance) is
to be computed. One of \code{"pearson"} (default), \code{"kendall"}, or
\code{"spearman"}. @seealso \code{\link[stats]{cor}}.}

\item{use}{an optional character string giving a method for computing covariances in the presence of missing values. This
must be one of the strings \code{"everything"}, \code{"all.obs"}, \code{"complete.obs"}, \code{"na.or.complete"},
or \code{"pairwise.complete.obs"} (default). @seealso \code{\link[stats]{cor}}.}

\item{vis}{A Boolean variable that will print the factor retention results when set to \code{TRUE}, and will not print
when set to \code{FALSE}. (default = \code{TRUE})}

\item{plot}{A Boolean variable that will print the KGC plot when set to \code{TRUE}, and will not print it when set to
\code{FALSE}. @seealso \code{\link[EFAfactors]{plot.KGC}}. (Default = \code{TRUE})}
}
\value{
An object of class \code{KGC} is a \code{list} containing the following components:
\item{nfact}{The number of factors to be retained.}
\item{eigen.value}{A vector containing the empirical eigenvalues}
}
\description{
This function implements the Kaiser-Guttman criterion (Guttman, 1954; Kaiser, 1960) for determining the number of factors to retain in factor analysis.
It is based on the eigenvalues of the correlation matrix of the responses. According to the criterion, factors are retained
if their corresponding eigenvalues are greater than 1.
}
\examples{
library(EFAfactors)
set.seed(123)

##Take the data.bfi dataset as an example.
data(data.bfi)

response <- as.matrix(data.bfi[, 1:25]) ## loading data
response <- na.omit(response) ## Remove samples with NA/missing values

## Transform the scores of reverse-scored items to normal scoring
response[, c(1, 9, 10, 11, 12, 22, 25)] <- 6 - response[, c(1, 9, 10, 11, 12, 22, 25)] + 1


## Run KGC function with default parameters.
\donttest{
KGC.obj <- KGC(response)

print(KGC.obj)

plot(KGC.obj)

## Get the eigen.value, eigen.ref and  nfact results.
eigen.value <- KGC.obj$eigen.value
nfact <- KGC.obj$nfact

print(eigen.value)
print(nfact)

}


}
\references{
Guttman, L. (1954). Some necessary conditions for common-factor analysis. Psychometrika, 19, 149–161. http://dx.doi.org/10.1007/BF02289162.

Kaiser, H. F. (1960). The application of electronic computers to factor analysis. Educational and Psychological Measurement, 20, 141–151. http://dx.doi.org/10.1177/001316446002000116.
}
