% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mann-whitney.R
\name{mann_whitney_test_pv}
\alias{mann_whitney_test_pv}
\title{Wilcoxon-Mann-Whitney \emph{U} test}
\usage{
mann_whitney_test_pv(
  x,
  y,
  mu = 0,
  alternative = "two.sided",
  exact = NULL,
  correct = TRUE,
  digits_rank = Inf,
  simple_output = FALSE
)
}
\arguments{
\item{x, y}{numerical vectors forming the samples to be tested or lists
of numerical vectors for multiple tests.}

\item{mu}{numerical vector of hypothesised location shift(s).}

\item{alternative}{character vector that indicates the alternative hypotheses; each value must be one of \code{"two.sided"} (the default), \code{"less"} or \code{"greater"}.}

\item{exact}{logical value that indicates whether p-values are to be calculated by exact computation (\code{TRUE}; the default) or by a continuous approximation (\code{FALSE}).}

\item{correct}{logical value that indicates if a continuity correction is to be applied (\code{TRUE}; the default) or not (\code{FALSE}). Ignored, if \code{exact = TRUE}.}

\item{digits_rank, }{single number giving the significant digits used to compute ranks for the test statistics.}

\item{simple_output, }{logical value that indicates whether an R6 class object, including the tests' parameters and support sets, i.e. all observable p-values under each null hypothesis, is to be returned (see below).}
}
\value{
If \code{simple.output = TRUE}, a vector of computed p-values is returned.
Otherwise, the output is a \code{\link{DiscreteTestResults}} R6 class object, which
also includes the p-value supports and testing parameters. These have to be
accessed by public methods, e.g. \verb{$get_pvalues()}.
}
\description{
\code{mann_whitney_test_pv()} performs an exact or approximate
Wilcoxon-Mann-Whitney \emph{U} test about the location shift between two
independent groups when the data is not necessarily normally distributed. In
contrast to \code{\link[stats:wilcox.test]{stats::wilcox.test()}}, it is vectorised and only calculates
\emph{p}-values. Furthermore, it is capable of returning the discrete \emph{p}-value
supports, i.e. all observable \emph{p}-values under a null hypothesis. Multiple
tests can be evaluated simultaneously.
}
\details{
We use a test statistic called the Wilcoxon Rank Sum Statistic, defined by
\deqn{U = \sum_{i = 1}^{n_X}{rank(X_i)} - \frac{n_X(n_X + 1)}{2},}
where \eqn{rank(X_i)} is the rank of \eqn{X_i} in the concatenated sample
of \eqn{X} and \eqn{Y}, and \eqn{n_X} and \eqn{n_Y} are the respective
sizes of the samples \eqn{X} and \eqn{Y}. Note that \eqn{U}
can range from \eqn{0} to \eqn{n_X \cdot n_Y}.
This is the same statistic used by \code{\link[stats:wilcox.test]{stats::wilcox.test()}} and
whose distribution is accessible with \code{\link{pwilcox}}.
This is also the statistic defined by the two given references.
Note, however, that it is not what is called the Mann-Whitney U Statistic
in the (English-language) Wikipedia article (as of February 12, 2026). The
latter is defined as, using our notation, \eqn{\min(U, n_X \cdot n_Y - U)}.
Using the Wikipedia notation, the Wilcoxon Rank Sum Statistic is \eqn{U_2}.

The parameters \code{x}, \code{y}, \code{mu} and \code{alternative} are vectorised. If \code{x} and
\code{y} are lists, they are replicated automatically to have the same lengths. In
case \code{x} or \code{y} are not lists, they are added to new ones, which are then
replicated to the appropriate lengths. This allows multiple hypotheses to be
tested simultaneously.

In the presence of ties, computation of exact \emph{p}-values is not possible.
Therefore, \code{exact} is ignored in these cases and \emph{p}-values of the
respective test settings are calculated by a normal approximation.

By setting \code{exact = NULL}, exact computation is performed if both samples in
a test setting do not have any ties and if both sample sizes are lower than
or equal to 200.

If \code{digits_rank = Inf} (the default), \code{\link[base:rank]{rank()}} is used to
compute ranks for the tests statistics instead of
\code{\link[base:rank]{rank}}(\code{\link[base:Round]{signif(., digits_rank)}})
}
\examples{
# Constructing
set.seed(1)
r1 <- rnorm(100)
r2 <- rnorm(100, 1)

# Exact two-sided p-values and their supports
results_ex  <- mann_whitney_test_pv(r1, r2)
print(results_ex)
results_ex$get_pvalues()
results_ex$get_pvalue_supports()

# Normal-approximated one-sided p-values ("less") and their supports
results_ap  <- mann_whitney_test_pv(r1, r2, alternative = "less", exact = FALSE)
print(results_ap)
results_ap$get_pvalues()
results_ap$get_pvalue_supports()

}
\references{
Mann, H. D. & Whitney, D. R. (1947). On a Test of Whether one of Two Random
Variables is Stochastically Larger than the Other. \emph{Ann. Math. Statist.},
\emph{18}(1), pp. 50-60. \doi{10.1214/aoms/1177730491}

Hollander, M. & Wolfe, D. (1973). \emph{Nonparametric Statistical Methods}. Third
Edition. New York: Wiley. pp. 115-135. \doi{10.1002/9781119196037}
}
\seealso{
\code{\link[stats:wilcox.test]{stats::wilcox.test()}}, \code{\link{pwilcox}}, \code{\link[=wilcox_test_pv]{wilcox_test_pv()}}
}
