% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jbd_coordinates_transposed.R
\name{jbd_coordinates_transposed}
\alias{jbd_coordinates_transposed}
\title{Identify transposed geographic coordinates}
\usage{
jbd_coordinates_transposed(
  data,
  idcol = "database_id",
  sci_names = "scientificName",
  lat = "decimalLatitude",
  lon = "decimalLongitude",
  country = "country",
  countryCode = "countryCode",
  border_buffer = 0.2,
  save_outputs = FALSE,
  fileName = NULL,
  scale = "large",
  path = NULL,
  mc.cores = 1
)
}
\arguments{
\item{data}{A data frame or tibble. Containing a unique identifier for each record,
geographical coordinates, and country names. Coordinates must be expressed
in decimal degrees and WGS84.}

\item{idcol}{A character string. The column name with a unique record identifier.
Default = "database_id".}

\item{sci_names}{A character string. The column name with species' scientific
names. Default = "scientificName".}

\item{lat}{A character string. The column name with latitudes. Coordinates must
be expressed in decimal degrees and WGS84. Default = "decimalLatitude".}

\item{lon}{A character string. The column name with longitudes. Coordinates must be
expressed in decimal degrees and WGS84. Default = "decimalLongitude".}

\item{country}{A character string. The column name with the country
assignment of each occurrence record. Default = "country".}

\item{countryCode}{A character string. The column name containing an ISO-2 country code for
each record.}

\item{border_buffer}{Numeric. Must have value greater than or equal to 0.
A distance in decimal degrees used to
created a buffer around each country. Records within a given country and at
a specified distance from the border will be not be corrected.
Default = 0.2 (~22 km at the equator).}

\item{save_outputs}{Logical. Indicates if a table containing transposed coordinates should be
saved for further inspection. Default = FALSE.}

\item{fileName}{A character string. The out file's name.}

\item{scale}{Passed to rnaturalearth's ne_countries().
Scale of map to return, one of 110, 50, 10 or 'small', 'medium', 'large'. Default = "large".}

\item{path}{A character string. A path as a character vector for where to create the directories
and save the figures. If
no path is provided (the default), the directories will be created using \code{\link[here:here]{here::here()}}.}

\item{mc.cores}{Numeric. If > 1, the jbd_correct_coordinates function will run in parallel
using mclapply using the number of cores specified. If = 1 then it will be run using a serial
loop. NOTE: Windows machines must use a value of 1 (see ?parallel::mclapply). Additionally,
be aware that each thread can use large chunks of memory.
Default = 1.#'}
}
\value{
A tibble containing the column "coordinates_transposed" which indicates if
verbatim coordinates were not transposed (TRUE). Otherwise
records are flagged as (FALSE) and, in this case, verbatim coordinates are
replaced by corrected coordinates.
}
\description{
This function flags and corrects records when latitude and longitude appear
to be transposed.
This function will preferably use the countryCode column generated by
\code{\link[bdc:bdc_country_standardized]{bdc::bdc_country_standardized()}}.
}
\details{
This test identifies transposed coordinates based on mismatches between the
country provided for a record and the record's latitude and longitude coordinates. Transposed
coordinates often fall outside of the indicated country (i.e., in other
countries or in the sea). Different coordinate transformations are
performed to correct country/coordinates mismatches. Importantly, verbatim
coordinates are replaced by the corrected ones in the returned database. A
database containing verbatim and corrected coordinates is created in
"Output/Check/01_coordinates_transposed.csv" if save_outputs == TRUE. The
columns "country" and "countryCode" can be retrieved by using the function
\link[bdc:bdc_country_standardized]{bdc::bdc_country_standardized}.
}
\examples{
\donttest{
if(require("rnaturalearthdata") && require("bdc")){
database_id <- c(1, 2, 3, 4)
scientificName <- c(
  "Rhinella major", "Scinax ruber",
  "Siparuna guianensis", "Psychotria vellosiana"
)
decimalLatitude <- c(63.43333, -14.43333, -41.90000, -46.69778)
decimalLongitude <- c(-17.90000, -67.91667, -13.25000, -13.82444)
country <- c("BOLIVIA", "bolivia", "Brasil", "Brazil")

x <- data.frame(
  database_id, scientificName, decimalLatitude,
  decimalLongitude, country
)

# Get country codes
x <- bdc::bdc_country_standardized(data = x, country = "country")

jbd_coordinates_transposed(
  data = x,
  idcol = "database_id",
  sci_names = "scientificName",
  lat = "decimalLatitude",
  lon = "decimalLongitude",
  country = "country_suggested",
  countryCode = "countryCode",
  border_buffer = 0.2,
  save_outputs = FALSE,
  scale = "medium"
) 
} # END if require
} # RND donttest

}
\concept{prefilter}
