#' @title Capture Warnings
#'
#' @description This function captures the warnings generated by an
#' expression without printing them to the console. It returns both the
#' result of the expression and the captured warnings. This is useful for
#' avoiding warnings appearing during routines, such as an iterative process.
#' To suppress or "muffle" a warning, it uses invokeRestart("muffleWarning").
#' The `new.env()` function creates a new environment that is isolated from
#' other environments. This allows the function to store and modify data
#' (in this case, warnings) without affecting the global environment or
#' any other environment in the R session.
#'
#' @param expr An expression to evaluate.
#' @param warn_env An optional environment to store the warnings.
#' If not provided, a new environment is created.
#' @return A list containing two elements:
#'   \item{result}{The result of evaluating the expression.}
#'   \item{warnings}{A character vector of the warnings generated during
#'                   the evaluation of the expression.}
#' @keywords internal

capture_warnings <- function(expr, warn_env = new.env()) {
  warn_env$warnings <- character()

  result <- withCallingHandlers(
    expr,
    warning = function(w) {
      warn_env$warnings <- c(warn_env$warnings, conditionMessage(w))
      invokeRestart("muffleWarning")
    }
  )

  list(result = result, warnings = warn_env$warnings)
}
