% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adaptive_sampling.R
\name{run_adaptive_sampling}
\alias{run_adaptive_sampling}
\title{Performs adaptive Monte Carlo sampling}
\usage{
run_adaptive_sampling(
  initial_samples_file,
  scenario_name,
  dissimilarity_matrix,
  max_cores = NULL,
  num_samples = 10,
  mapping_max_iter = 1000,
  relative_epsilon = 1e-04,
  folds = 20,
  output_dir,
  verbose = FALSE
)
}
\arguments{
\item{initial_samples_file}{Character. Path to a CSV file containing initial samples.}

\item{scenario_name}{Character. Name for the output files.}

\item{dissimilarity_matrix}{Matrix. The input dissimilarity matrix.}

\item{max_cores}{Integer. Number of cores to use for parallel execution. If NULL, uses all available cores minus 1.}

\item{num_samples}{Integer. Number of new samples to generate via adaptive
sampling.}

\item{mapping_max_iter}{Integer. Maximum number of map optimization iterations.}

\item{relative_epsilon}{Numeric. Convergence threshold for relative change in error. Default is 1e-4.}

\item{folds}{Integer. Number of cross-validation folds.}

\item{output_dir}{Character. Required directory for output files.}

\item{verbose}{Logical. Whether to print progress messages. Default is FALSE.}
}
\value{
No return value. Called for its side effect of writing results to a CSV file in \code{output_dir}.
}
\description{
Performs adaptive Monte Carlo sampling to explore and refine the parameter space,
running locally in parallel. Samples are drawn adaptively based on previously
evaluated likelihoods to focus sampling in high-likelihood regions. Results from all
parallel jobs accumulate in a single output file.
}
\examples{
\donttest{
# 1. Locate the example initial samples file included with the package
# In a real scenario, this file would be from an 'initial_parameter_optimization' run.
initial_file <- system.file(
  "extdata", "initial_samples_example.csv",
  package = "topolow"
)

# 2. Create a temporary directory for the function's output
# This function requires a writable directory for its results.
temp_out_dir <- tempdir()

# 3. Create a sample dissimilarity matrix for the function to use
dissim_mat <- matrix(runif(100, 1, 10), 10, 10)
diag(dissim_mat) <- 0

# 4. Run the adaptive sampling only if the example file is found
if (nzchar(initial_file)) {
  run_adaptive_sampling(
    initial_samples_file = initial_file,
    scenario_name = "adaptive_test_example",
    dissimilarity_matrix = dissim_mat,
    output_dir = temp_out_dir,
    max_cores = 1,
    num_samples = 1,
    verbose = FALSE
  )

  # 5. Verify output files were created
  print("Output files from adaptive sampling:")
  print(list.files(temp_out_dir, recursive = TRUE))

  # 6. Clean up the temporary directory
  unlink(temp_out_dir, recursive = TRUE)
}
}
}
