% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/resolve-links.R
\name{resolve_anchor_links}
\alias{resolve_anchor_links}
\title{Resolve Reference-Style Links}
\usage{
resolve_anchor_links(body, txt, ns = md_ns())
}
\arguments{
\item{body}{an XML body}

\item{txt}{the text of a source file}

\item{ns}{an the namespace that resolves the Markdown namespace (defaults to
\code{\link[=md_ns]{md_ns()}})}
}
\description{
\href{https://www.markdownguide.org/basic-syntax/#reference-style-links}{Reference style links and images}
are a form of markdown syntax that reduces duplication and makes markdown
more readable. They come in two parts:
\enumerate{
\item The inline part that uses two pairs of square brackets where the second
pair of square brackets contains the reference for the anchor part of the
link. Example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{[inline text describing link][link-reference]
}\if{html}{\out{</div>}}
\item The anchor part, which can be anywhere in the document, contains a pair
of square brackets followed by a colon and space with the link and
optionally the link title. Example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{[link-reference]: https://docs.ropensci.org/tinkr/ 'documentation for tinkr'
}\if{html}{\out{</div>}}
}

Commonmark treats reference-style links as regular links, which can be a
pain when converting large documents. This function resolves these
links by reading in the source document, finding the reference-style links,
and adding them back at the end of the document with the 'anchor' attribute
and appending the reference to the link with the 'ref' attribute.
}
\details{
\subsection{Nomenclature}{

The reference-style link contains two parts, but they don't have common names
(the \href{https://www.markdownguide.org/basic-syntax/}{markdown guide} calls
these "first part and second part"), so in this documentation, we call the
link pattern of \verb{[link text][link-ref]} as the "inline reference-style link"
and the pattern of \verb{[link-ref]: <URL>} as the "anchor references-style link".
}

\subsection{Reference-style links in commonmark's XML representation}{

A link or image in XML is represented by a node with the following attributes
\itemize{
\item destination: the URL for the link
\item title: an optional title for the link
}

For example, this markdown link \verb{[link text](https://example.com "example  link")} is represented in XML as text inside of a link node:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{lnk <- "[link text](https://example.com 'example link')"
xml <- xml2::read_xml(commonmark::markdown_xml(lnk))
cat(as.character(xml2::xml_find_first(xml, ".//d1:link")))
#> <link destination="https://example.com" title="example link">
#>   <text xml:space="preserve">link text</text>
#> </link>
}\if{html}{\out{</div>}}

However, reference-style links are rendered equivalently:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{lnk <- "
[link text][link-ref]

[link-ref]: https://example.com 'example link'
"
xml <- xml2::read_xml(commonmark::markdown_xml(lnk))
cat(as.character(xml2::xml_find_first(xml, ".//d1:link")))
#> <link destination="https://example.com" title="example link">
#>   <text xml:space="preserve">link text</text>
#> </link>
}\if{html}{\out{</div>}}
}

\subsection{XML attributes of reference-style links}{

To preserve the anchor reference-style links, we search the source document
for the destination attribute proceded by \verb{]: }, transform that information
into a new link node with the \code{anchor} attribute, and add it to the end of
the document. That node looks like this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{<link destination="https://example.com" title="example link" anchor="true">
  <text>link-ref</text>
</link>
}\if{html}{\out{</div>}}

From there, we add the anchor text to the node that is present in our
document as the \code{ref} attribute:

\if{html}{\out{<div class="sourceCode">}}\preformatted{<link destination="https://example.com" title="example link" rel="link-ref">
  <text xml:space="preserve">link text</text>
</link>
}\if{html}{\out{</div>}}
}
}
\note{
this function is internally used in the function \code{\link[=to_xml]{to_xml()}}.
}
\examples{
f <- system.file("extdata", "link-test.md", package = "tinkr")
md <- yarn$new(f, sourcepos = TRUE, anchor_links = FALSE)
md$show()
if (requireNamespace("withr")) {
lnks <- withr::with_namespace("tinkr", 
  resolve_anchor_links(md$body, readLines(md$path)))
md$body <- lnks
md$show()
}
}
\keyword{internal}
