% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quest_functions.R
\name{long2wide}
\alias{long2wide}
\title{Reshape Multiple Scores From Long to Wide}
\usage{
long2wide(
  data,
  vrb.nm,
  grp.nm,
  obs.nm,
  sep = ".",
  colnames.by.obs = TRUE,
  keep.attr = FALSE
)
}
\arguments{
\item{data}{data.frame of data.}

\item{vrb.nm}{character vector of colnames from \code{data} specifying the
variables to be reshaped. In longitudinal panel data, this would be the
scores.}

\item{grp.nm}{character vector of colnames from \code{data} specifying the
groups. In longitudnal panel data, this would be the participant ID
variable.}

\item{obs.nm}{character vector of length 1 with a colname from \code{data}
specifying the observation within each group. In longitudinal panel data,
this would be the time variable.}

\item{sep}{character vector of length 1 specifying the string that separates
the name prefix (e.g., score) from it's number suffix (e.g., timepoint). If
\code{sep} = "", then that implies there is no string separating the name
prefix and the number suffix (e.g., "outcome1").}

\item{colnames.by.obs}{logical vector of length 1 specifying whether to sort
the return object colnames by the observation label (TRUE) or by the order
of \code{vrb.nm}. See the example at the end of the "MULTIPLE GROUPING
VARIABLES" section of the examples.}

\item{keep.attr}{logical vector of length 1 specifying whether to keep the
"reshapeWide" attribute (from \code{reshape}) in the return object.}
}
\value{
data.frame with nrow equal to \code{nrow(unique(data[grp.nm]))} and
  number of reshaped columns equal to \code{length(vrb.nm) *
  unique(data[[obs.nm]])}. The colnames will have the structure
  \code{paste0(vrb.nm, sep, unique(data[[obs.nm]]))}. The reshaped colnames
  are sorted by the observation labels if \code{colnames.by.obs} = TRUE and
  sorted by \code{vrb.nm} if \code{colnames.by.obs} = FALSE. Overall, the
  columns are in the following order: 1) \code{grp.nm} of the groups, 2)
  reshaped columns, 3) additional columns that were not reshaped.
}
\description{
\code{long2wide} reshapes data from long to wide. This if often necessary to
do with multilevel data where variables in the long format seek to be
reshaped to multiple sets of variables in the wide format. If only one column
needs to be reshaped, then you can use \code{\link[str2str]{unstack2}} or
\code{\link[reshape]{cast}} - but that does not work for *multiple* columns.
}
\details{
\code{long2wide} uses \code{reshape(direction = "wide")} to reshape the data.
It attempts to streamline the task of reshaping long to wide as the
\code{reshape} arguments can be confusing because the same arguments are used
for wide vs. long reshaping. See \code{\link[stats]{reshape}} if you are
curious.
}
\examples{

# SINGLE GROUPING VARIABLE
dat_long <- as.data.frame(ChickWeight) # b/c groupedData class does weird things...
w1 <- long2wide(data = dat_long, vrb.nm = "weight", grp.nm = "Chick",
   obs.nm = "Time") # NAs inserted for missing observations in some groups
w2 <- long2wide(data = dat_long, vrb.nm = "weight", grp.nm = "Chick",
   obs.nm = "Time", sep = "_")
head(w1); head(w2)
w3 <- long2wide(data = dat_long, vrb.nm = "weight", grp.nm = "Chick",
   obs.nm = "Time", sep = "_T", keep.attr = TRUE)
attributes(w3)

# MULTIPLE GROUPING VARIABLE
tmp <- psychTools::sai
grps <- interaction(tmp[1:3], drop = TRUE)
dups <- duplicated(grps)
dat_long <- tmp[!(dups), ] # for some reason there are duplicate groups in the data
vrb_nm <- str2str::pick(names(dat_long), val = c("study","time","id"), not = TRUE)
w4 <- long2wide(data = dat_long, vrb.nm = vrb_nm, grp.nm = c("study","id"),
   obs.nm = "time")
w5 <- long2wide(data = dat_long, vrb.nm = vrb_nm, grp.nm = c("study","id"),
   obs.nm = "time", colnames.by.obs = FALSE) # colnames sorted by `vrb.nm` instead
head(w4); head(w5)

}
\seealso{
\code{\link{wide2long}}
   \code{\link[stats]{reshape}}
   \code{\link[str2str]{unstack2}}
}
