% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/frequencies.R
\name{frequencies}
\alias{frequencies}
\title{Display Frequency Tables of Single Variables}
\usage{
frequencies(
  data_frame,
  variables,
  formats = c(),
  by = c(),
  weight = NULL,
  titles = c(),
  footnotes = c(),
  style = excel_output_style(),
  output = "console",
  na.rm = FALSE,
  print = TRUE,
  monitor = FALSE
)
}
\arguments{
\item{data_frame}{A data frame in which are the variables to tabulate.}

\item{variables}{A vector of single variables to create frequency tables for.}

\item{formats}{A list in which is specified which formats should be applied to which variables.}

\item{by}{Compute tables stratified by the expressions of the provided variables.}

\item{weight}{Put in a weight variable to compute weighted results.}

\item{titles}{Specify one or more table titles.}

\item{footnotes}{Specify one or more table footnotes.}

\item{style}{A list of options can be passed to control the appearance of excel outputs.
Styles can be created with \code{\link[=excel_output_style]{excel_output_style()}}.}

\item{output}{The following output formats are available: console (default), text,
excel and excel_nostyle.}

\item{na.rm}{FALSE by default. If TRUE removes all NA values from the variables.}

\item{print}{TRUE by default. If TRUE prints the output, if FALSE doesn't print anything. Can be used
if one only wants to catch the output data frame.}

\item{monitor}{FALSE by default. If TRUE outputs two charts to visualize the functions time consumption.}
}
\value{
Returns a list of two data tables containing the results for the frequency tables.
}
\description{
\code{\link[=frequencies]{frequencies()}} produces two kinds of tables for a quick overview of single variables.
The first table is for a broader overview and contains mean, sd, min, max, freq and missings.
The second table is the actual frequency table which shows the weighted sums, percentages
and unweighted frequencies per expression.
}
\details{
\code{\link[=frequencies]{frequencies()}} is based on the 'SAS' procedure Proc Freq, which provides
efficient and readable ways to output frequency tables.

To create a frequency table you only need to provide a single variable.
Nothing special about this. The real power comes into play, when you output your
tables as a fully styled 'Excel' workbook. Setting up a custom, reusable style is
as easy as setting up options like: provide a color for the table header, set the
font size for the row header, should borders be drawn for the table cells yes/no,
and so on.

You also can provide multiple single variables to generate multiple, evenly designed
tables, all at once. For just a quick overview, rather than fully designed
tables, you can also just output the tables in ASCII style format.
}
\examples{
# Example data frame
my_data <- dummy_data(1000)

# Define titles and footnotes. If you want to add hyperlinks you can do so by
# adding "link:" followed by the hyperlink to the main text.
titles <- c("This is title number 1 link: https://cran.r-project.org/",
            "This is title number 2",
            "This is title number 3")
footnotes <- c("This is footnote number 1",
               "This is footnote number 2",
               "This is footnote number 3 link: https://cran.r-project.org/")

# Output frequency tables
my_data |> frequency(sex)
my_data |> frequency(c(age, education),
                     weight = weight)

# Also works with characters
my_data |> frequency("sex")
my_data |> frequency(c("age", "education"),
                     weight = "weight")

# Applying formats and titles
sex. <- discrete_format(
    "Total"  = 1:2,
    "Male"   = 1,
    "Female" = 2)

my_data |> frequency(sex, formats(sex = sex.),
                     titles    = titles,
                     footnotes = footnotes)

# Split frequencies by expressions of another variable
my_data |> frequency(sex, by = education)

# Get a list with two data tables for further usage
result_list <- my_data |> frequency(sex, formats(sex = sex.))

# Output in text file
my_data |> frequency(sex, output = "text")

# Output to Excel
my_data |> frequency(sex, output = "excel")

# With individual styling
my_style <- excel_output_style(header_back_color = "0077B6",
                               font              = "Times New Roman")

my_data |> frequency(sex, output = "excel", style = my_style)

}
\seealso{
Creating a custom table style: \code{\link[=excel_output_style]{excel_output_style()}}, \code{\link[=modify_output_style]{modify_output_style()}},
\code{\link[=number_format_style]{number_format_style()}}, \code{\link[=modify_number_formats]{modify_number_formats()}}.

Creating formats: \code{\link[=discrete_format]{discrete_format()}} and \code{\link[=interval_format]{interval_format()}}.

Functions that can handle formats and styles: \code{\link[=crosstabs]{crosstabs()}}, \code{\link[=any_table]{any_table()}}.

Additional functions that can handle styles: \code{\link[=export_with_style]{export_with_style()}}

Additional functions that can handle formats: \code{\link[=summarise_plus]{summarise_plus()}}, \code{\link[=recode]{recode()}},
\code{\link[=recode_multi]{recode_multi()}}
}
