% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kendall_corr.R
\name{kendall_tau}
\alias{kendall_tau}
\alias{print.kendall_matrix}
\alias{plot.kendall_matrix}
\title{Pairwise Kendall's Tau Rank Correlation}
\usage{
kendall_tau(data)

\method{print}{kendall_matrix}(x, digits = 4, max_rows = NULL, max_cols = NULL, ...)

\method{plot}{kendall_matrix}(
  x,
  title = "Kendall's Tau correlation heatmap",
  low_color = "indianred1",
  high_color = "steelblue1",
  mid_color = "white",
  value_text_size = 4,
  ...
)
}
\arguments{
\item{data}{A numeric matrix or a data frame with at least two numeric
columns. All non-numeric columns will be excluded. Each column must have
at least two non-missing values and contain no NAs.}

\item{x}{An object of class \code{kendall_matrix}.}

\item{digits}{Integer; number of decimal places to print}

\item{max_rows}{Optional integer; maximum number of rows to display.
If \code{NULL}, all rows are shown.}

\item{max_cols}{Optional integer; maximum number of columns to display.
If \code{NULL}, all columns are shown.}

\item{...}{Additional arguments passed to \code{ggplot2::theme()} or other
\code{ggplot2} layers.}

\item{title}{Plot title. Default is \code{"Kendall's Tau Correlation
Heatmap"}.}

\item{low_color}{Color for the minimum tau value. Default is
\code{"indianred1"}.}

\item{high_color}{Color for the maximum tau value. Default is
\code{"steelblue1"}.}

\item{mid_color}{Color for zero correlation. Default is \code{"white"}.}

\item{value_text_size}{Font size for displaying correlation values. Default
is \code{4}.}
}
\value{
A symmetric numeric matrix where the \code{(i, j)}-th element is
the Kendall's tau correlation between the \code{i}-th and \code{j}-th
numeric columns of the input.

Invisibly returns the \code{kendall_matrix} object.

A \code{ggplot} object representing the heatmap.
}
\description{
Computes all pairwise Kendall's tau rank correlation coefficients for the
numeric columns of a matrix or data frame using a high-performance
'C++'.

This function uses a fast and scalable algorithm implemented in 'C++'
to compute both Kendall's tau-a (when no ties are present) and tau-b
(when ties are detected), making it suitable for large datasets. Internally,
it calls a highly optimized function that uses a combination of merge-sort-
based inversion counting and a Fenwick Tree (binary indexed tree)
for efficient tie handling.

Prints a summary of the Kendall's tau correlation matrix,
including description and method metadata.

Generates a ggplot2-based heatmap of the Kendall's tau
correlation matrix.
}
\details{
Kendall's tau is a rank-based measure of association between two variables.
For a dataset with \eqn{n} observations of two variables \eqn{X} and
\eqn{Y}, Kendall's tau coefficient is defined as:

\deqn{ \tau = \frac{C - D}{\sqrt{(C + D + T_x)(C + D + T_y)}} }

where:
\itemize{
\item \eqn{C} is the number of concordant pairs defined by
\eqn{(x_i - x_j)(y_i - y_j) > 0}
\item \eqn{D} is the number of discordant pairs defined by
\eqn{(x_i - x_j)(y_i - y_j) < 0}
\item \eqn{T_x}, \eqn{T_y} are the number of tied pairs in \eqn{X} and
\eqn{Y}, respectively
}

When there are no ties, the function computes the faster \emph{tau-a}
version:
\deqn{ \tau_a = \frac{C - D}{n(n-1)/2} }

The function automatically selects \emph{tau-a} or \emph{tau-b} depending
on the presence of ties. Performance is maximized by computing correlations
in 'C++' directly from the matrix columns.
}
\note{
Missing values are not allowed. Columns with fewer than two
observations are excluded.
}
\examples{
# Basic usage with a matrix
mat <- cbind(a = rnorm(100), b = rnorm(100), c = rnorm(100))
kt <- kendall_tau(mat)
print(kt)
plot(kt)

# With a large data frame
df <- data.frame(x = rnorm(1e4), y = rnorm(1e4), z = rnorm(1e4))
kendall_tau(df)

# Including ties
tied_df <- data.frame(
  v1 = rep(1:5, each = 20),
  v2 = rep(5:1, each = 20),
  v3 = rnorm(100)
)
kt <- kendall_tau(tied_df)
print(kt)
plot(kt)

}
\references{
Kendall, M. G. (1938). A New Measure of Rank Correlation. Biometrika,
30(1/2), 81–93.
}
\seealso{
\code{\link{print.kendall_matrix}},
\code{\link{print.kendall_matrix}}
}
\author{
Thiago de Paula Oliveira \email{toliveira@abacusbio.com}

Thiago de Paula Oliveira
}
