% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/globpso.R
\name{globpso}
\alias{globpso}
\title{Particle Swarm Optimization Algorithms for Minimization Problems}
\usage{
globpso(
  objFunc,
  lower,
  upper,
  init = NULL,
  fixed = NULL,
  PSO_INFO = NULL,
  seed = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{objFunc}{The R or Rcpp compiled objective function. See the example.}

\item{lower}{The vector of finite lower bounds of the search domain.
Its length should be equal to the dimension of the domain space.}

\item{upper}{The vector of finite upper bounds of the search domain.
Its length should be equal to the dimension of the domain space.}

\item{init}{The vector of initial swarm.
Its length should be equal to the dimension of the domain space.
When there are more than one initial vectors, specify \code{init} as
a matrix. Each row vector represents one initial point.
The default for \code{init} is \code{NULL}.}

\item{fixed}{The vector of real values and NA values that controls PSO to search only 
for the NA-valued components.}

\item{PSO_INFO}{The list of PSO parameters generated by \code{getPSOInfo()}.}

\item{seed}{The random seed that controls initial swarm of PSO. The default is \code{NULL}.}

\item{verbose}{The logical value controls if PSO would reports the updating progress. The default is \code{TRUE}.}

\item{...}{Further arguments to be passed to \code{objFunc}.}
}
\value{
An List.
\describe{
\item{par}{ the global best particle.}
\item{val}{ the objective function value of the global best particle.}
\item{history}{ a vector of objective function values of the global best particle in PSO search history.}
\item{cputime}{ the computational time in seconds.}
}
}
\description{
The general-purpose implementation of particle swarm Optimization algorithms for minimizing an user-defined objective function.
}
\examples{
# three-dimensional function
objf <- function(x, loc) {
  val <- 0
  for (i in 1:length(x)) {
    val <- val + (x[i] - loc)^2
  }
  return(val)
}

upp_bound <- rep(5, 3)
low_bound <- rep(-5, 3)
loc_shift <- 1

alg_setting <- getPSOInfo(nSwarm = 32, maxIter = 100, psoType = "basic")
res <- globpso(objFunc = objf, lower = low_bound, upper = upp_bound, 
               PSO_INFO = alg_setting, loc = loc_shift)
res$par
res$val

# C++ function example
\donttest{
library(Rcpp)
library(RcppArmadillo)
objf_c <- cppFunction('double objf_c(SEXP x, SEXP loc) {
    double val = 0;
    double loc_c = (double)Rcpp::as<double>(loc);
    arma::rowvec x_c = (arma::rowvec)Rcpp::as<arma::rowvec>(x);
    for (arma::uword i = 0; i < x_c.n_elem; i++) {
      val += (x_c(i) - loc_c)*(x_c(i) - loc_c);
    }
    return val;
  }', depends = "RcppArmadillo")
  
alg_setting <- getPSOInfo(nSwarm = 32, maxIter = 100, psoType = "quantum")
res_c <- globpso(objFunc = objf_c, lower = low_bound, upper = upp_bound, 
                 PSO_INFO = alg_setting, loc = 1)
res_c$par
res_c$val
}
}
\references{
\enumerate{
  \item Bonyadi, M. R., & Michalewicz, Z. (2014). A locally convergent rotationally invariant particle swarm optimization algorithm. Swarm intelligence, 8(3), 159-198.
  \item Cheng, R., & Jin, Y. (2014). A competitive swarm optimizer for large scale optimization. IEEE transactions on cybernetics, 45(2), 191-204.
  \item Shi, Y., & Eberhart, R. (1998, May). A modified particle swarm optimizer. In Evolutionary Computation Proceedings, 1998. IEEE World Congress on Computational Intelligence., The 1998 IEEE International Conference on (pp. 69-73). IEEE.
  \item Stehlík, M., Chen, P. Y., Wong, W. K., and Kiseľák, J. (2024). A double exponential particle swarm optimization with non-uniform variates as stochastic tuning and guaranteed convergence to a global optimum with sample applications to finding optimal exact designs in biostatistics. Applied Soft Computing, 163, 111913.
  \item Sun, J., Feng, B., and Xu, W. (2004a). Particle swarm optimization with particles having quantum behavior. In Evolutionary Computation, 2004. CEC2004. Congress on, volume 1, pages 325-331. IEEE.
}
}
