% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/instancer.R
\name{simulate_election}
\alias{simulate_election}
\title{Simulate an Election}
\usage{
simulate_election(
  num_ballots,
  num_candidates,
  num_groups,
  ballot_voters = rep(100, num_ballots),
  lambda = 0.5,
  seed = NULL,
  group_proportions = rep(1/num_groups, num_groups),
  prob = NULL
)
}
\arguments{
\item{num_ballots}{Number of ballot boxes (\code{b}).}

\item{num_candidates}{Number of candidates (\code{c}).}

\item{num_groups}{Number of demographic groups (\code{g}).}

\item{ballot_voters}{A vector of length \code{num_ballots} representing the number of voters per ballot
box. Defaults to \code{rep(100, num_ballots)}.}

\item{lambda}{A numeric value between 0 and 1 that represents the fraction of voters that are randomly assigned to ballot-boxes. The remaining voters are assigned sequentially according to their demographic group.
\itemize{
\item \code{lambda = 0}: The assignment of voters to ballot-boxes is fully sequential in terms of their demographic group. This leads to a \strong{high heterogeneity} of the voters' groups across ballot-boxes.
\item \code{lambda = 1}: The assignment of voters to ballot-boxes is fully random. This leads to a \strong{low heterogeneity} of the voters' group across ballot-boxes.
}

Default value is set to \code{0.5}. See \strong{Shuffling Mechanish} for more details.}

\item{seed}{If provided, overrides the current global seed. Defaults to \code{NULL}.}

\item{group_proportions}{Optional. A vector of length \code{num_groups} that indicates the fraction of voters that belong to each group. Default is that all groups are of the same size.}

\item{prob}{Optional. A user-supplied probability matrix of dimension \verb{(g x c)}.
If provided, this matrix is used as the underlying voting probability distribution. If not supplied, each row is sampled from a Dirichlet distribution with each parameter set to one.}
}
\value{
An eim object with three attributes:
\describe{
\item{\code{X}}{A \code{(b x c)} matrix with candidates' votes for each ballot box.}
\item{\code{W}}{A \code{(b x g)} matrix with voters' groups for each ballot-box.}
\item{\code{real_prob}}{A \code{(g x c)} matrix with the probability that a voter from each group votes for each candidate. If prob is provided, it would equal such probability.}
\item{\code{outcome}}{A \code{(b x g x c)} array with the number of votes for each candidate in each ballot box, broken down by group.}
}
}
\description{
This function simulates an election by creating matrices representing candidate votes \code{(X)} and voters' demographic group \code{(W)} across a specified number of ballot-boxes. It either (i) receives as input or (ii) generates a probability matrix \code{(prob)}, indicating how likely each demographic group is to vote for each candidate.

By default, the number of voters per ballot box \code{(ballot_voters)} is set to a vector of 100 with
length \code{num_ballots}. You can optionally override this by providing a custom vector.

Optional parameters are available to control the distribution of votes:
\itemize{
\item \strong{\code{group_proportions}}: A vector of length \code{num_groups} specifying
the overall proportion of each demographic group. Entries must sum to one and be non-negative.
\item \strong{\code{prob}}: A user-supplied probability matrix of dimension
(\code{num_groups} \eqn{\times} \code{num_candidates}). If provided, this matrix is used directly. Otherwise, voting probabilities for each group are drawn from a Dirichlet distribution.
}
}
\section{Shuffling Mechanism}{

Without loss of generality, consider an order relation of groups and ballot-boxes. The shuffling step is controlled by the \code{lambda} parameter and operates as follows:
\enumerate{
\item \strong{Initial Assignment}: Voters are assigned to each ballot-box sequentially according to their demographic
group. More specifically, the first ballot-boxes receive voters from the first group. Then, the next ballot-boxes receive voters from the second group. This continues until all voters have been assigned. Note that most ballot-boxes will contain voters from a single group (as long as the number of ballot-boxes exceeds the number of groups).
\item \strong{Shuffling}: A fraction \code{lambda} of voters who have already been assigned is selected at random. Then, the ballot-box assignment of this sample is shuffled. Hence, different \code{lambda} values are interpreted as follows:
\itemize{
\item \code{lambda = 0} means no one is shuffled (the initial assignment remains).
\item \code{lambda = 1} means all individuals are shuffled.
\item Intermediate values like \code{lambda = 0.5} shuffle half the voters.
}
}

Using a high level of \code{lambda} (greater than 0.7) is not recommended, as this could make identification of the voting probabilities difficult. This is because higher values of lambda induce similar ballot-boxes in terms of voters' group.
}

\examples{
# Example 1: Default usage with 200 ballot boxes, each having 100 voters
result1 <- simulate_election(
    num_ballots = 200,
    num_candidates = 3,
    num_groups = 5
)

# Example 2: Using a custom ballot_voters vector
result2 <- simulate_election(
    num_ballots = 340,
    num_candidates = 3,
    num_groups = 7,
    ballot_voters = rep(200, 340)
)

# Example 3: Supplying group_proportions
result3 <- simulate_election(
    num_ballots = 93,
    num_candidates = 3,
    num_groups = 4,
    group_proportions = c(0.3, 0.5, 0.1, 0.1)
)

# Example 4: Providing a user-defined prob matrix
custom_prob <- matrix(c(
    0.9,  0.1,
    0.4,  0.6,
    0.25, 0.75,
    0.32, 0.68,
    0.2,  0.8
), nrow = 5, byrow = TRUE)

result4 <- simulate_election(
    num_ballots = 200,
    num_candidates = 2,
    num_groups = 5,
    lambda = 0.3,
    prob = custom_prob
)

result4$real_prob == custom_prob
# The attribute of the output real_prob matches the input custom_prob.
}
\references{
The algorithm is fully explained in \href{https://papers.ssrn.com/sol3/papers.cfm?abstract_id=4832834}{'Thraves, C. Ubilla, P. and Hermosilla, D.: \emph{"A Fast Ecological Inference Algorithm for the R×C Case"}}.
}
