% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_obs.R
\name{plot_obs}
\alias{plot_obs}
\title{Visualizations for an estimated ecological niche in covariate space}
\usage{
plot_obs(
  input,
  plot_cols = c("#8B3A3A", "#CCCCCC", "#0000CD"),
  alpha = input$p_critical,
  lower_lrr = NULL,
  upper_lrr = NULL,
  digits = 1,
  ...
)
}
\arguments{
\item{input}{An object of class 'list' from the \code{\link{lrren}} function.}

\item{plot_cols}{Character string of length three (3) specifying the colors for plotting: 1) presence, 2) neither, and 3) absence. The default colors in hex are \code{c("#8B3A3A", "#CCCCCC", "#0000CD")} or \code{c("indianred4", "grey80", "blue3")}.}

\item{alpha}{Optional, numeric. The two-tailed alpha level for significance threshold (default is the \code{p_critical} value imported from \code{input}).}

\item{lower_lrr}{Optional, numeric. Lower cut-off value for the log relative risk value in the color key (typically a negative value). The default is no limit, and the color key will include the minimum value of the log relative risk surface.}

\item{upper_lrr}{Optional, numeric. Upper cut-off value for the log relative risk value in the color key (typically a positive value). The default is no limit, and the color key will include the maximum value of the log relative risk surface.}

\item{digits}{Optional, integer. The number of significant digits for the color key labels using the \code{\link[base]{round}} function (default is 1).}

\item{...}{Arguments passed to \code{\link[spatstat.geom]{plot.ppp}} and \code{\link[fields]{image.plot}} for additional graphical features.}
}
\value{
This function produces three plots in a two-dimensional space where the axes are the two specified covariates: 1) observation locations by group, 2) log relative risk surface, and 3) significant p-value surface.
}
\description{
Create multiple plots of output from the \code{\link{lrren}} function, specifically for the observation data and estimated ecological niche.
}
\examples{
if (interactive()) {
  set.seed(1234) # for reproducibility

# Using the 'bei' and 'bei.extra' data within {spatstat.data}

# Covariate data (centered and scaled)
  elev <- spatstat.data::bei.extra[[1]]
  grad <- spatstat.data::bei.extra[[2]]
  elev$v <- scale(elev)
  grad$v <- scale(grad)
  elev_raster <- terra::rast(elev)
  grad_raster <- terra::rast(grad)

# Presence data
  presence <- spatstat.data::bei
  spatstat.geom::marks(presence) <- data.frame("presence" = rep(1, presence$n),
                                               "lon" = presence$x,
                                               "lat" = presence$y)
  spatstat.geom::marks(presence)$elev <- elev[presence]
  spatstat.geom::marks(presence)$grad <- grad[presence]

# (Pseudo-)Absence data
  absence <- spatstat.random::rpoispp(0.008, win = elev)
  spatstat.geom::marks(absence) <- data.frame("presence" = rep(0, absence$n),
                                              "lon" = absence$x,
                                              "lat" = absence$y)
  spatstat.geom::marks(absence)$elev <- elev[absence]
  spatstat.geom::marks(absence)$grad <- grad[absence]

# Combine into readable format
  obs_locs <- spatstat.geom::superimpose(presence, absence, check = FALSE)
  obs_locs <- spatstat.geom::marks(obs_locs)
  obs_locs$id <- seq(1, nrow(obs_locs), 1)
  obs_locs <- obs_locs[ , c(6, 2, 3, 1, 4, 5)]
  
# Run lrren
  test_lrren <- lrren(obs_locs = obs_locs,
                      cv = FALSE)
                      
# Run plot_obs   
  plot_obs(input = test_lrren)
}

}
