% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_cmat.R
\name{check_cmat}
\alias{check_cmat}
\title{Check constraint matrix irreducibility}
\usage{
check_cmat(Cmat)
}
\arguments{
\item{Cmat}{A constraint matrix as passed to \code{\link[=cirls.fit]{cirls.fit()}}}
}
\value{
A list with two elements:
\item{redundant}{Vector of indices of redundant constraints}
\item{equality}{Indicates which constraints are part of an underlying equality constraint}
}
\description{
Checks a constraint matrix does not contains redundant rows
}
\details{
The user typically doesn't need to use \code{check_cmat} as it is internally called by \code{\link[=cirls.control]{cirls.control()}}. However, it might be useful to undertsand if \code{Cmat} can be reduced for inference purpose. See the note in \code{\link[=confint.cirls]{confint.cirls()}}.

A constraint matrix is irreducible if no row can be expressed as a \emph{positive} linear combination of the other rows. When it happens, it means the constraint is actually implicitly included in other constraints in the matrix and can be dropped. Note that this a less restrictive condition than the constraint matrix having full row rank (see some examples).

The function starts by checking if some constraints are redundant and, if so, checks if they underline equality constraints. In the latter case, the constraint matrix can be reduced by expressing these constraints as a single equality constraint with identical lower and upper bounds (see \code{\link[=cirls.fit]{cirls.fit()}}).
}
\examples{
###################################################
# Example of reducible matrix

# Constraints: successive coefficients should increase and be convex
p <- 5
cmatic <- rbind(diff(diag(p)), diff(diag(p), diff = 2))

# Checking indicates that constraints 2 to 4 are redundant.
# Intuitively, if the first two coefficients increase,
# then convexity forces the rest to increase
check_cmat(cmatic)

# Check without contraints
check_cmat(cmatic[-(2:4),])

###################################################
# Example of irreducible matrix

# Constraints: coefficients form an S-shape
p <- 4
cmats <- rbind(
  diag(p)[1,], # positive
  diff(diag(p))[c(1, p - 1),], # Increasing at both end
  diff(diag(p), diff = 2)[1:(p/2 - 1),], # First half convex
  -diff(diag(p), diff = 2)[(p/2):(p-2),] # second half concave
)

# Note, this matrix is not of full row rank
qr(t(cmats))$rank
all.equal(cmats[2,] + cmats[4,] - cmats[5,], cmats[3,])

# However, it is irreducible: all constraints are necessary
check_cmat(cmats)

###################################################
# Example of underlying equality constraint

# Contraint: Parameters sum is >= 0 and sum is <= 0
cmateq <- rbind(rep(1, 3), rep(-1, 3))

# Checking indicates that both constraints imply equality constraint (sum == 0)
check_cmat(cmateq)
}
\references{
Meyer, M.C., 1999. An extension of the mixed primal–dual bases algorithm to the case of more constraints than dimensions. \emph{Journal of Statistical Planning and Inference} \strong{81}, 13–31. \doi{10.1016/S0378-3758(99)00025-7}
}
\seealso{
\code{\link[=confint.cirls]{confint.cirls()}}
}
