% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmtest.R
\name{glance.coeftest}
\alias{glance.coeftest}
\title{Glance at a(n) coeftest object}
\usage{
\method{glance}{coeftest}(x, ...)
}
\arguments{
\item{x}{A \code{coeftest} object returned from \code{\link[lmtest:coeftest]{lmtest::coeftest()}}.}

\item{...}{Additional arguments. Not used. Needed to match generic
signature only. \strong{Cautionary note:} Misspelled arguments will be
absorbed in \code{...}, where they will be ignored. If the misspelled
argument has a default value, the default value will be used.
For example, if you pass \code{conf.lvel = 0.9}, all computation will
proceed using \code{conf.level = 0.95}. Two exceptions here are:
\itemize{
\item \code{tidy()} methods will warn when supplied an \code{exponentiate} argument if
it will be ignored.
\item \code{augment()} methods will warn when supplied a \code{newdata} argument if it
will be ignored.
}}
}
\description{
Glance accepts a model object and returns a \code{\link[tibble:tibble]{tibble::tibble()}}
with exactly one row of model summaries. The summaries are typically
goodness of fit measures, p-values for hypothesis tests on residuals,
or model convergence information.

Glance never returns information from the original call to the modeling
function. This includes the name of the modeling function or any
arguments passed to the modeling function.

Glance does not calculate summary measures. Rather, it farms out these
computations to appropriate methods and gathers the results together.
Sometimes a goodness of fit measure will be undefined. In these cases
the measure will be reported as \code{NA}.

Glance returns the same number of columns regardless of whether the
model matrix is rank-deficient or not. If so, entries in columns
that no longer have a well-defined value are filled in with an \code{NA}
of the appropriate type.
}
\note{
Because of the way that lmtest::coeftest() retains information about
the underlying model object, the returned columns for glance.coeftest() will
vary depending on the arguments. Specifically, four columns are returned
regardless: "Loglik", "AIC", "BIC", and "nobs". Users can obtain additional
columns (e.g. "r.squared", "df") by invoking the "save = TRUE" argument as
part of lmtest::coeftest(). See examples.

As an aside, goodness-of-fit measures such as R-squared are unaffected by the
presence of heteroskedasticity. For further discussion see, e.g. chapter 8.1
of Wooldridge (2016).
}
\examples{
\dontshow{if (rlang::is_installed("lmtest")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

# load libraries for models and data
library(lmtest)

m <- lm(dist ~ speed, data = cars)

coeftest(m)
tidy(coeftest(m))
tidy(coeftest(m, conf.int = TRUE))

# a very common workflow is to combine lmtest::coeftest with alternate
# variance-covariance matrices via the sandwich package. The lmtest
# tidiers support this workflow too, enabling you to adjust the standard
# errors of your tidied models on the fly.
library(sandwich)

# "HC3" (default) robust SEs
tidy(coeftest(m, vcov = vcovHC))

# "HC2" robust SEs
tidy(coeftest(m, vcov = vcovHC, type = "HC2"))

# N-W HAC robust SEs
tidy(coeftest(m, vcov = NeweyWest))

# the columns of the returned tibble for glance.coeftest() will vary
# depending on whether the coeftest object retains the underlying model.
# Users can control this with the "save = TRUE" argument of coeftest().
glance(coeftest(m))
glance(coeftest(m, save = TRUE))
\dontshow{\}) # examplesIf}
}
\references{
Wooldridge, Jeffrey M. (2016) \cite{Introductory econometrics: A
modern approach.} (6th edition). Nelson Education.
}
\seealso{
\code{\link[=glance]{glance()}}, \code{\link[lmtest:coeftest]{lmtest::coeftest()}}
}
\concept{coeftest_tidiers}
\value{
A \code{\link[tibble:tibble]{tibble::tibble()}} with exactly one row and columns:
  \item{adj.r.squared}{Adjusted R squared statistic, which is like the R squared statistic except taking degrees of freedom into account.}
  \item{AIC}{Akaike's Information Criterion for the model.}
  \item{BIC}{Bayesian Information Criterion for the model.}
  \item{deviance}{Deviance of the model.}
  \item{df}{Degrees of freedom used by the model.}
  \item{df.residual}{Residual degrees of freedom.}
  \item{logLik}{The log-likelihood of the model. [stats::logLik()] may be a useful reference.}
  \item{nobs}{Number of observations used.}
  \item{p.value}{P-value corresponding to the test statistic.}
  \item{r.squared}{R squared statistic, or the percent of variation explained by the model. Also known as the coefficient of determination.}
  \item{sigma}{Estimated standard error of the residuals.}
  \item{statistic}{Test statistic.}

}
