% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/backbone_from_weighted.R
\name{backbone_from_weighted}
\alias{backbone_from_weighted}
\title{Extract the backbone from a weighted network}
\usage{
backbone_from_weighted(
  W,
  model = "disparity",
  alpha = 0.05,
  signed = FALSE,
  mtc = "none",
  parameter = 0,
  missing_as_zero = FALSE,
  narrative = FALSE,
  return = "backbone"
)
}
\arguments{
\item{W}{A weighted network as a valued adjacency matrix or \link[Matrix]{Matrix}, or a weighted unipartite \link[igraph]{igraph} object}

\item{model}{string: backbone model, one of: \code{"disparity"}, \code{"lans"}, \code{"mlf"}, or \code{"global"}}

\item{alpha}{real: significance level of hypothesis test(s) in statistical models}

\item{signed}{boolean: return a signed backbone from a statistical model}

\item{mtc}{string: type of Multiple Test Correction, either \code{"none"} or a method allowed by \code{\link[=p.adjust]{p.adjust()}}.}

\item{parameter}{real: parameter used to control structural backbone models}

\item{missing_as_zero}{boolean: treat missing edges as edges with zero weight and consider them for inclusion/exclusion in backbone}

\item{narrative}{boolean: display suggested text & citations}

\item{return}{string: return either only the \code{"backbone"} or \code{"everything"}}
}
\value{
If \code{return = "backbone"}, a backbone in the same class as \code{W}. If \code{return = "everything"}, then the backbone
is returned as an element in a list that also includes the original weighted network, (for statistical backbone models) the edgewise
p-values, a narrative description, and the original function call.
}
\description{
\code{backbone_from_weighted()} extracts the unweighted backbone from a weighted network
}
\details{
The \code{backbone_from_weighted} function extracts the backbone from a weighted unipartite network. The backbone is an unweighted
unipartite network that contains only edges whose weights are statistically significant (based on \code{alpha} for statistical models),
or which exhibit certain structural properties (based on \code{parameter} for structural models). For statistical models, when
\code{signed = FALSE}, the backbone contains edges that are statistically significantly strong under a one-tailed test. When
\code{signed = TRUE}, the backbone contains positive edges that are statistically significantly strong, and negative edges that are
statistically significantly weak, under a two-tailed test.

The \code{model} parameter controls the model used to evaluate the edge weights. The available models include:

\emph{Statistical Models} (controlled by \code{alpha}, \code{signed}, and \code{mtc})
\itemize{
\item \code{disparity} (default) - The disparity filter (Serrano et al., 2009)
\item \code{lans} - Locally adaptive network sparsification (Foti et al., 2011)
\item \code{mlf} - Marginal likelihood filter (Dianati, 2016)
}

\emph{Structural Models} (controlled by \code{parameter})
\itemize{
\item \code{global} - \code{parameter} is a numeric vector of length 1 or 2. If \code{length(parameter)==1}, then edges with weights
above \code{parameter} are preserved. If \code{length(parameter)==2}, then edges with weights above \code{max(parameter)} are
preserved as positive, and edges with weights above \code{min(parameter)} are preserved as negative.
}

The models implemented in \code{backbone_from_weighted()} can be applied to a weighted network that was obtained by projecting a
bipartite network or hypergraph. However, if the original bipartite network or hypergraph is available, it is better to use \code{\link[=backbone_from_projection]{backbone_from_projection()}}.
}
\examples{
#A weighted network with heterogeneous (i.e. multiscale) weights
W <- matrix(c(0,10,10,10,10,75,0,0,0,0,
              10,0,1,1,1,0,0,0,0,0,
              10,1,0,1,1,0,0,0,0,0,
              10,1,1,0,1,0,0,0,0,0,
              10,1,1,1,0,0,0,0,0,0,
              75,0,0,0,0,0,100,100,100,100,
              0,0,0,0,0,100,0,10,10,10,
              0,0,0,0,0,100,10,0,10,10,
              0,0,0,0,0,100,10,10,0,10,
              0,0,0,0,0,100,10,10,10,0),10)

W <- igraph::graph_from_adjacency_matrix(W, mode = "undirected", weighted = TRUE)
plot(W, edge.width = sqrt(igraph::E(W)$weight)) #A stronger clique & a weaker clique

mean_weight <- mean(igraph::E(W)$weight)  #Find average edge weight
bb <- backbone_from_weighted(W, model = "global", #A backbone with stronger-than-average edges...
      parameter = mean_weight)
plot(bb) #...ignores the weaker clique

bb <- backbone_from_weighted(W, model = "disparity") #A disparity filter backbone...
plot(bb) #...preserves edges at multiple scales
}
\references{
package: {Neal, Z. P. (2025). backbone: An R Package to Extract Network Backbones. CRAN. \doi{10.32614/CRAN.package.backbone}}

disparity: {Serrano, M. A., Boguna, M., & Vespignani, A. (2009). Extracting the multiscale backbone of complex weighted networks. \emph{Proceedings of the National Academy of Sciences, 106}, 6483-6488. \doi{10.1073/pnas.0808904106}}

lans: {Foti, N. J., Hughes, J. M., & Rockmore, D. N. (2011). Nonparametric sparsification of complex multiscale networks. \emph{PLOS One, 6}, e16431. \doi{10.1371/journal.pone.0016431}}

mlf: {Dianati, N. (2016). Unwinding the hairball graph: Pruning algorithms for weighted complex networks. \emph{Physical Review E, 93}, 012304. \doi{10.1103/PhysRevE.93.012304}}
}
