% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SVEMnet.R
\name{SVEMnet}
\alias{SVEMnet}
\title{Fit an SVEMnet Model (with optional relaxed elastic net)}
\usage{
SVEMnet(
  formula,
  data,
  nBoot = 200,
  glmnet_alpha = c(0.25, 0.5, 0.75, 1),
  weight_scheme = c("SVEM", "FWR", "Identity"),
  objective = c("auto", "wAIC", "wBIC", "wGIC", "wSSE"),
  auto_ratio_cutoff = 1.3,
  gamma = 2,
  relaxed = TRUE,
  ...
)
}
\arguments{
\item{formula}{A formula specifying the model to be fitted.}

\item{data}{A data frame containing the variables in the model.}

\item{nBoot}{Number of bootstrap iterations (default 200).}

\item{glmnet_alpha}{Elastic Net mixing parameter(s). May be a vector with entries in the range between 0 and 1, inclusive,
where alpha = 1 is Lasso and alpha = 0 is Ridge. Defaults to c(0.25, 0.5, 0.75, 1).}

\item{weight_scheme}{Weighting scheme for SVEM (default "SVEM").
One of "SVEM", "FWR", or "Identity".}

\item{objective}{Objective used to pick lambda on each bootstrap path
(default "auto"). One of "auto", "wAIC", "wBIC", "wGIC", or "wSSE".}

\item{auto_ratio_cutoff}{Single cutoff for the automatic rule when
objective = "auto" (default 1.3). Let r = n_X / p_X, where n_X is the
number of training rows and p_X is the number of predictors in the model
matrix after dropping the intercept column. If r >= auto_ratio_cutoff,
SVEMnet uses wAIC; otherwise it uses wBIC.}

\item{gamma}{Penalty weight used only when objective = "wGIC" (numeric, default 2).
Setting gamma = 2 reproduces wAIC. Larger values approach a BIC-like penalty.}

\item{relaxed}{Logical, TRUE or FALSE (default TRUE). When TRUE, use glmnet's
relaxed elastic net path and select both lambda and relaxed gamma on each bootstrap.
When FALSE, fit the standard glmnet path (equivalent to gamma = 1).
Note: if relaxed = TRUE and glmnet_alpha includes 0 (ridge), alpha = 0 is dropped.}

\item{...}{Additional args passed to glmnet() (e.g., penalty.factor,
lower.limits, upper.limits, offset, standardize.response, etc.).
Any user-supplied weights are ignored so SVEM can supply its own bootstrap weights.
Any user-supplied standardize is ignored; SVEMnet always uses standardize = TRUE.}
}
\value{
An object of class svem_model with elements:
\itemize{
\item parms: averaged coefficients (including intercept).
\item parms_debiased: averaged coefficients adjusted by the calibration fit.
\item debias_fit: lm(y ~ y_pred) calibration model used for debiasing (or NULL).
\item coef_matrix: per-bootstrap coefficient matrix.
\item nBoot, glmnet_alpha, best_alphas, best_lambdas, weight_scheme, relaxed.
\item best_relax_gammas: per-bootstrap relaxed gamma chosen (NA if relaxed = FALSE).
\item objective_input, objective_used, objective (same as objective_used),
auto_used, auto_decision, auto_rule, gamma (when wGIC).
\item dropped_alpha0_for_relaxed: whether alpha = 0 was removed because relaxed = TRUE.
\item schema: list(feature_names, terms_str, xlevels, contrasts, terms_hash) for safe predict.
\item sampling_schema: list(
predictor_vars, var_classes,
num_ranges = rbind(min=..., max=...) for numeric raw predictors,
factor_levels = list(...) for factor/character raw predictors).
\item diagnostics: list with k_summary (median and IQR of selected size),
fallback_rate, n_eff_summary, alpha_freq, relax_gamma_freq.
\item actual_y, training_X, y_pred, y_pred_debiased, nobs, nparm, formula, terms,
xlevels, contrasts.
}
}
\description{
Wrapper for glmnet (Friedman et al. 2010) to fit an ensemble of Elastic Net
models using the Self-Validated Ensemble Model method (SVEM; Lemkus et al. 2021),
with an option to use glmnet's built-in relaxed elastic net (Meinshausen 2007).
Supports searching over multiple alpha values in the Elastic Net penalty.
}
\details{
SVEM applies fractional bootstrap weights to training data and anti-correlated
weights for validation when weight_scheme = "SVEM". For each bootstrap, glmnet
paths are fit for each alpha in glmnet_alpha, and the lambda (and, if relaxed = TRUE,
relaxed gamma) minimizing a weighted validation criterion is selected.

Predictors are always standardized internally via glmnet::glmnet(..., standardize = TRUE).

When relaxed = TRUE, coef(fit, s = lambda, gamma = g) is used to obtain the
coefficient path at each relaxed gamma in the internal grid. Metrics are computed
from validation-weighted errors and model size is taken as the number of nonzero
coefficients including the intercept (support size), keeping selection consistent
between standard and relaxed paths. When relaxed = FALSE, gamma is fixed at 1.

Automatic objective rule ("auto"): This function uses a single ratio cutoff,
auto_ratio_cutoff, applied to r = n_X / p_X, where p_X is computed from
the model matrix with the intercept column removed. If r >= auto_ratio_cutoff
the selection criterion is wAIC; otherwise it is wBIC.

Implementation notes for safety:
\itemize{
\item The training terms object is stored with environment set to baseenv() to avoid
accidental lookups in the calling environment.
\item A compact schema (feature names, xlevels, contrasts) is stored to let predict()
reconstruct the design matrix deterministically.
\item A lightweight sampling schema (numeric ranges and factor levels for raw predictors)
is cached to enable random-table generation without needing the original data.
}
}
\section{Acknowledgments}{

Development of this package was assisted by GPT o1-preview for structuring parts of the code
and documentation.
}

\examples{
set.seed(42)

n  <- 30
X1 <- rnorm(n)
X2 <- rnorm(n)
X3 <- rnorm(n)
eps <- rnorm(n, sd = 0.5)
y <- 1 + 2*X1 - 1.5*X2 + 0.5*X3 + 1.2*(X1*X2) + 0.8*(X1^2) + eps
dat <- data.frame(y, X1, X2, X3)

mod_relax <- SVEMnet(
  y ~ (X1 + X2 + X3)^2 + I(X1^2) + I(X2^2),
  data          = dat,
  glmnet_alpha  = c(1, 0.5),
  nBoot         = 75,
  objective     = "auto",
  weight_scheme = "SVEM",
  relaxed       = FALSE #to run faster to pass CRAN checks
)

pred_in_raw <- predict(mod_relax, dat, debias = FALSE)
pred_in_db  <- predict(mod_relax, dat, debias = TRUE)

}
\references{
Gotwalt, C., & Ramsey, P. (2018). Model Validation Strategies for Designed Experiments Using
Bootstrapping Techniques With Applications to Biopharmaceuticals. JMP Discovery Conference.
https://community.jmp.com/t5/Abstracts/Model-Validation-Strategies-for-Designed-Experiments-Using/ev-p/849873/redirect_from_archived_page/true

Karl, A. T. (2024). A randomized permutation whole-model test heuristic for Self-Validated
Ensemble Models (SVEM). Chemometrics and Intelligent Laboratory Systems, 249, 105122.
doi:10.1016/j.chemolab.2024.105122

Karl, A., Wisnowski, J., & Rushing, H. (2022). JMP Pro 17 Remedies for Practical Struggles with
Mixture Experiments. JMP Discovery Conference. doi:10.13140/RG.2.2.34598.40003/1

Lemkus, T., Gotwalt, C., Ramsey, P., & Weese, M. L. (2021). Self-Validated Ensemble Models for
Design of Experiments. Chemometrics and Intelligent Laboratory Systems, 219, 104439.
doi:10.1016/j.chemolab.2021.104439

Xu, L., Gotwalt, C., Hong, Y., King, C. B., & Meeker, W. Q. (2020). Applications of the
Fractional-Random-Weight Bootstrap. The American Statistician, 74(4), 345-358.
doi:10.1080/00031305.2020.1731599

Ramsey, P., Gaudard, M., & Levin, W. (2021). Accelerating Innovation with Space Filling Mixture
Designs, Neural Networks and SVEM. JMP Discovery Conference.
https://community.jmp.com/t5/Abstracts/Accelerating-Innovation-with-Space-Filling-Mixture-Designs/ev-p/756841

Ramsey, P., & Gotwalt, C. (2018). Model Validation Strategies for Designed Experiments Using
Bootstrapping Techniques With Applications to Biopharmaceuticals. JMP Discovery Conference - Europe.
https://community.jmp.com/t5/Abstracts/Model-Validation-Strategies-for-Designed-Experiments-Using/ev-p/849647/redirect_from_archived_page/true

Ramsey, P., Levin, W., Lemkus, T., & Gotwalt, C. (2021). SVEM: A Paradigm Shift in Design and
Analysis of Experiments. JMP Discovery Conference - Europe.
https://community.jmp.com/t5/Abstracts/SVEM-A-Paradigm-Shift-in-Design-and-Analysis-of-Experiments-2021/ev-p/756634

Ramsey, P., & McNeill, P. (2023). CMC, SVEM, Neural Networks, DOE, and Complexity:
It's All About Prediction. JMP Discovery Conference.

Friedman, J. H., Hastie, T., & Tibshirani, R. (2010). Regularization Paths for Generalized
Linear Models via Coordinate Descent. Journal of Statistical Software, 33(1), 1-22.

Meinshausen, N. (2007).
Relaxed Lasso. Computational Statistics & Data Analysis, 52(1), 374-393.
}
