% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeItemsScale.R
\name{makeItemsScale}
\alias{makeItemsScale}
\title{Generate scale items from a summated scale, with desired Cronbach's Alpha}
\usage{
makeItemsScale(
  scale,
  lowerbound,
  upperbound,
  items,
  alpha = 0.8,
  variance = 0.5
)
}
\arguments{
\item{scale}{(int) a vector or dataframe of the summated rating scale.
Should range from ('lowerbound' * 'items') to ('upperbound' * 'items')}

\item{lowerbound}{(int) lower bound of the scale item
(example: '1' in a '1' to '5' rating)}

\item{upperbound}{(int) upper bound of the scale item
(example: '5' in a '1' to '5' rating)}

\item{items}{(positive, int) k, or number of columns to generate}

\item{alpha}{(posiitve, real) desired \emph{Cronbach's Alpha} for the
new dataframe of items.
Default = '0.8'.

See \verb{@details} for further information on the \code{alpha} parameter}

\item{variance}{(positive, real) the quantile from which to select
items that give given summated scores.
Must lie between '0' and '1'.
Default = '0.5'.

See \verb{@details} for further information on the \code{variance} parameter}
}
\value{
a dataframe with 'items' columns and 'length(scale)' rows
}
\description{
\code{makeItemsScale()} generates a random dataframe
of scale items based on a predefined summated scale
(such as created by the \code{lfast()} function),
and a desired \emph{Cronbach's Alpha}.

scale, lowerbound, upperbound, items, alpha, variance
}
\details{
\subsection{alpha}{

\code{makeItemsScale()} rearranges the item values within each row,
attempting to give a dataframe of Likert-scale items that produce a
predefined \emph{Cronbach's Alpha}.

Default value for target alpha is '0.8'.

More extreme values for the 'variance' parameter may reduce the chances
of achieving the desired Alpha. So you may need to experiment a little.
}

\subsection{variance}{

There may be many ways to find a combination of integers that sum to a
specific value, and these combinations have different levels of variance:
\itemize{
\item low-variance: '3 + 4 = 7'
\item high-variance: '1 + 6 = 7'
}

The 'variance' parameter defines guidelines for the amount of variance
among item values that your new dataframe should have.

For example, consider a summated value of '9' on which we apply
the \code{makeItemsScale()} function to generate three items.
With zero variance (variance parameter = '0'), then we see all items with
the same value, the mean of '3'.
With variance = '1', then we see all items with values
that give the maximum variance among those items.\tabular{lllll}{
   variance \tab v1 \tab v2 \tab v3 \tab sum \cr
   0.0 \tab 3 \tab 3 \tab 3 \tab 9 \cr
   0.2 \tab 3 \tab 3 \tab 3 \tab 9 \cr
   0.4 \tab 2 \tab 3 \tab 4 \tab 9 \cr
   0.6 \tab 1 \tab 4 \tab 4 \tab 9 \cr
   0.8 \tab 2 \tab 2 \tab 5 \tab 9 \cr
   1.0 \tab 1 \tab 3 \tab 5 \tab 9 \cr
}


Similarly, the same mean value applied to six items with
\code{makeItemsScale()} gives the following combinations at
different values of the 'variance' parameter.\tabular{llllllll}{
   variance \tab v1 \tab v2 \tab v3 \tab v4 \tab v5 \tab v6 \tab sum \cr
   0.0 \tab 3 \tab 3 \tab 3 \tab 3 \tab 3 \tab 3 \tab 18 \cr
   0.2 \tab 1 \tab 3 \tab 3 \tab 3 \tab 4 \tab 4 \tab 18 \cr
   0.4 \tab 1 \tab 2 \tab 3 \tab 4 \tab 4 \tab 4 \tab 18 \cr
   0.6 \tab 1 \tab 1 \tab 4 \tab 4 \tab 4 \tab 4 \tab 18 \cr
   0.8 \tab 1 \tab 1 \tab 3 \tab 4 \tab 4 \tab 5 \tab 18 \cr
   1.0 \tab 1 \tab 1 \tab 1 \tab 5 \tab 5 \tab 5 \tab 18 \cr
}


And a mean value of '3.5' gives the following combinations.\tabular{llllllll}{
   variance \tab v1 \tab v2 \tab v3 \tab v4 \tab v5 \tab v6 \tab sum \cr
   0.0 \tab 3 \tab 3 \tab 3 \tab 4 \tab 4 \tab 4 \tab 21 \cr
   0.2 \tab 3 \tab 3 \tab 3 \tab 3 \tab 4 \tab 5 \tab 21 \cr
   0.4 \tab 2 \tab 2 \tab 4 \tab 4 \tab 4 \tab 5 \tab 21 \cr
   0.6 \tab 1 \tab 3 \tab 4 \tab 4 \tab 4 \tab 5 \tab 21 \cr
   0.8 \tab 1 \tab 2 \tab 4 \tab 4 \tab 5 \tab 5 \tab 21 \cr
   1.0 \tab 1 \tab 1 \tab 4 \tab 5 \tab 5 \tab 5 \tab 21 \cr
}


The default value for 'variance' is '0.5' which gives a reasonable
range of item values.
But if you want 'responses' that are more consistent then choose
a lower variance value.
}
}
\examples{

## define parameters
k <- 4
lower <- 1
upper <- 5

## scale properties
n <- 64
mean <- 3.0
sd <- 0.85

## create scale
set.seed(42)
meanScale <- lfast(
  n = n, mean = mean, sd = sd,
  lowerbound = lower, upperbound = upper,
  items = k
)
summatedScale <- meanScale * k

## create new items
newItems <- makeItemsScale(
  scale = summatedScale,
  lowerbound = lower, upperbound = upper,
  items = k
)

### test new items
# str(newItems)
# alpha(data = newItems) |> round(2)


## very low variance usually gives higher Cronbach's Alpha
mydat_20 <- makeItemsScale(
  scale = summatedScale,
  lowerbound = lower, upperbound = upper,
  items = k, alpha = 0.8, variance = 0.20
)

### test new data frame
# str(mydat_20)

# moments <- data.frame(
#   means = apply(mydat_20, MARGIN = 2, FUN = mean) |> round(3),
#   sds = apply(mydat_20, MARGIN = 2, FUN = sd) |> round(3)
# ) |> t()

# moments

# cor(mydat_20) |> round(2)
# alpha(data = mydat_20) |> round(2)


## default alpha (0.8) and higher variance (0.8)
mydat_80 <- makeItemsScale(
  scale = summatedScale,
  lowerbound = lower, upperbound = upper,
  items = k, variance = 0.80
)

### test new dataframe
# str(mydat_80)

# moments <- data.frame(
#   means = apply(mydat_80, MARGIN = 2, FUN = mean) |> round(3),
#   sds = apply(mydat_80, MARGIN = 2, FUN = sd) |> round(3)
# ) |> t()

# moments

# cor(mydat_80) |> round(2)
# alpha(data = mydat_80) |> round(2)

}
