# WARNING - Generated by {fusen} from dev/flat_teaching.Rmd: do not edit by hand

#' Flexible `CSV`/`TXT` File Import with Multiple Backend Support
#'
#' @description
#' A comprehensive `CSV` or `TXT` file import function offering advanced reading capabilities 
#' through `data.table` and `arrow` packages with intelligent data combination strategies.
#'
#' @param file A `character` vector of file paths to `CSV` files.
#'   Must point to existing and accessible files.
#'
#' @param package A `character` string specifying the backend package:
#'   - `"data.table"`: Uses [`data.table::fread()`] (default)
#'   - `"arrow"`: Uses [`arrow::read_csv_arrow()`]
#'   Determines the underlying reading mechanism.
#'
#' @param rbind A `logical` value controlling data combination strategy:
#'   - `TRUE`: Combines all files into a single data object (default)
#'   - `FALSE`: Returns a list of individual data objects
#'
#' @param rbind_label A `character` string or `NULL` for source file tracking:
#'   - `character`: Specifies the column name for file source labeling (default: `"_file"`)
#'   - `NULL`: Disables source file tracking
#'
#' @param full_path A `logical` value controlling path display in file labels:
#'   - `TRUE`: Uses full file path
#'   - `FALSE`: Uses only filename (default)
#'
#' @param keep_ext A `logical` value controlling file extension in labels:
#'   - `TRUE`: Retains file extension (e.g., `.csv`)
#'   - `FALSE`: Removes file extension (default)
#'
#' @param ... Additional arguments passed to backend-specific reading functions 
#'   (e.g., `col_types`, `na.strings`, `skip`).
#'
#' @details
#' The function provides a unified interface for reading CSV files using either data.table
#' or arrow package. When reading multiple files, it can either combine them into a single
#' data object or return them as a list. File source tracking is supported through the
#' `rbind_label` parameter.
#'
#' File labeling behavior is controlled by `full_path` and `keep_ext` parameters:
#' \itemize{
#'   \item `full_path = FALSE, keep_ext = FALSE`: Filename without extension (e.g., `"data"`)
#'   \item `full_path = FALSE, keep_ext = TRUE`: Filename with extension (e.g., `"data.csv"`)
#'   \item `full_path = TRUE, keep_ext = FALSE`: Full path without extension (e.g., `"/path/to/data"`)
#'   \item `full_path = TRUE, keep_ext = TRUE`: Full path with extension (e.g., `"/path/to/data.csv"`)
#' }
#'
#' @return 
#' Depends on the `rbind` parameter:
#' \itemize{
#'   \item If `rbind = TRUE`: A single data object (from chosen package) 
#'     containing all imported data, with source file information in `rbind_label` column
#'   \item If `rbind = FALSE`: A named list of data objects with names 
#'     derived from input file paths based on `full_path` and `keep_ext` settings
#' }
#'
#' @note
#' Critical Import Considerations:
#' \itemize{
#'   \item Requires all specified files to be accessible `CSV/TXT` files
#'   \item Supports flexible backend selection via `package` parameter
#'   \item `rbind = TRUE` assumes compatible data structures across files
#'   \item Missing columns are automatically aligned when combining data
#'   \item File labeling is customizable through `full_path` and `keep_ext` parameters
#' }
#'
#' @seealso
#' \itemize{
#'   \item [`data.table::fread()`] for `data.table` backend
#'   \item [`arrow::read_csv_arrow()`] for `arrow` backend
#'   \item [`data.table::rbindlist()`] for data combination
#' }
#'
#' @import data.table
#' @import arrow
#'
#' @export
#' @examples
#' # Example: CSV file import demonstrations
#'
#' # Setup test files
#' csv_files <- mintyr_example(
#'   mintyr_examples("csv_test")     # Get example CSV files
#' )
#'
#' # Example 1: Import and combine CSV files using data.table
#' import_csv(
#'   csv_files,                      # Input CSV file paths
#'   package = "data.table",         # Use data.table for reading
#'   rbind = TRUE,                   # Combine all files into one data.table
#'   rbind_label = "_file",          # Column name for file source
#'   keep_ext = TRUE,                # Include .csv extension in _file column
#'   full_path = TRUE                # Show complete file paths in _file column
#' )
#'
#' # Example 2: Import files separately using arrow
#' import_csv(
#'   csv_files,                      # Input CSV file paths
#'   package = "arrow",              # Use arrow for reading
#'   rbind = FALSE                   # Keep files as separate data.tables
#' )
import_csv <- function (file, package = "data.table", rbind = TRUE, rbind_label = "_file", 
                        full_path = FALSE, keep_ext = FALSE, ...) {
  # Validations
  if (!is.character(file) || !all(file.exists(file))) {
    stop("file must be a vector of existing file paths.")
  }
  if (!package %in% c("data.table", "arrow")) {
    stop("package must be one of 'data.table', 'arrow'.")
  }
  if (!is.logical(full_path) || !is.logical(keep_ext)) {
    stop("full_path and keep_ext must be logical (TRUE or FALSE).")
  }
  
  # Function to process file name/path
  process_filename <- function(filename) {
    # Step 1: Path handling
    processed <- if (full_path) filename else basename(filename)
    
    # Step 2: Extension handling
    if (!keep_ext) {
      processed <- sub("\\.[^.]*$", "", processed)
    }
    
    return(processed)
  }
  
  # Read Functionality with naming
  read_files <- function(read_function) {
    file_data <- lapply(file, function(file_path) {
      df <- read_function(file_path, ...)
      if (!is.null(rbind_label) && rbind && length(file) > 1) {
        df <- cbind(stats::setNames(data.frame(process_filename(file_path)), rbind_label), df)
      }
      return(df)
    })
    if (rbind && length(file) > 1) {
      return(data.table::rbindlist(file_data, use.names = TRUE, fill = TRUE))
    } else {
      names(file_data) <- process_filename(file)
      return(file_data)
    }
  }
  
  # Package specific operations
  if (package == "data.table") {
    return(read_files(data.table::fread))
  } else if (package == "arrow") {
    return(read_files(arrow::read_csv_arrow))
  }
}
