/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5F_FRIEND      
#include "H5MFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5MFpkg.h"     

static H5FS_section_info_t *H5MF__sect_deserialize(const H5FS_section_class_t *cls, const uint8_t *buf,
                                                   haddr_t sect_addr, hsize_t sect_size, unsigned *des_flags);
static herr_t H5MF__sect_valid(const H5FS_section_class_t *cls, const H5FS_section_info_t *sect);
static H5FS_section_info_t *H5MF__sect_split(H5FS_section_info_t *sect, hsize_t frag_size);

static htri_t H5MF__sect_simple_can_merge(const H5FS_section_info_t *sect1, const H5FS_section_info_t *sect2,
                                          void *udata);
static herr_t H5MF__sect_simple_merge(H5FS_section_info_t **sect1, H5FS_section_info_t *sect2, void *udata);
static htri_t H5MF__sect_simple_can_shrink(const H5FS_section_info_t *_sect, void *udata);
static herr_t H5MF__sect_simple_shrink(H5FS_section_info_t **_sect, void *udata);

static herr_t H5MF__sect_small_add(H5FS_section_info_t **_sect, unsigned *flags, void *_udata);
static htri_t H5MF__sect_small_can_merge(const H5FS_section_info_t *sect1, const H5FS_section_info_t *sect2,
                                         void *udata);
static herr_t H5MF__sect_small_merge(H5FS_section_info_t **sect1, H5FS_section_info_t *sect2, void *udata);

static htri_t H5MF__sect_large_can_merge(const H5FS_section_info_t *sect1, const H5FS_section_info_t *sect2,
                                         void *udata);
static herr_t H5MF__sect_large_merge(H5FS_section_info_t **sect1, H5FS_section_info_t *sect2, void *udata);
static htri_t H5MF__sect_large_can_shrink(const H5FS_section_info_t *_sect, void *udata);
static herr_t H5MF__sect_large_shrink(H5FS_section_info_t **_sect, void *udata);

const H5FS_section_class_t H5MF_FSPACE_SECT_CLS_SIMPLE[1] = {{
    
    H5MF_FSPACE_SECT_SIMPLE,                 
    0,                                       
    H5FS_CLS_MERGE_SYM | H5FS_CLS_ADJUST_OK, 
    NULL,                                    

    
    NULL, 
    NULL, 

    
    NULL,                         
    NULL,                         
    H5MF__sect_deserialize,       
    H5MF__sect_simple_can_merge,  
    H5MF__sect_simple_merge,      
    H5MF__sect_simple_can_shrink, 
    H5MF__sect_simple_shrink,     
    H5MF__sect_free,              
    H5MF__sect_valid,             
    H5MF__sect_split,             
    NULL,                         
}};

const H5FS_section_class_t H5MF_FSPACE_SECT_CLS_SMALL[1] = {{
    
    H5MF_FSPACE_SECT_SMALL,                  
    0,                                       
    H5FS_CLS_MERGE_SYM | H5FS_CLS_ADJUST_OK, 
    NULL,                                    

    
    NULL, 
    NULL, 

    
    H5MF__sect_small_add,       
    NULL,                       
    H5MF__sect_deserialize,     
    H5MF__sect_small_can_merge, 
    H5MF__sect_small_merge,     
    NULL,                       
    NULL,                       
    H5MF__sect_free,            
    H5MF__sect_valid,           
    H5MF__sect_split,           
    NULL,                       
}};

const H5FS_section_class_t H5MF_FSPACE_SECT_CLS_LARGE[1] = {{
    
    H5MF_FSPACE_SECT_LARGE,                  
    0,                                       
    H5FS_CLS_MERGE_SYM | H5FS_CLS_ADJUST_OK, 
    NULL,                                    

    
    NULL, 
    NULL, 

    
    NULL,                        
    NULL,                        
    H5MF__sect_deserialize,      
    H5MF__sect_large_can_merge,  
    H5MF__sect_large_merge,      
    H5MF__sect_large_can_shrink, 
    H5MF__sect_large_shrink,     
    H5MF__sect_free,             
    H5MF__sect_valid,            
    H5MF__sect_split,            
    NULL,                        
}};

H5FL_DEFINE_STATIC(H5MF_free_section_t);

H5MF_free_section_t *
H5MF__sect_new(unsigned ctype, haddr_t sect_off, hsize_t sect_size)
{
    H5MF_free_section_t *sect;             
    H5MF_free_section_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(sect_size);

    
    if (NULL == (sect = H5FL_MALLOC(H5MF_free_section_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "memory allocation failed for direct block free list section");

    
    sect->sect_info.addr = sect_off;
    sect->sect_info.size = sect_size;

    
    sect->sect_info.type  = ctype;
    sect->sect_info.state = H5FS_SECT_LIVE;

    
    ret_value = sect;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5MF__sect_free(H5FS_section_info_t *_sect)
{
    H5MF_free_section_t *sect = (H5MF_free_section_t *)_sect; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);

    
    sect = H5FL_FREE(H5MF_free_section_t, sect);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static H5FS_section_info_t *
H5MF__sect_deserialize(const H5FS_section_class_t *cls, const uint8_t H5_ATTR_UNUSED *buf, haddr_t sect_addr,
                       hsize_t sect_size, unsigned H5_ATTR_UNUSED *des_flags)
{
    H5MF_free_section_t *sect;             
    H5FS_section_info_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(cls);
    assert(H5_addr_defined(sect_addr));
    assert(sect_size);

    
    if (NULL == (sect = H5MF__sect_new(cls->type, sect_addr, sect_size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, NULL, "can't initialize free space section");

    
    ret_value = (H5FS_section_info_t *)sect;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5MF__sect_valid(const H5FS_section_class_t H5_ATTR_UNUSED *cls, const H5FS_section_info_t
#ifdef NDEBUG
                                                                     H5_ATTR_UNUSED
#endif 
                                                                         *_sect)
{
#ifndef NDEBUG
    const H5MF_free_section_t *sect = (const H5MF_free_section_t *)_sect; 
#endif                                                                    

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static H5FS_section_info_t *
H5MF__sect_split(H5FS_section_info_t *sect, hsize_t frag_size)
{
    H5MF_free_section_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ret_value = H5MF__sect_new(sect->type, sect->addr, frag_size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, NULL, "can't initialize free space section");

    
    sect->addr += frag_size;
    sect->size -= frag_size;

done:
    FUNC_LEAVE_NOAPI((H5FS_section_info_t *)ret_value)
} 

static htri_t
H5MF__sect_simple_can_merge(const H5FS_section_info_t *_sect1, const H5FS_section_info_t *_sect2,
                            void H5_ATTR_UNUSED *_udata)
{
    const H5MF_free_section_t *sect1     = (const H5MF_free_section_t *)_sect1; 
    const H5MF_free_section_t *sect2     = (const H5MF_free_section_t *)_sect2; 
    htri_t                     ret_value = FAIL;                                

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect1);
    assert(sect2);
    assert(sect1->sect_info.type == sect2->sect_info.type); 
    assert(H5_addr_lt(sect1->sect_info.addr, sect2->sect_info.addr));

    
    ret_value = H5_addr_eq(sect1->sect_info.addr + sect1->sect_info.size, sect2->sect_info.addr);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5MF__sect_simple_merge(H5FS_section_info_t **_sect1, H5FS_section_info_t *_sect2,
                        void H5_ATTR_UNUSED *_udata)
{
    H5MF_free_section_t **sect1     = (H5MF_free_section_t **)_sect1; 
    H5MF_free_section_t  *sect2     = (H5MF_free_section_t *)_sect2;  
    herr_t                ret_value = SUCCEED;                        

    FUNC_ENTER_PACKAGE

    
    assert(sect1);
    assert((*sect1)->sect_info.type == H5MF_FSPACE_SECT_SIMPLE);
    assert(sect2);
    assert(sect2->sect_info.type == H5MF_FSPACE_SECT_SIMPLE);
    assert(H5_addr_eq((*sect1)->sect_info.addr + (*sect1)->sect_info.size, sect2->sect_info.addr));

    
    (*sect1)->sect_info.size += sect2->sect_info.size;

    
    if (H5MF__sect_free((H5FS_section_info_t *)sect2) < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL, "can't free section node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5MF__sect_simple_can_shrink(const H5FS_section_info_t *_sect, void *_udata)
{
    const H5MF_free_section_t *sect  = (const H5MF_free_section_t *)_sect; 
    H5MF_sect_ud_t            *udata = (H5MF_sect_ud_t *)_udata;           
    haddr_t                    eoa;              
    haddr_t                    end;              
    htri_t                     ret_value = FAIL; 

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(udata);
    assert(udata->f);

    
    if (HADDR_UNDEF == (eoa = H5F_get_eoa(udata->f, udata->alloc_type)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "driver get_eoa request failed");

    
    end = sect->sect_info.addr + sect->sect_info.size;

    
    if (H5_addr_eq(end, eoa)) {
        
        udata->shrink = H5MF_SHRINK_EOA;
#ifdef H5MF_ALLOC_DEBUG_MORE
        Rfprintf(Rstderr, "%s: section {%" PRIuHADDR ", %" PRIuHSIZE "}, shrinks file, eoa = %" PRIuHADDR "\n",
                __func__, sect->sect_info.addr, sect->sect_info.size, eoa);
#endif 

        
        HGOTO_DONE(true);
    } 
    else {
        
        if (udata->allow_eoa_shrink_only)
            HGOTO_DONE(false);

        
        if (udata->f->shared->fs_aggr_merge[udata->alloc_type] & H5F_FS_MERGE_METADATA) {
            htri_t status; 

            
            if ((status = H5MF__aggr_can_absorb(udata->f, &(udata->f->shared->meta_aggr), sect,
                                                &(udata->shrink))) < 0)
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTMERGE, FAIL,
                            "error merging section with aggregation block");
            else if (status > 0) {
                
                udata->aggr = &(udata->f->shared->meta_aggr);
#ifdef H5MF_ALLOC_DEBUG_MORE
                Rfprintf(Rstderr, "%s: section {%" PRIuHADDR ", %" PRIuHSIZE "}, adjoins metadata aggregator\n",
                        __func__, sect->sect_info.addr, sect->sect_info.size);
#endif 

                
                HGOTO_DONE(true);
            } 
        }     

        
        if (udata->f->shared->fs_aggr_merge[udata->alloc_type] & H5F_FS_MERGE_RAWDATA) {
            htri_t status; 

            
            if ((status = H5MF__aggr_can_absorb(udata->f, &(udata->f->shared->sdata_aggr), sect,
                                                &(udata->shrink))) < 0)
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTMERGE, FAIL,
                            "error merging section with aggregation block");
            else if (status > 0) {
                
                udata->aggr = &(udata->f->shared->sdata_aggr);
#ifdef H5MF_ALLOC_DEBUG_MORE
                Rfprintf(Rstderr,
                        "%s: section {%" PRIuHADDR ", %" PRIuHSIZE "}, adjoins small data aggregator\n",
                        __func__, sect->sect_info.addr, sect->sect_info.size);
#endif 

                
                HGOTO_DONE(true);
            } 
        }     
    }         

    
    ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5MF__sect_simple_shrink(H5FS_section_info_t **_sect, void *_udata)
{
    H5MF_free_section_t **sect      = (H5MF_free_section_t **)_sect; 
    H5MF_sect_ud_t       *udata     = (H5MF_sect_ud_t *)_udata;      
    herr_t                ret_value = SUCCEED;                       

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(udata);
    assert(udata->f);

    
    if (H5MF_SHRINK_EOA == udata->shrink) {
        
        assert(H5F_INTENT(udata->f) & H5F_ACC_RDWR);

        
        if (H5F__free(udata->f, udata->alloc_type, (*sect)->sect_info.addr, (*sect)->sect_info.size) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, FAIL, "driver free request failed");
    } 
    else {
        
        assert(udata->aggr);

        
        if (H5MF__aggr_absorb(udata->f, udata->aggr, *sect, udata->allow_sect_absorb) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTMERGE, FAIL,
                        "can't absorb section into aggregator or vice versa");
    } 

    
    if (udata->shrink != H5MF_SHRINK_SECT_ABSORB_AGGR) {
        
        if (H5MF__sect_free((H5FS_section_info_t *)*sect) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL, "can't free simple section node");

        
        *sect = NULL;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5MF__sect_small_add(H5FS_section_info_t **_sect, unsigned *flags, void *_udata)
{
    H5MF_free_section_t **sect  = (H5MF_free_section_t **)_sect; 
    H5MF_sect_ud_t       *udata = (H5MF_sect_ud_t *)_udata;      
    haddr_t               sect_end;
    hsize_t               rem, prem;
    herr_t                ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

#ifdef H5MF_ALLOC_DEBUG_MORE
    Rfprintf(Rstderr, "%s: Entering, section {%" PRIuHADDR ", %" PRIuHSIZE "}\n", __func__,
            (*sect)->sect_info.addr, (*sect)->sect_info.size);
#endif 

    
    if (udata->alloc_type == H5FD_MEM_DRAW || udata->alloc_type == H5FD_MEM_GHEAP)
        HGOTO_DONE(ret_value);

    sect_end = (*sect)->sect_info.addr + (*sect)->sect_info.size;
    if (0 == udata->f->shared->fs_page_size)
        HGOTO_ERROR(H5E_RESOURCE, H5E_BADVALUE, FAIL, "page size of zero would result in division by zero");
    rem  = sect_end % udata->f->shared->fs_page_size;
    prem = udata->f->shared->fs_page_size - rem;

    
    if (!rem && (*sect)->sect_info.size <= H5F_PGEND_META_THRES(udata->f) &&
        (*flags & H5FS_ADD_RETURNED_SPACE)) {
        if (H5MF__sect_free((H5FS_section_info_t *)(*sect)) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL, "can't free section node");
        *sect = NULL;
        *flags &= (unsigned)~H5FS_ADD_RETURNED_SPACE;
        *flags |= H5FS_PAGE_END_NO_ADD;
#ifdef H5MF_ALLOC_DEBUG_MORE
        Rfprintf(Rstderr, "%s: section is dropped\n", __func__);
#endif 
    }  
    
    else if (prem <= H5F_PGEND_META_THRES(udata->f)) {
        (*sect)->sect_info.size += prem;
#ifdef H5MF_ALLOC_DEBUG_MORE
        Rfprintf(Rstderr, "%s: section is adjusted {%" PRIuHADDR ", %" PRIuHSIZE "}\n", __func__,
                (*sect)->sect_info.addr, (*sect)->sect_info.size);
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5MF__sect_small_can_merge(const H5FS_section_info_t *_sect1, const H5FS_section_info_t *_sect2, void *_udata)
{
    const H5MF_free_section_t *sect1     = (const H5MF_free_section_t *)_sect1; 
    const H5MF_free_section_t *sect2     = (const H5MF_free_section_t *)_sect2; 
    H5MF_sect_ud_t            *udata     = (H5MF_sect_ud_t *)_udata;            
    htri_t                     ret_value = false;                               

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect1);
    assert(sect2);
    assert(sect1->sect_info.type == sect2->sect_info.type); 
    assert(H5_addr_lt(sect1->sect_info.addr, sect2->sect_info.addr));

    
    ret_value = H5_addr_eq(sect1->sect_info.addr + sect1->sect_info.size, sect2->sect_info.addr);
    if (ret_value > 0)
        
        if ((sect1->sect_info.addr / udata->f->shared->fs_page_size) !=
            (((sect2->sect_info.addr + sect2->sect_info.size - 1) / udata->f->shared->fs_page_size)))
            ret_value = false;

#ifdef H5MF_ALLOC_DEBUG_MORE
    Rfprintf(Rstderr, "%s: Leaving: ret_value = %d\n", __func__, ret_value);
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5MF__sect_small_merge(H5FS_section_info_t **_sect1, H5FS_section_info_t *_sect2, void *_udata)
{
    H5MF_free_section_t **sect1     = (H5MF_free_section_t **)_sect1; 
    H5MF_free_section_t  *sect2     = (H5MF_free_section_t *)_sect2;  
    H5MF_sect_ud_t       *udata     = (H5MF_sect_ud_t *)_udata;       
    herr_t                ret_value = SUCCEED;                        

    FUNC_ENTER_PACKAGE

    
    assert(sect1);
    assert((*sect1)->sect_info.type == H5MF_FSPACE_SECT_SMALL);
    assert(sect2);
    assert(sect2->sect_info.type == H5MF_FSPACE_SECT_SMALL);
    assert(H5_addr_eq((*sect1)->sect_info.addr + (*sect1)->sect_info.size, sect2->sect_info.addr));

    
    (*sect1)->sect_info.size += sect2->sect_info.size;

    if ((*sect1)->sect_info.size == udata->f->shared->fs_page_size) {
        if (H5MF_xfree(udata->f, udata->alloc_type, (*sect1)->sect_info.addr, (*sect1)->sect_info.size) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, FAIL, "can't free merged section");

        
        
        
        
        if (udata->f->shared->page_buf != NULL && udata->alloc_type != H5FD_MEM_DRAW)
            if (H5PB_remove_entry(udata->f->shared, (*sect1)->sect_info.addr) < 0)
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, FAIL, "can't free merged section");

        if (H5MF__sect_free((H5FS_section_info_t *)(*sect1)) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL, "can't free section node");
        *sect1 = NULL;
    } 

    
    if (H5MF__sect_free((H5FS_section_info_t *)sect2) < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL, "can't free section node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5MF__sect_large_can_merge(const H5FS_section_info_t *_sect1, const H5FS_section_info_t *_sect2,
                           void H5_ATTR_UNUSED *_udata)
{
    const H5MF_free_section_t *sect1     = (const H5MF_free_section_t *)_sect1; 
    const H5MF_free_section_t *sect2     = (const H5MF_free_section_t *)_sect2; 
    htri_t                     ret_value = false;                               

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sect1);
    assert(sect2);
    assert(sect1->sect_info.type == sect2->sect_info.type); 
    assert(H5_addr_lt(sect1->sect_info.addr, sect2->sect_info.addr));

    ret_value = H5_addr_eq(sect1->sect_info.addr + sect1->sect_info.size, sect2->sect_info.addr);

#ifdef H5MF_ALLOC_DEBUG_MORE
    Rfprintf(Rstderr, "%s: Leaving: ret_value = %d\n", __func__, ret_value);
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5MF__sect_large_merge(H5FS_section_info_t **_sect1, H5FS_section_info_t *_sect2, void H5_ATTR_UNUSED *_udata)
{
    H5MF_free_section_t **sect1     = (H5MF_free_section_t **)_sect1; 
    H5MF_free_section_t  *sect2     = (H5MF_free_section_t *)_sect2;  
    herr_t                ret_value = SUCCEED;                        

    FUNC_ENTER_PACKAGE

    
    assert(sect1);
    assert((*sect1)->sect_info.type == H5MF_FSPACE_SECT_LARGE);
    assert(sect2);
    assert(sect2->sect_info.type == H5MF_FSPACE_SECT_LARGE);
    assert(H5_addr_eq((*sect1)->sect_info.addr + (*sect1)->sect_info.size, sect2->sect_info.addr));

    
    (*sect1)->sect_info.size += sect2->sect_info.size;

    
    if (H5MF__sect_free((H5FS_section_info_t *)sect2) < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL, "can't free section node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5MF__sect_large_can_shrink(const H5FS_section_info_t *_sect, void *_udata)
{
    const H5MF_free_section_t *sect  = (const H5MF_free_section_t *)_sect; 
    H5MF_sect_ud_t            *udata = (H5MF_sect_ud_t *)_udata;           
    haddr_t                    eoa;               
    haddr_t                    end;               
    htri_t                     ret_value = false; 

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(sect->sect_info.type == H5MF_FSPACE_SECT_LARGE);
    assert(udata);
    assert(udata->f);

    
    if (HADDR_UNDEF == (eoa = H5FD_get_eoa(udata->f->shared->lf, udata->alloc_type)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "driver get_eoa request failed");

    
    end = sect->sect_info.addr + sect->sect_info.size;

    
    if (H5_addr_eq(end, eoa) && sect->sect_info.size >= udata->f->shared->fs_page_size) {
        
        udata->shrink = H5MF_SHRINK_EOA;
#ifdef H5MF_ALLOC_DEBUG_MORE
        Rfprintf(Rstderr, "%s: section {%" PRIuHADDR ", %" PRIuHSIZE "}, shrinks file, eoa = %" PRIuHADDR "\n",
                __func__, sect->sect_info.addr, sect->sect_info.size, eoa);
#endif 

        
        HGOTO_DONE(true);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5MF__sect_large_shrink(H5FS_section_info_t **_sect, void *_udata)
{
    H5MF_free_section_t **sect      = (H5MF_free_section_t **)_sect; 
    H5MF_sect_ud_t       *udata     = (H5MF_sect_ud_t *)_udata;      
    hsize_t               frag_size = 0;                             
    herr_t                ret_value = SUCCEED;                       

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert((*sect)->sect_info.type == H5MF_FSPACE_SECT_LARGE);
    assert(udata);
    assert(udata->f);
    assert(udata->shrink == H5MF_SHRINK_EOA);
    assert(H5F_INTENT(udata->f) & H5F_ACC_RDWR);
    assert(H5F_PAGED_AGGR(udata->f));

    
    H5MF_EOA_MISALIGN(udata->f, (*sect)->sect_info.addr, udata->f->shared->fs_page_size, frag_size);

    
    
    if (H5F__free(udata->f, udata->alloc_type, (*sect)->sect_info.addr + frag_size,
                  (*sect)->sect_info.size - frag_size) < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, FAIL, "driver free request failed");

    if (frag_size) 
        (*sect)->sect_info.size = frag_size;
    else {
        
        if (H5MF__sect_free((H5FS_section_info_t *)*sect) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL, "can't free simple section node");

        
        *sect = NULL;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
