% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/state_to_trajectory.R
\name{state_to_trajectory}
\alias{state_to_trajectory}
\title{Position of a state with respect to a trajectory}
\usage{
state_to_trajectory(
  d,
  trajectories,
  states,
  target_states,
  reference,
  method,
  coordStates = NULL
)
}
\arguments{
\item{d}{Either a symmetric matrix or an object of class \code{\link{dist}} containing
the dissimilarities between each pair of states.}

\item{trajectories}{Vector indicating the trajectory or site to which each
state in \code{d} belongs.}

\item{states}{Vector of integers indicating the order of the states in \code{d} for
each trajectory (assign 1 if the state does not belong to any trajectory).}

\item{target_states}{Vector of integers indicating the indices in \code{trajectories}
and \code{states} of the ecological states for which their relative position will
be calculated.}

\item{reference}{Vector of the same class of \code{trajectories} or object of class
\code{RETRA} indicating the reference trajectory to calculate the relative position
of the \code{target_states}}

\item{method}{Method to calculate the distance and relative position of the
\code{target_states} and the \code{reference}. One of \code{"nearest_state"}, \code{"projection"}
or \code{"mixed"} (see Details).}

\item{coordStates}{Matrix containing the coordinates of each state (rows) and
axis (columns) of a metric ordination space (see Details)}
}
\value{
The function \code{state_to_trajectory()} returns a data frame of four columns
including the \code{distance} and \code{relative_position} between the \code{target_state} and
the \code{reference}.
\itemize{
\item Depending on the \code{method}, \code{distance} is calculated as the dissimilarity
between the \code{target_states} and their respective nearest state in the \code{reference}
or the dissimilarity to their projections onto the \code{reference}.
\item The \code{relative_position} is a value that ranges between 0 (if the nearest
state or projected point coincides with the first \code{reference} state) and 1
(if the nearest state or projected point coincides with the last \code{reference}
state).
}
}
\description{
Define the position of a state with respect to a reference trajectory based on
its distance from the trajectory and the length and direction of the trajectory.
}
\details{
\code{state_to_trajectory()} can calculate the distance and relative position of
one or more \code{target_states} relative to a \code{reference} trajectory by three
different methods:
\itemize{
\item \code{"nearest_state"} returns the dissimilarity of the \code{target_states} to the
nearest state of the \code{reference} trajectory (\code{distance}) and calculates the
relative position of the nearest state within the \code{reference}.
\item \code{"projection"} returns the dissimilarity of the \code{target_states} to their
projection onto the \code{reference} trajectory and calculates the relative position
of the projected state within the \code{reference}. This method requires \code{d} to be
metric (i.e. to satisfy the triangle inequality). If \code{d} is not metric,
\code{state_to_trajectory()} calculates the Euclidean distance within a transformed
space generated through multidimensional scaling (Borg and Groenen, 2005). To
use the state coordinates in a different metric space, use the \code{coordStates}
argument. When the \code{target_states} cannot be projected onto any of the segments
forming the \code{reference} trajectory, \code{state_to_trajectory()} returns \code{NA} for
both \code{distance} and \code{relative_position}.
\item \code{"mixed"} calculates the dissimilarity between the \code{target_states} and the
\code{reference} trajectory, as well as their relative position by computing its
projection onto any of the segments of the reference (analogous to
\code{method = "projection"}). For the \code{target_states} that cannot be projected,
\code{state_to_trajectory()} uses the nearest state in the \code{reference} to compute
\code{distance} and \code{relative_position} (analogous to \code{method = "nearest_state"}).
}
}
\examples{
# State dissimilarities
d <- vegan::vegdist(EDR_data$EDR3$abundance[, paste0("sp", 1:12)], method = "bray")
trajectories <- EDR_data$EDR3$abundance$traj
states <- EDR_data$EDR3$abundance$state

# Calculate the representative trajectories of an EDR to be used as reference
RT <- retra_edr(d,
               trajectories = trajectories,
               states = states,
               minSegs = 10)

# Define the target states
target_states <- as.integer(c(1, 16, 55))

# Calculate the position of the target states with respect to  the representative
# trajectories of an EDR
state_to_trajectory(d, trajectories = trajectories,
                    states = states,
                    target_states = target_states,
                    reference = RT,
                    method = "nearest_state")

}
\author{
Martina Sánchez-Pinillos
}
