% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/alm.R
\name{alm}
\alias{alm}
\alias{alm.gp}
\alias{alm.dgp}
\alias{alm.bundle}
\title{Locate the next design point(s) for a (D)GP emulator or a bundle of (D)GP emulators using Active Learning MacKay (ALM)}
\usage{
alm(object, ...)

\method{alm}{gp}(
  object,
  x_cand = NULL,
  n_start = 20,
  batch_size = 1,
  M = 50,
  workers = 1,
  limits = NULL,
  int = FALSE,
  ...
)

\method{alm}{dgp}(
  object,
  x_cand = NULL,
  n_start = 20,
  batch_size = 1,
  M = 50,
  workers = 1,
  limits = NULL,
  int = FALSE,
  aggregate = NULL,
  ...
)

\method{alm}{bundle}(
  object,
  x_cand = NULL,
  n_start = 20,
  batch_size = 1,
  M = 50,
  workers = 1,
  limits = NULL,
  int = FALSE,
  aggregate = NULL,
  ...
)
}
\arguments{
\item{object}{can be one of the following:
\itemize{
\item the S3 class \code{gp}.
\item the S3 class \code{dgp}.
\item the S3 class \code{bundle}.
}}

\item{...}{any arguments (with names different from those of arguments used in \code{\link[=alm]{alm()}}) that are used by \code{aggregate}
can be passed here.}

\item{x_cand}{a matrix (with each row being a design point and column being an input dimension) that gives a candidate set
from which the next design point(s) are determined. If \code{object} is an instance of the \code{bundle} class and \code{aggregate} is not supplied, \code{x_cand} can also be a list.
The list must have a length equal to the number of emulators in \code{object}, with each element being a matrix representing the candidate set for a corresponding
emulator in the bundle. Defaults to \code{NULL}.}

\item{n_start}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#new}{\figure{lifecycle-new.svg}{options: alt='[New]'}}}{\strong{[New]}} an integer that gives the number of initial design points to be used to determine next design point(s). This argument
is only used when \code{x_cand} is \code{NULL}. Defaults to \code{20}.}

\item{batch_size}{an integer that gives the number of design points to be chosen. Defaults to \code{1}.}

\item{M}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#new}{\figure{lifecycle-new.svg}{options: alt='[New]'}}}{\strong{[New]}} the size of the conditioning set for the Vecchia approximation in the criterion calculation. This argument is only used if the emulator \code{object}
was constructed under the Vecchia approximation. Defaults to \code{50}.}

\item{workers}{the number of processes to be used for design point selection. If set to \code{NULL},
the number of processes is set to \verb{max physical cores available \%/\% 2}. Defaults to \code{1}. The argument does not currently support Windows machines when the \code{aggregate}
function is provided, due to the significant overhead caused by initializing the Python environment for each worker under spawning.}

\item{limits}{a two-column matrix that gives the ranges of each input dimension, or a vector of length two if there is only one input dimension.
If a vector is provided, it will be converted to a two-column row matrix. The rows of the matrix correspond to input dimensions, and its
first and second columns correspond to the minimum and maximum values of the input dimensions. This
argument is only used when \code{x_cand = NULL}. Defaults to \code{NULL}.}

\item{int}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#new}{\figure{lifecycle-new.svg}{options: alt='[New]'}}}{\strong{[New]}} a bool or a vector of bools that indicates if an input dimension is an integer type. If a single bool is given, it will be applied to
all input dimensions. If a vector is provided, it should have a length equal to the input dimensions and will be applied to individual
input dimensions. This argument is only used when \code{x_cand = NULL}. Defaults to \code{FALSE}.}

\item{aggregate}{an R function that aggregates scores of the ALM across different output dimensions (if \code{object} is an instance
of the \code{dgp} class) or across different emulators (if \code{object} is an instance of the \code{bundle} class). The function should be specified in the
following basic form:
\itemize{
\item the first argument is a matrix representing scores. The rows of the matrix correspond to different design points. The number of columns
of the matrix is equal to:
\itemize{
\item the emulator output dimension if \code{object} is an instance of the \code{dgp} class; or
\item the number of emulators contained in \code{object} if \code{object} is an instance of the \code{bundle} class.
}
\item the output should be a vector that gives aggregate scores at different design points.
}

Set to \code{NULL} to disable aggregation. Defaults to \code{NULL}.}
}
\value{
\enumerate{
\item If \code{x_cand} is not \code{NULL}:
\itemize{
\item When \code{object} is an instance of the \code{gp} class, a vector of length \code{batch_size} is returned, containing the positions
(row numbers) of the next design points from \code{x_cand}.
\item When \code{object} is an instance of the \code{dgp} class, a vector of length \code{batch_size * D} is returned, containing the positions
(row numbers) of the next design points from \code{x_cand} to be added to the DGP emulator.
\itemize{
\item \code{D} is the number of output dimensions of the DGP emulator if no likelihood layer is included.
\item For a DGP emulator with a \code{Hetero} or \code{NegBin} likelihood layer, \code{D = 2}.
\item For a DGP emulator with a \code{Categorical} likelihood layer, \code{D = 1} for binary output or \code{D = K} for multi-class output with \code{K} classes.
}
\item When \code{object} is an instance of the \code{bundle} class, a matrix is returned with \code{batch_size} rows and a column for each emulator in
the bundle, containing the positions (row numbers) of the next design points from \code{x_cand} for individual emulators.
}
\item If \code{x_cand} is \code{NULL}:
\itemize{
\item When \code{object} is an instance of the \code{gp} class, a matrix with \code{batch_size} rows is returned, giving the next design points to be evaluated.
\item When \code{object} is an instance of the \code{dgp} class, a matrix with \code{batch_size * D} rows is returned, where:
\itemize{
\item \code{D} is the number of output dimensions of the DGP emulator if no likelihood layer is included.
\item For a DGP emulator with a \code{Hetero} or \code{NegBin} likelihood layer, \code{D = 2}.
\item For a DGP emulator with a \code{Categorical} likelihood layer, \code{D = 1} for binary output or \code{D = K} for multi-class output with \code{K} classes.
}
\item When \code{object} is an instance of the \code{bundle} class, a list is returned with a length equal to the number of emulators in the bundle. Each
element of the list is a matrix with \code{batch_size} rows, where each row represents a design point to be added to the corresponding emulator.
}
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#updated}{\figure{lifecycle-updated.svg}{options: alt='[Updated]'}}}{\strong{[Updated]}}

This function searches from a candidate set to locate the next design point(s) to be added to a (D)GP emulator
or a bundle of (D)GP emulators using the Active Learning MacKay (ALM) criterion (see the reference below).
}
\details{
See further examples and tutorials at \url{https://mingdeyu.github.io/dgpsi-R/}.
}
\note{
The first column of the matrix supplied to the first argument of \code{aggregate} must correspond to the first output dimension of the DGP emulator
if \code{object} is an instance of the \code{dgp} class, and so on for subsequent columns and dimensions. If \code{object} is an instance of the \code{bundle} class,
the first column must correspond to the first emulator in the bundle, and so on for subsequent columns and emulators.
}
\examples{
\dontrun{

# load packages and the Python env
library(lhs)
library(dgpsi)

# construct a 1D non-stationary function
f <- function(x) {
 sin(30*((2*x-1)/2-0.4)^5)*cos(20*((2*x-1)/2-0.4))
}

# generate the initial design
X <- maximinLHS(10,1)
Y <- f(X)

# training a 2-layered DGP emulator with the global connection off
m <- dgp(X, Y, connect = F)

# specify the input range
lim <- c(0,1)

# locate the next design point using ALM
X_new <- alm(m, limits = lim)

# obtain the corresponding output at the located design point
Y_new <- f(X_new)

# combine the new input-output pair to the existing data
X <- rbind(X, X_new)
Y <- rbind(Y, Y_new)

# update the DGP emulator with the new input and output data and refit
m <- update(m, X, Y, refit = TRUE)

# plot the LOO validation
plot(m)
}
}
\references{
MacKay, D. J. (1992). Information-based objective functions for active data selection. \emph{Neural Computation}, \strong{4(4)}, 590-604.
}
