% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SterrettFunctions.R
\name{Sterrett}
\alias{Sterrett}
\title{Summary measures for Sterrett algorithms}
\usage{
Sterrett(
  p,
  Sp,
  Se,
  plot = FALSE,
  plot.cut.dorf = FALSE,
  cond.prob.plot = FALSE,
  font.name = "sans"
)
}
\arguments{
\item{p}{a vector of individual risk probabilities.}

\item{Sp}{the specificity of the diagnostic test.}

\item{Se}{the sensitivity of the diagnostic test.}

\item{plot}{logical; if TRUE, a plot of the informative Sterrett
CDFs will be displayed. Further details are given under 'Details'.}

\item{plot.cut.dorf}{logical; if TRUE, the cut-tree for Dorfman
testing will be displayed. Further details are given under 'Details'.}

\item{cond.prob.plot}{logical; if TRUE, a second axis for the
conditional probability plot will be displayed on the right side
of the plot.}

\item{font.name}{the name of the font to be used in plots.}
}
\value{
A list containing:
\item{mean.sd}{a data frame containing the mean and standard deviation
of the expected number of tests for one-stage informative Sterrett (1SIS),
two-stage informative Sterrett (2SIS), full informative Sterrett (FIS),
and Dorfman testing.}
\item{PMF}{a data frame containing the probability mass function
for the number of tests possible for one-stage informative Sterrett (1SIS),
two-stage informative Sterrett (2SIS), full informative Sterrett (FIS),
and Dorfman testing.}
\item{CDF}{a data frame containing the cumulative distribution function
for the number of tests possible for one-stage informative Sterrett (1SIS),
two-stage informative Sterrett (2SIS), full informative Sterrett (FIS),
and Dorfman testing.}
\item{cond.PMF}{a data frame containing the conditional probability
mass function for the number of tests possible for one-stage informative
Sterrett (1SIS), two-stage informative Sterrett (2SIS), and full
informative Sterrett (FIS) testing.}
\item{cond.moments}{a data frame containing the mean and standard deviation
of the conditional moments for one-stage informative
Sterrett (1SIS), two-stage informative Sterrett (2SIS), and full
informative Sterrett (FIS) testing.}
\item{save.diff.CDF}{a data frame containing the sum of the differences
in the cumulative distribution function for each pairwise comparison of
one-stage informative Sterrett (1SIS), two-stage informative Sterrett
(2SIS), full informative Sterrett (FIS), and Dorfman testing.}
\item{p}{a vector containing the probabilities of positivity for each individual.}
}
\description{
Summary measures for Sterrett algorithms.
}
\details{
This function calculates summary measures for informative
Sterrett algorithms. Informative algorithms include one-stage informative
Sterrett (1SIS), two-stage informative Sterrett (2SIS), full informative
Sterrett (FIS), and Dorfman (two-stage hierarchical testing).

The mean and standard deviation of the number of tests,
probability mass function (PMF), and cumulative distribution function (CDF)
are calculated for all informative Sterrett algorithms and
Dorfman testing. Conditional PMFs and conditional moments are calculated
for all informative Sterrett algorithms. Subtracting the mean number of
tests for two procedures gives the area difference between their CDFs.
This area difference is calculated for each pairwise comparison of
1SIS, 2SIS, FIS, and Dorfman testing. CDF plots provide a visualization
of how probabilities are distributed over the number of tests. CDFs that
increase more rapidly to 1 correspond to more efficient retesting
procedures.

Non-informative Sterrett (NIS) decodes positive groups by retesting
individuals at random, so there are \eqn{I!} different possible
NIS implementations. CDFs are found by permuting the elements in the
vector of individual risk probabilities and using the FIS CDF expression
without reordering the individual probabilities. That is, the FIS
procedure uses the most efficient NIS implementation, which is to
retest individuals in order of descending probabilities.
When implementing the informative Sterrett algorithms with a large
number of individuals, an algorithm is used to compute the PMF
for the number of tests under FIS. This is done automatically
by \kbd{Sterrett} for \eqn{I>12}. The algorithm is described in
detail in the Appendix of Bilder et al. (2010).
}
\examples{
# Example 1: FIS provides the smallest mean
#   number of tests and the smallest standard
#   deviation. 2SIS has slightly larger mean
#   and standard deviation than FIS, but
#   its performance is comparable, indicating
#   2SIS may be preferred because it is
#   easier to implement.
set.seed(1231)
p.vec1 <- rbeta(n = 8, shape1 = 1, shape2 = 10)
save.it1 <- Sterrett(p = p.vec1, Sp = 0.90, Se = 0.95)
save.it1

# Example 2: One individual is "high risk" and
#   the others are "low risk". Since there is
#   only one high-risk individual, the three
#   informative Sterrett procedures perform
#   similarly. All three informative Sterrett
#   procedures offer large improvements over
#   Dorfman testing.
p.vec2 <- c(rep(x = 0.01, times = 9), 0.5)
save.it2 <- Sterrett(p = p.vec2, Sp = 0.99, Se = 0.99)
save.it2

# Example 3: Two individuals are at higher
#   risk than the others. All three informative
#   Sterrett procedures provide large
#   improvements over Dorfman testing.
# Due to the large initial group size, an
#   algorithm (described in the Appendix of
#   Bilder et al. (2010)) is used for FIS.
#   The Sterrett() function does this
#   automatically for I>12.
p.vec3 <- c(rep(x = 0.01, times = 98), 0.1, 0.1)
save.it3 <- Sterrett(p = p.vec3, Sp = 0.99, Se = 0.99)
save.it3
}
\references{
\insertRef{Bilder2010a}{binGroup2}
}
\seealso{
\code{\link{expectOrderBeta}} for generating a vector of individual risk
probabilities for informative group testing and \code{\link{opChar1}} for
calculating operating characteristics with hierarchical and array-based
group testing algorithms.

Other operating characteristic functions: 
\code{\link{GroupMembershipMatrix}()},
\code{\link{TOD}()},
\code{\link{halving}()},
\code{\link{operatingCharacteristics1}()},
\code{\link{operatingCharacteristics2}()}
}
\author{
This function was originally written as \kbd{info.gt}
by Christopher Bilder for Bilder et al. (2010). The function was obtained
from \url{http://chrisbilder.com/grouptesting/}. Minor modifications were
made for inclusion of the function in the \code{binGroup2} package.
}
\concept{operating characteristic functions}
