% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cluster_track.R
\name{cluster_track}
\alias{cluster_track}
\title{Cluster tracks based on movement parameters}
\usage{
cluster_track(data, veltrack, variables)
}
\arguments{
\item{data}{A \code{track} R object, which is a list consisting of two elements:
\itemize{
\item \strong{\code{Trajectories}}: A list of interpolated trajectories, where each trajectory is a series of midpoints between consecutive footprints.
\item \strong{\code{Footprints}}: A list of data frames containing footprint coordinates, metadata (e.g., image reference, ID), and a marker indicating whether the footprint is actual or inferred.
}}

\item{veltrack}{A \code{track velocity} R object consisting of a list of lists, where each sublist contains the computed parameters for a corresponding track.}

\item{variables}{A character vector specifying the movement parameters to be used in the clustering analysis. Valid parameter names include: \code{"TurnAng"}, \code{"sdTurnAng"}, \code{"Distance"}, \code{"Length"}, \code{"StLength"}, \code{"sdStLength"}, \code{"Sinuosity"}, \code{"Straightness"}, \code{"Velocity"}, \code{"sdVelocity"}, \code{"MaxVelocity"}, \code{"MinVelocity"}.}
}
\value{
A \code{track clustering} R object consisting of a list containing the following elements:
\itemize{
\item \code{matrix}: A data frame containing the movement parameters calculated for each track.
\item \code{clust}: An \code{Mclust} object containing the results of the model-based clustering analysis. This object provides the optimal (according to BIC) mixture model estimation. The output components are:
\itemize{
\item \code{call}: The matched call.
\item \code{data}: The input data matrix.
\item \code{modelName}: A character string denoting the model at which the optimal BIC occurs.
\item \code{n}: The number of observations in the data.
\item \code{d}: The dimension of the data.
\item \code{G}: The optimal number of mixture components.
\item \code{BIC}: All BIC values.
\item \code{loglik}: The log-likelihood corresponding to the optimal BIC.
\item \code{df}: The number of estimated parameters.
\item \code{bic}: BIC value of the selected model.
\item \code{icl}: ICL value of the selected model.
\item \code{hypvol}: The hypervolume parameter for the noise component if required, otherwise set to NULL.
\item \code{parameters}: A list with the following components:
\itemize{
\item \code{pro}: A vector whose k\eqn{^{th}} component is the mixing proportion for the k\eqn{^{th}} component of the mixture model. If missing, equal proportions are assumed.
\item \code{mean}: The mean for each component. If there is more than one component, this is a matrix whose k\eqn{^{th}} column is the mean of the k\eqn{^{th}} component of the mixture model.
}
\item \code{variance}: A list of variance parameters for the model. The components of this list depend on the model specification. See the help file for mclustVariance for details.
\item \code{z}: A matrix whose i,k\eqn{^{th}} entry is the probability that observation i in the test data belongs to the k\eqn{^{th}} class.
\item \code{classification}: The classification corresponding to z, i.e., map(z).
\item \code{uncertainty}: The uncertainty associated with the classification.
}
}
}
\description{
\code{cluster_track()} clusters trajectories based on various movement and velocity parameters calculated for each track.
}
\details{
The \code{cluster_track()} function performs a model-based clustering analysis on track parameters using the \code{Mclust()} function from the \pkg{mclust} package.

The function first filters out tracks with fewer than four steps, as these tracks may not provide reliable movement data.
It then calculates various movement parameters for each remaining track, including turning angles, distances, lengths, sinuosity, straightness, and velocities.
Finally, the selected movement parameters are used as input for clustering the tracks.

If only one parameter is selected, the clustering is performed using equal variance (\code{"E"}) and variable variance (\code{"V"}) Gaussian models.
If more than one parameter is selected, all Gaussian models available in \code{mclust.options("emModelNames")} are considered.

The following movement parameters can be included in the clustering:
\itemize{
\item \code{"TurnAng"}: Turning angles for the track, measured in degrees. This measures how much the direction of movement changes at each step.
\item \code{"sdTurnAng"}: The standard deviation of the turning angles, indicating how variable the turning angles are across the track.
\item \code{"Distance"}: The total distance covered by the track, calculated as the sum of the straight-line distances between consecutive points (in meters).
\item \code{"Length"}: The overall length of the track, a straight-line distance between the starting and ending points (in meters).
\item \code{"StLength"}: Step lengths for each step of the track, representing how far the object moved between two consecutive points (in meters).
\item \code{"sdStLength"}: The standard deviation of the step lengths, showing how consistent the steps are in length.
\item \code{"Sinuosity"}: A measure of the track's winding nature, calculated as the ratio of the actual track length to the straight-line distance (dimensionless).
\item \code{"Straightness"}: The straightness of the track, calculated as the straight-line distance divided by the total path length (dimensionless).
\item \code{"Velocity"}: The average velocity of the track, calculated as the total distance divided by the time elapsed between the first and last footprint (in meters per second).
\item \code{"sdVelocity"}: The standard deviation of the velocity, indicating how much the velocity fluctuates throughout the track.
\item \code{"MaxVelocity"}: The maximum velocity achieved during the track, identifying the fastest point (in meters per second).
\item \code{"MinVelocity"}: The minimum velocity during the track, identifying the slowest point (in meters per second).
}

The \code{cluster_track()} function has biological relevance in identifying groups of tracks with similar movement parameters,
providing insights into ecological and behavioral patterns. By clustering tracks based on characteristics such as sinuosity,
velocity, and turning angles, it allows detecting movement patterns associated with specific behaviors. This can help identify tracks
potentially made by individuals moving together, which is useful for investigating hypotheses on gregarious behavior, predation
strategies, or coordinated movement. Additionally, clustering serves as a preliminary step before similarity tests and
simulations, refining track selection and improving hypothesis testing in movement ecology studies.
}
\section{Logo}{

\if{html}{\figure{Logo.png}{options: width=30\%}}
}

\examples{
# Example 1: Cluster MountTom tracks using TurnAng and Velocity
H_mounttom <- c(
  1.380, 1.404, 1.320, 1.736, 1.364, 1.432, 1.508, 1.768, 1.600,
  1.848, 1.532, 1.532, 0.760, 1.532, 1.688, 1.620, 0.636, 1.784,
  1.676, 1.872, 1.648, 1.760, 1.612
) # Hip heights for MountTom tracks
veltrack_MountTom <- velocity_track(MountTom, H = H_mounttom)
result1 <- cluster_track(MountTom, veltrack_MountTom,
  variables = c("TurnAng", "Velocity")
)
result1$clust$classification

# Example 2: Cluster MountTom tracks using Sinuosity and Step Length
result2 <- cluster_track(MountTom, veltrack_MountTom,
  variables = c("Sinuosity", "StLength")
)
plot(result2$clust)

# Example 3: Cluster MountTom tracks using Maximum and Minimum Velocity
result3 <- cluster_track(MountTom, veltrack_MountTom,
  variables = c("MaxVelocity", "MinVelocity")
)
result3$clust$classification

# Example 4: Cluster MountTom tracks using Straightness
result4 <- cluster_track(MountTom, veltrack_MountTom, variables = "Straightness")
result4$clust$classification

# Example 5: Cluster PaluxyRiver tracks using Distance and Straightness
H_paluxyriver <- c(3.472, 2.200) # Hip heights for PaluxyRiver tracks
Method_paluxyriver <- c("A", "B") # Different methods for different tracks
veltrack_PaluxyRiver <- velocity_track(PaluxyRiver,
  H = H_paluxyriver,
  method = Method_paluxyriver
)
result5 <- cluster_track(PaluxyRiver, veltrack_PaluxyRiver,
  variables = c("Distance", "Straightness")
)
result5$matrix
result5$clust$classification

# Example 6: Cluster PaluxyRiver tracks using Length and SD of Velocity
result6 <- cluster_track(PaluxyRiver, veltrack_PaluxyRiver,
  variables = c("Length", "sdVelocity")
)
plot(result6$clust)

# Example 7: Cluster PaluxyRiver tracks using TurnAng and SD of TurnAng
result7 <- cluster_track(PaluxyRiver, veltrack_PaluxyRiver,
  variables = c("TurnAng", "sdTurnAng")
)
result7$clust$classification

# Example 8: Cluster PaluxyRiver tracks using Sinuosity
result8 <- cluster_track(PaluxyRiver, veltrack_PaluxyRiver,
  variables = c("Sinuosity")
)
result8$clust$classification

}
\references{
Alexander, R. M. (1976). Estimates of speeds of dinosaurs. Nature, 261(5556), 129-130.

Ruiz, J., & Torices, A. (2013). Humans running at stadiums and beaches and the accuracy of speed estimations from fossil trackways. Ichnos, 20(1), 31-35.

Scrucca L., Fop M., Murphy T. B., & Raftery A. E. (2016) mclust 5: clustering, classification and density estimation using Gaussian finite mixture models. The R Journal, 8(1), 289-317.
}
\seealso{
\code{\link{track_param}}, \code{\link{velocity_track}}, \code{\link[mclust]{Mclust}}
}
\author{
Humberto G. Ferrón

humberto.ferron@uv.es

Macroevolution and Functional Morphology Research Group (www.macrofun.es)

Cavanilles Institute of Biodiversity and Evolutionary Biology

Calle Catedrático José Beltrán Martínez, nº 2

46980 Paterna - Valencia - Spain

Phone: +34 (9635) 44477
}
