% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sar.R
\name{simsar}
\alias{simsar}
\title{Simulating Data from Linear-in-Mean Models with Social Interactions}
\usage{
simsar(formula, Glist, theta, cinfo = TRUE, data)
}
\arguments{
\item{formula}{A symbolic description of the model, passed as a class object of type \link[stats]{formula}.
The formula must specify the endogenous variable and control variables, for example:
\code{y ~ x1 + x2 + gx1 + gx2}, where \code{y} is the endogenous vector,
and \code{x1}, \code{x2}, \code{gx1}, and \code{gx2} are the control variables, which may include contextual variables (peer averages).
Peer averages can be computed using the function \code{\link{peer.avg}}.}

\item{Glist}{A list of network adjacency matrices representing multiple subnets. The \code{m}-th element in the list should be an
\code{ns * ns} matrix, where \code{ns} is the number of nodes in the \code{m}-th subnet.}

\item{theta}{A numeric vector defining the true values of the model parameters \eqn{\theta = (\lambda, \Gamma, \sigma)}.
These parameters are used to define the model specification in the details section.}

\item{cinfo}{A Boolean flag indicating whether the information is complete (\code{cinfo = TRUE}) or incomplete (\code{cinfo = FALSE}).
If information is incomplete, the model operates under rational expectations.}

\item{data}{An optional data frame, list, or environment (or an object coercible by \link[base]{as.data.frame} to a data frame)
containing the variables in the model. If not provided, the variables are taken from the environment of the function call.}
}
\value{
A list containing the following elements:
\item{y}{the observed count data.}
\item{Gy}{the average of y among friends.}
}
\description{
\code{simsar} simulates continuous variables under linear-in-mean models with social interactions, following the specifications described
in Lee (2004) and Lee et al. (2010). The model incorporates peer interactions, where the value of an individual’s outcome depends
not only on their own characteristics but also on the average characteristics of their peers in the network.
}
\details{
In the complete information model, the outcome \eqn{y_i} for individual \eqn{i} is defined as:
\deqn{y_i = \lambda \bar{y}_i + \mathbf{z}_i'\Gamma + \epsilon_i,}
where \eqn{\bar{y}_i} represents the average outcome \eqn{y} among individual \eqn{i}'s peers,
\eqn{\mathbf{z}_i} is a vector of control variables, and \eqn{\epsilon_i \sim N(0, \sigma^2)} is the error term.
In the case of incomplete information models with rational expectations, the outcome \eqn{y_i} is defined as:
\deqn{y_i = \lambda E(\bar{y}_i) + \mathbf{z}_i'\Gamma + \epsilon_i,}
where \eqn{E(\bar{y}_i)} is the expected average outcome of \eqn{i}'s peers, as perceived by individual \eqn{i}.
}
\examples{
\donttest{
# Groups' size
set.seed(123)
M      <- 5 # Number of sub-groups
nvec   <- round(runif(M, 100, 1000))
n      <- sum(nvec)

# Parameters
lambda <- 0.4
Gamma  <- c(2, -1.9, 0.8, 1.5, -1.2)
sigma  <- 1.5
theta  <- c(lambda, Gamma, sigma)

# X
X      <- cbind(rnorm(n, 1, 1), rexp(n, 0.4))

# Network
G      <- list()

for (m in 1:M) {
  nm           <- nvec[m]
  Gm           <- matrix(0, nm, nm)
  max_d        <- 30
  for (i in 1:nm) {
    tmp        <- sample((1:nm)[-i], sample(0:max_d, 1))
    Gm[i, tmp] <- 1
  }
  rs           <- rowSums(Gm); rs[rs == 0] <- 1
  Gm           <- Gm/rs
  G[[m]]       <- Gm
}

# data
data   <- data.frame(X, peer.avg(G, cbind(x1 = X[,1], x2 =  X[,2])))
colnames(data) <- c("x1", "x2", "gx1", "gx2")

ytmp    <- simsar(formula = ~ x1 + x2 + gx1 + gx2, Glist = G, 
                  theta = theta, data = data) 
y       <- ytmp$y
}
}
\references{
Lee, L. F. (2004). Asymptotic distributions of quasi-maximum likelihood estimators for spatial autoregressive models. \emph{Econometrica}, 72(6), 1899-1925, \doi{10.1111/j.1468-0262.2004.00558.x}.

Lee, L. F., Liu, X., & Lin, X. (2010). Specification and estimation of social interaction models with network structures. The Econometrics Journal, 13(2), 145-176, \doi{10.1111/j.1368-423X.2010.00310.x}
}
\seealso{
\code{\link{sar}}, \code{\link{simsart}}, \code{\link{simcdnet}}.
}
