/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5TSmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5TSpkg.h"    

#ifdef H5_HAVE_THREADS

#ifdef H5_HAVE_C11_THREADS

herr_t
H5TS_thread_create(H5TS_thread_t *thread, H5TS_thread_start_func_t func, void *udata)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(thrd_create(thread, func, udata) != thrd_success))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_thread_join(H5TS_thread_t thread, H5TS_thread_ret_t *ret_val)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(thrd_join(thread, ret_val) != thrd_success))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_thread_detach(H5TS_thread_t thread)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(thrd_detach(thread) != thrd_success))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

void
H5TS_thread_yield(void)
{
    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    thrd_yield();

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
} 
#else
#ifdef H5_HAVE_WIN_THREADS

herr_t
H5TS_thread_create(H5TS_thread_t *thread, H5TS_thread_start_func_t func, void *udata)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == (*thread = CreateThread(NULL, 0, func, udata, 0, NULL))))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_thread_join(H5TS_thread_t thread, H5TS_thread_ret_t *ret_val)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(WAIT_OBJECT_0 != WaitForSingleObject(thread, INFINITE)))
        HGOTO_DONE(FAIL);
    if (ret_val)
        if (H5_UNLIKELY(0 == GetExitCodeThread(thread, ret_val)))
            HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(0 == CloseHandle(thread)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_thread_detach(H5TS_thread_t thread)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(0 == CloseHandle(thread)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

void
H5TS_thread_yield(void)
{
    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    SwitchToThread();

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
} 
#else

herr_t
H5TS_thread_create(H5TS_thread_t *thread, H5TS_thread_start_func_t func, void *udata)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(pthread_create(thread, NULL, func, udata)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_thread_join(H5TS_thread_t thread, H5TS_thread_ret_t *ret_val)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(pthread_join(thread, ret_val)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_thread_detach(H5TS_thread_t thread)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(pthread_detach(thread)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

void
H5TS_thread_yield(void)
{
    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    sched_yield();

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
} 
#endif
#endif
#endif 
