% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cast_if_not.r
\name{cast_if_not}
\alias{cast_if_not}
\title{Cast objects to a given type}
\usage{
cast_if_not(..., .env = caller_env(), .error_call = caller_env())
}
\arguments{
\item{...}{any number of named R expressions.}

\item{.env}{the environment to use for the evaluation of the casting
expressions and the assignment of the casted objects.}

\item{.error_call}{the call environment to use for error messages
(passed to \link[rlang:abort]{abort}).}
}
\value{
NULL, but objects named in \code{...} will be changed in the
\code{.env} environment specified.
}
\description{
The names of the \code{...} expressions, which should be variables within
the \code{.env} envrionment, are attempted to be casted to the type specified
in the expression:
e.g., \code{name_of_object_to_cast = object_of_type_to_cast_to}. Expressions
are evaluated in the environment specified and objects are assigned back
into that same environment. Lossy casting can be undertaken by wrapping the
expression in a call to \link{lossy}, e.g., \code{x = lossy(integer())}. The
type conversion is from the \href{https://vctrs.r-lib.org/}{vctrs} package
and thus sticks to the \href{https://vctrs.r-lib.org/reference/faq-compatibility-types.html}{vctrs type conversion rules}.
}
\details{
See \link{abort_if_not} for general validation,
\link{recycle_if_not} for recycling, and \link{enforce} and \link{schema}
for non data-masked and data-masked validations, recycling and casting.
}
\examples{
# NB: Some of these examples are expected to produce an error. To
#     prevent them from terminating a run with example() they are
#     piped into a call to try().

x <- 1L
cast_if_not(x = double())
class(x) # numeric

# By default, lossy casting is not allowed:
x <- c(1, 1.5)
cast_if_not(x = integer()) |> try()

# lossy casting can be enabled using `lossy()` call:
cast_if_not(x = lossy(integer()))
class(x) # integer

# Other objects can be used as the type to cast to, e.g.:
x <- 1L
y <- 2.3
cast_if_not(x = y)
class(x) # numeric

# Changed objects are available immediately:
x <- y <- 1L
cast_if_not(x = double(), y = x)
cat(class(x), class(y), sep = ", ") # numeric, numeric

myfunc <- function(x) {
  cast_if_not(x = double())
  class(x)
}
x <- 1L
myfunc(x) # x is cast to double within the function
class(x) # x is still an integer outside the function

# The `.env` argument determines the expression and assignment
# environment:
x <- 1L
e <- new.env()
e$x <- 1L
cast_if_not(x = 1.5, .env = e)
cat(
  "environment 'e'", class(e$x), "local environment", class(x),
  sep = ", "
) # environment 'e', numeric, local environment, integer

# Named objects (lhs) are checked to be in the `.env` environment,
# throwing an error if not found:
x <- 1L
e <- new.env()
cast_if_not(x = 1.5, .env = e) |> try()

# For expressions (rhs), the `.env` argument is preferentially
# chosen, but if not found then the normal R scoping rules
# apply:
x <- 1.5
e <- new.env()
e$z <- 1L
cast_if_not(z = x, .env = e)
class(e$z) # numeric

# The `.error_call` argument can be used to specify where the
# error occurs, by default this is the caller environment:
myfunc <- function(x) cast_if_not(x = character())
myfunc(FALSE) |> try()

# Injection can be used:
y <- 1L
x <- "y"
cast_if_not(!!x := double()) |> try()
class(y) # numeric

y <- 1L
x <- list(y = double())
cast_if_not(!!!x)
class(y) # numeric

# Objects are reverted to their original values if an error
# occur:
x <- y <- 1L
cast_if_not(x = double(), y = character()) |> try()
class(x) # integer
}
