% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/polyreg.R
\name{polyreg}
\alias{polyreg}
\title{Fit coherent regression models of CIFs using polytomous log odds products}
\usage{
polyreg(
  nuisance.model,
  exposure,
  strata = NULL,
  data,
  subset.condition = NULL,
  na.action = na.omit,
  code.event1 = 1,
  code.event2 = 2,
  code.censoring = 0,
  code.exposure.ref = 0,
  effect.measure1 = "RR",
  effect.measure2 = "RR",
  time.point = NULL,
  outcome.type = "competing-risk",
  conf.int = 0.95,
  report.nuisance.parameter = FALSE,
  report.optim.convergence = FALSE,
  report.sandwich.conf = TRUE,
  report.boot.conf = NULL,
  boot.bca = FALSE,
  boot.multiplier = "rademacher",
  boot.replications = 200,
  boot.seed = 46,
  nleqslv.method = "Newton",
  optim.parameter1 = 1e-06,
  optim.parameter2 = 1e-06,
  optim.parameter3 = 100,
  optim.parameter4 = 50,
  optim.parameter5 = 50,
  optim.parameter6 = 50,
  optim.parameter7 = 1e-10,
  optim.parameter8 = 1e-06,
  optim.parameter9 = 1e-06,
  optim.parameter10 = 40,
  optim.parameter11 = 0.025,
  optim.parameter12 = 2,
  optim.parameter13 = 0.5,
  data.initial.values = NULL,
  normalize.covariate = TRUE,
  terminate.time.point = TRUE,
  prob.bound = 1e-07
)
}
\arguments{
\item{nuisance.model}{A \code{formula} describing the outcome and
nuisance covariates, excluding the exposure of interest.
The LHS must be \code{Event(time, status)} or \code{survival::Surv(time, status)}.}

\item{exposure}{A character string giving the name of the categorical exposure
variable in \code{data}.}

\item{strata}{Optional character string with the name of the stratification
variable used to adjust for dependent censoring (default \code{NULL}).}

\item{data}{A data frame containing the outcome, exposure and nuisance
covariates referenced by \code{nuisance.model}.}

\item{subset.condition}{Optional character string giving a logical condition to subset
\code{data} (default \code{NULL}).}

\item{na.action}{A function specifying the action to take on missing values (default \code{na.omit}).}

\item{code.event1}{Integer code of the event of interest (default \code{1}).}

\item{code.event2}{Integer code of the competing event (default \code{2}).}

\item{code.censoring}{Integer code of censoring (default \code{0}).}

\item{code.exposure.ref}{Integer code identifying the reference exposure
category (default \code{0}).}

\item{effect.measure1}{Character string specifying the effect measure for the
primary event. Supported values are \code{"RR"}, \code{"OR"} and
\code{"SHR"}.}

\item{effect.measure2}{Character string specifying the effect measure for the
competing event. Supported values are \code{"RR"}, \code{"OR"} and
\code{"SHR"}.}

\item{time.point}{Numeric time point at which the exposure effect is evaluated for
time-point models. Required for \code{"competing-risk"} and \code{"survival"}
outcomes.}

\item{outcome.type}{Character string selecting the outcome type. Valid values are
\code{"competing-risk"}, \code{"survival"}, \code{"binomial"}, \code{"proportional-survival"},
and \code{"proportional-competing-risk"}. The default is \code{"competing-risk"}.
If explicitly set to \code{NULL}, \code{polyreg()} attempts to infer the outcome type from the data: if the
event variable has more than two distinct levels, \code{"competing-risk"}
is assumed; otherwise, \code{"survival"} is assumed. Abbreviations such as
\code{"S"} or \code{"C"} are accepted; mixed or ambiguous inputs trigger
automatic detection from the event coding in \code{data}.}

\item{conf.int}{Numeric two-sided level of CIs (default \code{0.95}).}

\item{report.nuisance.parameter}{Logical; if \code{TRUE}, the returned object
includes estimates of the nuisance model parameters (default \code{FALSE}).}

\item{report.optim.convergence}{Logical; if \code{TRUE}, optimization
convergence summaries are returned (default \code{FALSE}).}

\item{report.sandwich.conf}{Logical or \code{NULL}. When \code{TRUE},
CIs based on sandwich variance are computed.
When \code{FALSE}, they are omitted (default \code{TRUE}).
This CI is default for time-point models
(\code{"outcome.type=competing-risk"}, \code{"survival"} or \code{"binomial"}) and
is not available otherwise.}

\item{report.boot.conf}{Logical or \code{NULL}. When \code{TRUE}, bootstrap
CIs are computed. When \code{FALSE}, they are omitted.
If \code{NULL}, the function chooses based on \code{outcome.type} (default \code{NULL}).
This CI is default for proportional models
(\code{outcome.type="proportional-competing-risk"} or \code{"proportional-survival"}).}

\item{boot.bca}{Logical indicating the bootstrap CI method.
Use \code{TRUE} for bias-corrected and accelerated intervals or \code{FALSE}
for the normal approximation (default \code{FALSE}).}

\item{boot.multiplier}{Character string specifying the wild bootstrap weight distribution.
One of \code{"rademacher"}, \code{"mammen"} or \code{"gaussian"} (default \code{"rademacher"}).}

\item{boot.replications}{Integer giving the number of bootstrap replications
(default \code{200}).}

\item{boot.seed}{Numeric seed used for resampling of bootstrap.}

\item{nleqslv.method}{Character string specifying the solver used in
\pkg{nleqslv()}. Available choices are \code{"Broyden"} and \code{"Newton"}.}

\item{optim.parameter1}{Numeric tolerance for convergence of the outer loop
(default \code{1e-6}).}

\item{optim.parameter2}{Numeric tolerance for convergence of the inner loop
(default \code{1e-6}).}

\item{optim.parameter3}{Numeric constraint on the absolute value of
parameters (default \code{100}).}

\item{optim.parameter4}{Integer maximum number of outer loop iterations
(default \code{50}).}

\item{optim.parameter5}{Integer maximum number of \code{nleqslv}
iterations per outer iteration (default \code{50}).}

\item{optim.parameter6}{Integer maximum number of iterations for the
Levenberg-Marquardt routine (default \code{50}).}

\item{optim.parameter7}{Numeric convergence tolerance for the
Levenberg-Marquardt routine (default \code{1e-10}).}

\item{optim.parameter8}{Numeric tolerance for updating the Hessian in the
Levenberg-Marquardt routine (default \code{1e-6}).}

\item{optim.parameter9}{Numeric starting value for the Levenberg-Marquardt
damping parameter lambda (default \code{1e-6}).}

\item{optim.parameter10}{Numeric upper bound for lambda in the
Levenberg-Marquardt routine (default \code{40}).}

\item{optim.parameter11}{Numeric lower bound for lambda in the
Levenberg-Marquardt routine (default \code{0.025}).}

\item{optim.parameter12}{Numeric multiplicative increment applied to lambda
when the Levenberg-Marquardt step is successful (default \code{2}).}

\item{optim.parameter13}{Numeric multiplicative decrement applied to lambda
when the Levenberg-Marquardt step is unsuccessful (default \code{0.5}).}

\item{data.initial.values}{Optional data frame providing starting values for
the optimization (default \code{NULL}).}

\item{normalize.covariate}{Logical indicating whether covariates should
be centered and scaled prior to optimization (default \code{TRUE}).}

\item{terminate.time.point}{Logical indicating whether time points
that contribute estimation are terminated by min of max follow-up times
of each exposure level (default \code{TRUE}).}

\item{prob.bound}{Numeric lower bound used to internally truncate probabilities away
from 0 and 1 (default \code{1e-5}).}
}
\value{
A list of class \code{"polyreg"} containing the fitted exposure effects and
supporting results. Key components and methods include:
\itemize{
\item \code{coef}: regression coefficients on the chosen effect-measure scale
\item \code{vcov}: variance–covariance matrix of the regression coefficients
\item \code{diagnostic.statistics}: a data frame with inverse probability weights,
influence function contributions, and predicted potential outcomes
\item \code{summary}: event-wise tidy/glance summaries used by
\code{summary.polyreg()} or \code{modelsummary::modelsummary()}
\item additional elements storing convergence information and internal
tuning parameters.
}

Standard S3 methods are available: \code{coef.polyreg()}, \code{vcov.polyreg()},
\code{nobs.polyreg()}, and \code{summary.polyreg()}.
}
\description{
\code{polyreg()} fits regression models of CIFs, targeting familiar effect measures
(risk ratios, odds ratios and subdistribution hazard ratios).
Modeling the nuisance structure using polytomous log odds products ensures that
the sum of cause-specific CIFs does not exceed one, and enables coherent modelling
of the multiplicative effects.

This function follows a familiar formula–data workflow: the outcome and
covariates other than the exposure are specified through a formula in \code{nuisance.model}
(with \code{Event()} or \code{survival::Surv()} on the LHS), and the exposure of interest
is given by a separate variable name in \code{exposure}. The fitted object contains
tidy summaries of exposure effects (point estimates, SEs, CIs, and p-values)
and can be summarised with \code{summary.polyreg()} or formatted with external tools
such as \code{modelsummary::modelsummary()}.
}
\details{
\subsection{Overview}{

\code{polyreg()} implements \strong{log odds product modeling} for CIFs at user-specified
time points, focusing on multiplicative effects of a categorical exposure, or
constant effects over time like Cox regression and Fine-Gray models. It estimates
multiplicative effects such as \strong{risk ratios}, \strong{odds ratios}, or
\strong{subdistribution hazard ratios}, while ensuring that the probabilities across
competing events sum to one. This is achieved through
\strong{reparameterization using polytomous log odds products}, which fits so-called
effect-measure models and nuisance models on multiple competing events
simultaneously. Additionally, \code{polyreg()} supports direct binomial regression
for survival outcomes and the Richardson model for binomial outcomes,
both of which use log odds products.
}

\subsection{Key arguments}{
\itemize{
\item \code{nuisance.model}: a formula with \code{Event()} or \code{survivai::Surv()}
describing the outcome and nuisance covariates, excluding the exposure of interest.
\item \code{exposure}: name of the categorical exposure variable
\item \code{effect.measure1} and \code{effect.measure2}: the effect measures
for event1 and event2 (\code{"RR"}, \code{"OR"} or \code{"SHR"}).
\item \code{outcome.type}: type of the outcome variable (\code{"competing-risk"}, \code{"survival"},
\code{"binomial"}, \code{"proportional-survival"} or \code{"proportional-competing-risk"}).
\item \code{time.point}: time point(s) at which the exposure effect is evaluated.
Required for \code{"competing-risk"} and \code{"survival"} outcomes.
\item \code{strata}: name of the stratification variable used for IPCW adjustment for dependent censoring.
}
}

\subsection{Outcome type and event status coding}{

The \code{outcome.type} argument must be set to:
\itemize{
\item Effects on cumulative incidence probabilities at a specific time:
\code{"competing-risk"}.
\item Effects on a risk at a specific time: \code{"survival"}.
\item Common effects on cumulative incidence probabilities over time:
\code{"proportional-competing-risk"}.
\item Common effects on a risk over time: \code{"proportional-survival"}.
\item Effects on a risk of a binomial outcome: \code{"binomial"}.
}\tabular{lll}{
   Setting \tab Codes \tab Meaning \cr
   competing-risk \tab \code{code.event1}, \code{code.event2}, \code{code.censoring} \tab event of interest / competing event / censoring \cr
   competing-risk (default) \tab \code{code.event1 = 1}, \code{code.event2 = 2}, \code{code.censoring = 0} \tab event of interest / competing event / censoring \cr
   survival \tab \code{code.event1}, \code{code.censoring} \tab event / censoring \cr
   survival (default) \tab \code{code.event1 = 1}, \code{code.censoring = 0} \tab event / censoring \cr
   survival (ADaM-ADTTE) \tab \code{code.event1 = 0}, \code{code.censoring = 1} \tab set to match ADaM convention \cr
   proportional-survival \tab \code{code.event1}, \code{code.censoring} \tab event / censoring \cr
   proportional-survival (default) \tab \code{code.event1 = 1}, \code{code.censoring = 0} \tab event / censoring \cr
   proportional-survival (ADaM) \tab \code{code.event1 = 0}, \code{code.censoring = 1} \tab set to match ADaM convention \cr
   proportional-competing-risk \tab \code{code.event1}, \code{code.event2}, \code{code.censoring} \tab event of interest / competing event / censoring \cr
   proportional-competing-risk (default) \tab \code{code.event1 = 1}, \code{code.event2 = 2}, \code{code.censoring = 0} \tab event of interest / competing event / censoring \cr
}

}

\subsection{Effect measures for categorical exposure}{

Choose the effect scale for event 1 and (optionally) event 2:\tabular{llll}{
   Argument \tab Applies to \tab Choices \tab Default \cr
   \code{effect.measure1} \tab event of interest \tab \code{"RR"}, \code{"OR"}, \code{"SHR"} \tab \code{"RR"} \cr
   \code{effect.measure2} \tab competing event \tab \code{"RR"}, \code{"OR"}, \code{"SHR"} \tab \code{"RR"} \cr
}

\itemize{
\item \code{RR}: risk ratio at \code{time.point} or common over time.
\item \code{OR}: odds ratio at \code{time.point} or common over time.
\item \code{SHR}: subdistribution hazard ratio or common over time.
}
}

\subsection{Inference and intervals (advanced)}{\tabular{lll}{
   Argument \tab Meaning \tab Default \cr
   \code{conf.int} \tab Wald-type CI level \tab \code{0.95} \cr
   \code{report.sandwich.conf} \tab Sandwich variance CIs \tab \code{TRUE} \cr
   \code{report.boot.conf} \tab Bootstrap CIs (used by \code{"proportional-*"} types) \tab \code{NULL} \cr
   \code{boot.bca} \tab Use BCa intervals (else normal approximation) \tab \code{FALSE} \cr
   \code{boot.multiplier} \tab Method for wild bootstrap \tab \code{"rademacher"} \cr
   \code{boot.replications} \tab Bootstrap replications \tab \code{200} \cr
   \code{boot.seed} \tab Seed for resampling \tab \code{46} \cr
}

}

\subsection{Optimization & solver controls (advanced)}{

\code{polyreg()} solves estimating equations with optional inner routines.\tabular{lll}{
   Argument \tab Role \tab Default \cr
   \code{nleqslv.method} \tab Root solver \tab \code{"Newton"} \cr
   \code{optim.parameter1}, \code{optim.parameter2} \tab Outer / inner convergence tolerances \tab \code{1e-6}, \code{1e-6} \cr
   \code{optim.parameter3} \tab Parameter absolute bound \tab \code{100} \cr
   \code{optim.parameter4} \tab Max outer iterations \tab \code{50} \cr
   \code{optim.parameter5} \tab Max \code{nleqslv} iterations per outer \tab \code{50} \cr
   \code{optim.parameter6:13} \tab Levenberg–Marquardt controls (iterations, tolerances, lambda) \tab see defaults \cr
}

}

\subsection{Data handling and stability}{\tabular{lll}{
   Argument \tab Meaning \tab Default \cr
   \code{subset.condition} \tab Expression (as character) to subset \code{data} \tab \code{NULL} \cr
   \code{na.action} \tab NA handling function \tab \code{stats::na.omit} \cr
   \code{normalize.covariate} \tab Center/scale nuisance covariates \tab \code{TRUE} \cr
   \code{terminate.time.point} \tab Truncate support by exposure-wise follow-up maxima \tab \code{TRUE} \cr
   \code{prob.bound} \tab Truncate probabilities away from 0/1 (numerical guard) \tab \code{1e-5} \cr
   \code{data.initial.values} \tab Optional starting values data frame \tab \code{NULL} \cr
}

}

\subsection{Downstream use}{

\code{polyreg()} returns an object of class \code{"polyreg"} that contains
regression coefficients (\code{coef}), variance-covariance matrix (\code{vcov})
and a list of event-wise \emph{tidy} and \emph{glance} tables (\code{summary}).
Users should typically access results via the S3 methods:
\itemize{
\item \code{coef()} — extract regression coefficients.
\item \code{vcov()} — extract the variance–covariance matrix
(sandwich or bootstrap, depending on \code{outcome.type} and the
\verb{report.*} arguments).
\item \code{nobs()} — number of observations used in the fit.
\item \code{summary()} — print an event-wise, modelsummary-like table of estimates,
CIs and p-values, and return the underlying list of tidy/glance tables invisibly.
}

For backward compatibility, components named \code{coefficient} and \code{cov}
may also be present and mirror \code{coef} and \code{vcov}, respectively.
The \code{summary} component can be passed to external functions such as
\code{modelsummary()} for further formatting, if desired.
}

\subsection{Reproducibility and conventions}{
\itemize{
\item If convergence warnings appear, relax/tighten tolerances or cap the parameter
bound (\code{optim.parameter1}–\code{3}) and inspect the output with
\code{report.optim.convergence = TRUE}.
\item If necessary, modify other \code{optim.parameter}, provide user-specified
initial values, or reduce the number of nuisance parameters (e.g., provide
a small set of time points contributing to estimation when using
\code{"proportional-survival"} or \code{"proportional-competing-risk"}).
\item Set \code{boot.seed} for reproducible bootstrap results.
\item Match CDISC ADaM conventions via \code{code.event1 = 0}, \code{code.censoring = 1}
(and, if applicable, \code{code.event2} for competing events).
}
}
}
\section{Lifecycle}{

\lifecycle{experimental}
}

\examples{
data(diabetes.complications)
output <- polyreg(
  nuisance.model = Event(t, epsilon) ~ +1,
  exposure = "fruitq1",
  data = diabetes.complications,
  effect.measure1 = "RR",
  effect.measure2 = "RR",
  time.point = 8,
  outcome.type = "competing-risk"
)

coef(output)
vcov(output)
nobs(output)
summary(output)

}
\seealso{
\code{\link[=cifcurve]{cifcurve()}} for KM/AJ estimators; \code{\link[=cifplot]{cifplot()}} for display of a CIF; \code{\link[=cifpanel]{cifpanel()}} for display of multiple CIFs; \link[ggsurvfit:ggsurvfit]{ggsurvfit::ggsurvfit}, \link[patchwork:patchwork-package]{patchwork::patchwork} and \link[modelsummary:modelsummary]{modelsummary::modelsummary} for display helpers.
}
