#' Load base-rate database, model typicality matrices, or human validation ratings
#'
#' This function gives access to key datasets included in the `baserater` package.
#'
#' @details
#' - The `"database"` object includes all base-rate items along with stereotype strength estimates from 'GPT-4' and 'LLaMA 3.3'.
#'
#' - The `"validation_ratings"` object contains average typicality judgments from 50 human participants on 100 group–adjective pairs, as well as ratings from 'GPT-4' and 'LLaMA 3.3'.
#'
#' - The `"typicality_matrix_gpt4"` and `"typicality_matrix_llama3.3"` objects are raw typicality matrices generated by each model.
#'
#' - The `"material"` object contains the lists of individual groups and adjectives used to build the base-rate database.
#'
#' @param which One of "database", "validation_ratings", "typicality_matrix_gpt4", "typicality_matrix_llama3.3", or "material".
#' @param dest  Optional path to copy the file to (returns the data either way).
#' @return A tibble with the requested data.
#' @export
#' @examples
#'
#' database <- download_data("database")
#'
#' ratings <- download_data("validation_ratings")
#'
#' gpt4_matrix <- download_data("typicality_matrix_gpt4")
#'
#' llama3_matrix <- download_data("typicality_matrix_llama3.3")
#'
#' material <- download_data("material")

download_data <- function(which = c("database", "validation_ratings", "typicality_matrix_gpt4", "typicality_matrix_llama3.3", "material"),
                          dest = NULL) {

  which <- match.arg(which)

  filename <- switch(
    which,
    database                    = "base_rate_database.rds",
    validation_ratings          = "typicality_validation_df.csv",
    typicality_matrix_gpt4      = "typicality_matrix_gpt4.csv",
    typicality_matrix_llama3.3  = "typicality_matrix_llama3.3.csv",
    material                    = "adjective_description_material.csv"
  )

  path <- system.file("extdata", filename, package = "baserater")

  if (path == "") {
    stop("Internal data file not found: ", filename)
  }

  if (!is.null(dest)) {
    file.copy(path, dest, overwrite = TRUE)
  }

  if (grepl("\\.rds$", filename)) {
    readRDS(path)
  } else {
    readr::read_csv(path, show_col_types = FALSE)
  }
}
