% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.cv.R
\name{predict_cv}
\alias{predict_cv}
\alias{predict.svem_cv}
\title{Predict from glmnet_with_cv Fits (svem_cv Objects)}
\usage{
predict_cv(object, newdata, debias = FALSE, strict = FALSE, ...)

\method{predict}{svem_cv}(object, newdata, debias = FALSE, strict = FALSE, ...)
}
\arguments{
\item{object}{A fitted object returned by \code{glmnet_with_cv()}
(class \code{"svem_cv"}).}

\item{newdata}{A data frame of new predictor values.}

\item{debias}{Logical; if \code{TRUE} and a debiasing fit is available,
apply it. Has an effect only for Gaussian models where
\code{debias_fit} is present.}

\item{strict}{Logical; if \code{TRUE}, enforce a strict column-name match
between the aligned design for \code{newdata} and the training design
(including the intercept position). Default \code{FALSE}.}

\item{...}{Additional arguments (currently unused).}
}
\value{
A numeric vector of predictions on the response scale:
numeric fitted values for Gaussian models; probabilities in \code{[0,1]}
for binomial models. Rows with unseen factor/character levels return
\code{NA}.
}
\description{
Generate predictions from a fitted object returned by
\code{glmnet_with_cv()} (class \code{"svem_cv"}).
}
\details{
The design matrix for \code{newdata} is rebuilt using the stored training
\code{terms} (with environment set to \code{baseenv()}), together with the
saved factor \code{xlevels} and \code{contrasts} (cached in
\code{object$schema}). Columns are then aligned back to the training
design in a robust way:
\itemize{
\item Any training columns that \code{model.matrix()} drops for
\code{newdata} (for example, a factor collapsing to a single level)
are added back as zero columns.
\item Columns are reordered to exactly match the training order.
\item Rows containing unseen factor/character levels are warned about and
their predictions are set to \code{NA}.
}

For Gaussian fits (\code{family = "gaussian"}), the returned values are on
the original response (identity-link) scale. For binomial fits
(\code{family = "binomial"}), the returned values are probabilities in
\code{[0,1]} (logit-link inverted via \code{plogis()}).

If \code{debias = TRUE} and a calibration model \code{lm(y ~ y_pred)} is
present with a finite slope, predictions are adjusted via
\code{a + b * pred}. Debiasing is only fitted and used for Gaussian models;
for binomial models the \code{debias} argument has no effect.

\code{predict_cv()} is a small convenience wrapper that simply calls the
underlying S3 method \code{predict.svem_cv()}, keeping a single code path
for prediction from \code{glmnet_with_cv()} objects.
}
\examples{
set.seed(1)
n <- 50; p <- 5
X <- matrix(rnorm(n * p), n, p)
y <- X[, 1] - 0.5 * X[, 2] + rnorm(n)
df_ex <- data.frame(y = as.numeric(y), X)
colnames(df_ex) <- c("y", paste0("x", 1:p))

fit <- glmnet_with_cv(
  y ~ ., df_ex,
  glmnet_alpha = 1,
  nfolds = 5,
  repeats = 2,
  seed = 9,
  family = "gaussian"
)

preds_raw <- predict_cv(fit, df_ex)
preds_db  <- predict_cv(fit, df_ex, debias = TRUE)
cor(preds_raw, df_ex$y)

# Binomial example (probability predictions on [0,1] scale)
set.seed(2)
n2 <- 80; p2 <- 4
X2 <- matrix(rnorm(n2 * p2), n2, p2)
eta2 <- X2[, 1] - 0.8 * X2[, 2]
pr2 <- plogis(eta2)
y2 <- rbinom(n2, size = 1, prob = pr2)
df_bin <- data.frame(y = y2, X2)
colnames(df_bin) <- c("y", paste0("x", 1:p2))

fit_bin <- glmnet_with_cv(
  y ~ ., df_bin,
  glmnet_alpha = c(0.5, 1),
  nfolds = 5,
  repeats = 2,
  seed = 11,
  family = "binomial"
)

prob_hat <- predict_cv(fit_bin, df_bin)
summary(prob_hat)

}
