#' @title Weighted Mean Membership Function Value for Stress Tolerance (WMFVST)
#' @description
#' Its a composite measure which computes the weighted mean of MFVST values (0–1 range) for each genotype
#' across multiple stress indices, considering whether higher or lower values are better.
#'
#' @param data A data frame containing genotype IDs (Gen) and MFVST values for the indices
#' (GMP, HM, MP, MRP, PYR, RSI, SSPI, STI, TOL, YI, YSI), each within the range of 0 to 1.
#' @return A data frame with Gen and its Weighted Mean MFVST (WMFVST).
#' @details
#' The Weighted Mean MFVST for genotype *i* is:
#' \deqn{WMFVST_i = \frac{\sum_{j=1}^{k} (MFVST_{ij} \times R_{ij})}
#'                      {\sum_{j=1}^{k} R_{ij}}}
#' where \eqn{MFVST_{ij}} is the MFVST value of genotype *i* for index *j*, and
#' \eqn{R_{ij}} is its rank, determined by whether higher or lower values are favorable.
#'
#' @examples
#' set.seed(123)
#' df <- data.frame(
#'   Gen = paste0("G", 1:5),
#'   GMP  = runif(5),
#'   HM   = runif(5),
#'   MP   = runif(5),
#'   MRP  = runif(5),
#'   PYR  = runif(5),
#'   RSI  = runif(5),
#'   SSPI = runif(5),
#'   STI  = runif(5),
#'   TOL  = runif(5),
#'   YI   = runif(5),
#'   YSI  = runif(5)
#' )
#' WMFVST(df)
#' @export
WMFVST <- function(data) {
  # Here input data comprises MFVST values for all 11 indices within the range of 0-1.
  # Define groups based on direction
  lower_better  <- c("TOL","SSPI","RSI","PYR")
  higher_better <- c("STI","YI","YSI","MP","GMP","HM","MRP")
  indices <- c(lower_better, higher_better)

  #  Structural checks
  if (!("Gen" %in% names(data))) {
    stop("Input data must contain a 'Gen' column.")
  }
  missing_cols <- setdiff(indices, names(data))
  if (length(missing_cols)) {
    stop("Missing required columns: ", paste(missing_cols, collapse = ", "))
  }

  #  Type & range checks
  non_num <- vapply(data[indices], function(x) !is.numeric(x), logical(1))
  if (any(non_num)) {
    stop("Non-numeric MFVST columns: ", paste(names(non_num)[non_num], collapse = ", "))
  }

  has_na <- vapply(data[indices], function(x) any(is.na(x)), logical(1))
  if (any(has_na)) {
    stop("NA values detected in: ", paste(names(has_na)[has_na], collapse = ", "),
         ". Please impute or remove before WMFVST.")
  }

  out_of_range <- vapply(data[indices], function(x) any(x < 0 | x > 1), logical(1))
  if (any(out_of_range)) {
    bad_cols <- names(out_of_range)[out_of_range]
    stop("Values outside [0,1] detected in: ", paste(bad_cols, collapse = ", "),
         ". Only compute WMFVST after fixing/scaling MFVST values.")
  }

  #  Direction-aware ranks (ties.method='min')
  rank_mat <- sapply(indices, function(ind) {
    if (ind %in% higher_better) {
      rank(-data[[ind]], ties.method = "min")  # higher = better
    } else {
      rank( data[[ind]], ties.method = "min")  # lower = better
    }
  })

  #  WMFVST computation
  mfvst_mat <- data.matrix(data[indices])  # enforce numeric matrix
  weighted_sum <- rowSums(mfvst_mat * rank_mat)
  rank_sum     <- rowSums(rank_mat)
  WMFVST_val    <- round(weighted_sum / rank_sum, 5)

  #  Output
  result <- data.frame(Gen = data$Gen, WMFVST = WMFVST_val)
  result <- result[order(result$WMFVST, decreasing = TRUE), , drop = FALSE]
  rownames(result) <- NULL
  return(result)
}
