\name{findmleHMMnostarting}
\alias{findmleHMMnostarting}
\title{Multiple Initialization Maximum Likelihood Estimation for Hidden Markov Models}
\description{
Fits a Hidden Markov Model (HMM) by repeatedly initializing observation and transition parameters and selecting the fit with the highest log-likelihood. This approach helps avoid convergence to poor local optima. For the generalized extreme value (GEV) distribution, starting values are generated from repeated maximum likelihood fits on random data subsets.
}
\usage{
findmleHMMnostarting(J, x, obsdist, no.initials = 50, EM = FALSE,
                     verbose = TRUE, seed = NULL, ...)
}
\arguments{
\item{J}{Integer. The number of hidden states in the HMM. Must be strictly greater than 1.}
\item{x}{Numeric vector. The observed data sequence.}
\item{obsdist}{Character string. The observation distribution. Supported distributions are:
"norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}
\item{no.initials}{Integer. The number of random initializations to attempt. Defaults to 50.}
\item{EM}{Logical. If \code{TRUE}, uses an EM-based semi-Markov approximation for estimation.
If \code{FALSE}, maximizes the likelihood directly using \code{nlm}. Defaults to \code{FALSE}.}
\item{verbose}{Logical. If \code{TRUE}, progress messages are printed to the console. Default is \code{TRUE}.}
\item{seed}{Integer or \code{NULL}. Random seed for reproducibility. Default is \code{NULL}.}
\item{...}{Further arguments to be passed to \code{\link{findmleHMM}} in the case of \code{EM=TRUE}.}
}
\value{
A list corresponding to the best fit across all initializations, containing:
\item{estimate}{List of estimated HMM parameters, including state-dependent observation parameters and transition probabilities.}
\item{loglik}{The maximized log-likelihood value.}
\item{AIC}{The Akaike Information Criterion for the fitted model.}
\item{BIC}{Bayesian Information Criteria for the fitted model.}
\item{hessian}{Optional. The Hessian matrix at the maximum likelihood estimates (returned if \code{EM = FALSE}).}
}
\details{
This function automates multiple trials of \code{\link{findmleHMM}} with randomized starting values, returning the fit that achieves the highest log-likelihood.
\itemize{
  \item For most observation distributions, starting values are generated via \code{clusterHMM}.
  \item For the GEV distribution, starting values are drawn from repeated fits of \code{evd::fgev} on random data segments. Up to 20,000 attempts are made, and a warning is issued if fewer than 1000 valid estimates are obtained.
}
During each iteration:
\enumerate{
  \item Observation parameters are perturbed slightly to encourage exploration.
  \item A transition matrix \code{Pi} is drawn from a random uniform distribution with added self-transition bias.
  \item The HMM is estimated via \code{\link{findmleHMM}}.
  \item If the resulting log-likelihood exceeds the current best, the model is updated.
}
At the end of all iterations, the best-fitting model is returned.
When \code{verbose = TRUE}, iteration numbers and error messages are displayed during the fitting process.
}
\examples{
set.seed(123)
J <- 3
Pi <- matrix(c(0.7, 0.2, 0.1,
               0.1, 0.8, 0.1,
               0.2, 0.3, 0.5), nrow = 3, byrow = TRUE)
obspar <- list(mean = c(-2, 0, 3),
               sd   = c(0.5, 1, 1.5))
x <- generateHMM(n = 200, J = J, Pi = Pi, obsdist = "norm", obspar = obspar)$x

\donttest{
fit <- findmleHMMnostarting(J = J, x = x, obsdist = "norm",
                            no.initials = 30)

fit$loglik
fit$estimate

fit_silent <- findmleHMMnostarting(J = J, x = x, obsdist = "norm",
                                   no.initials = 30, verbose = FALSE)
}
}
\seealso{
\code{\link{findmleHMM}} for fitting an HMM with user-supplied starting values.
\code{\link{generateHMM}} for simulating HMM data.
\code{\link{findmleHSMMnostarting}} for the non-initialised estimation of hidden semi-Markov models.
}
\author{
Aimee Cody
}

