\name{exceedanceplotHSMMgev}
\alias{exceedanceplotHSMMgev}
\title{Plot Exceedance Probabilities from GEV-HSMM}
\description{
Computes and plots exceedance probabilities of a given threshold from a fitted Hidden Semi-Markov Model (HSMM) with generalized extreme value (GEV) state-dependent distributions.
Bootstrap confidence intervals are included. Probabilities are aggregated over user-defined time periods.
}
\usage{
exceedanceplotHSMMgev(x, HSMM, threshold, dwelldist, M = NA,
                      varcov = NULL, B = 10000, level = 0.95,
                      time_structure = NULL, shift = FALSE,
                      plot_title = "Exceedance Probabilities Over Time",
                      save_plot = FALSE, filename = NULL,
                      width = 12, height = 8, dpi = 300,
                      verbose = TRUE)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}

\item{HSMM}{A fitted HSMM object (output from \code{\link{findmleHSMM}}),
with state-dependent GEV parameters \code{loc}, \code{scale}, \code{shape}, dwell parameters, and estimated transition probabilities.}

\item{threshold}{Numeric. The exceedance threshold. Probabilities of exceeding this value are computed.}

\item{dwelldist}{Character string. The dwell-time distribution used in the HSMM, e.g. \code{"pois"}, \code{"nbinom"}, or \code{"betabinom"}.}

\item{M}{Optional integer. Truncation parameter for dwell-time distribution. Default is \code{NA} (no truncation).}

\item{varcov}{Optional variance–covariance matrix of parameter estimates.
If \code{NULL}, computed internally via \code{\link{HSMMVarianceMatrix}} with \code{obsdist = "gev"}.}

\item{B}{Integer. Number of bootstrap replicates used for confidence intervals. Default is 10000.}

\item{level}{Numeric between 0 and 1. Confidence level for intervals. Default is 0.95.}

\item{time_structure}{Optional list describing the time scale.
Includes \code{unit}, \code{observations_per_unit}, and optionally \code{start_point}, \code{end_point}.
Supported units: "year", "day", "hour", "week", "month", or \code{"custom"} with \code{conversion_factor} and \code{unit_name}.}

\item{shift}{Logical. If \code{TRUE}, uses shifted dwell-time distributions. Default is \code{FALSE}.}

\item{plot_title}{Character string. Title for the plot. Default is "Exceedance Probabilities Over Time".}

\item{save_plot}{Logical. If \code{TRUE}, the plot is saved to file. Default is \code{FALSE}.}

\item{filename}{Character string or \code{NULL}. Filename for saved plot (if \code{save_plot = TRUE}).
Must be specified when \code{save_plot = TRUE}. Default is \code{NULL}.}

\item{width}{Numeric. Width of saved plot in inches. Default is 12.}

\item{height}{Numeric. Height of saved plot in inches. Default is 8.}

\item{dpi}{Integer. Resolution of saved plot in dots per inch. Default is 300.}

\item{verbose}{Logical. If \code{TRUE}, progress messages are printed to the console. Default is \code{TRUE}.}
}
\value{
Invisibly returns a list containing:
\item{exceedance_probs}{Matrix of exceedance probabilities over time (3 rows: estimate, lower CI, upper CI).}
\item{time_info}{List containing time axis information and labels.}
\item{decoded_states}{Vector of decoded hidden states for each time point.}
\item{time_unit}{Number of observations per aggregation period.}
\item{threshold}{The threshold value used.}

A time-series plot is also produced, showing aggregated exceedance probabilities with confidence intervals.
}
\details{
For each state, the exceedance probability of \code{threshold} is computed from the fitted GEV parameters.
Uncertainty is estimated by parametric bootstrap using draws from a multivariate normal approximation of parameter estimates.
Observation-level probabilities are aggregated over time periods defined by \code{time_structure} (e.g., years, months, weeks).
When \code{verbose = TRUE}, progress messages are displayed during computation.
}
\examples{
set.seed(123)

J <- 2
Pi <- matrix(c(0, 1,
               1, 0), nrow = J, byrow = TRUE)
delta <- c(0.5, 0.5)

obspar <- list(
  loc   = c(0, 5),
  scale = c(1, 2),
  shape = c(0.1, -0.1)
)

dwellpar <- list(lambda = c(5, 10))

sim <- generateHSMM(
  n = 200, J = J, obsdist = "gev", dwelldist = "pois",
  obspar = obspar, dwellpar = dwellpar, Pi = Pi, delta = delta
)

HSMM_fit <- findmleHSMM(
  x = sim$x, J = J, obsdist = "gev", dwelldist = "pois",
  obspar = obspar, dwellpar = dwellpar, Pi = Pi, delta = delta
)

time_struct <- list(
  unit = "week", observations_per_unit = 10, start_point = 1
)

\donttest{
result <- exceedanceplotHSMMgev(
  x = sim$x, HSMM = HSMM_fit,
  threshold = 20, dwelldist = "pois",
  B = 10000, time_structure = time_struct,
  plot_title = "GEV-HSMM Exceedance Probabilities"
)

result <- exceedanceplotHSMMgev(
  x = sim$x, HSMM = HSMM_fit,
  threshold = 20, dwelldist = "pois",
  B = 10000, time_structure = time_struct,
  save_plot = TRUE,
  filename = tempfile(fileext = ".png"),
  verbose = FALSE
)
}
}
\seealso{
\code{\link{findmleHSMM}}, for fitting HSMMs.
\code{\link{generateHSMM}}, for simulating HSMM data.
\code{\link{globaldecodeHSMM}}, for decoding the most likely state sequence.
\code{\link{HSMMVarianceMatrix}}, for variance–covariance estimation.
}
\author{Aimee Cody}


