#' @title Distributed Fan Principal Component Analysis
#' @description This function performs distributed Fan-type principal component analysis
#' on a numeric dataset split across multiple nodes.
#'
#' @param data A numeric matrix containing the total dataset.
#' @param m An integer specifying the number of principal components.
#' @param n1 An integer specifying the length of each data subset.
#' @param K An integer specifying the number of nodes.
#'
#' @return A list with the following components:
#' \describe{
#'   \item{AF}{List of estimated loading matrices for each node.}
#'   \item{DF}{List of diagonal residual variance matrices for each node.}
#'   \item{SigmahatF}{List of covariance matrices for each node.}
#' }
#'
#' @examples
#' set.seed(123)
#' data <- matrix(rnorm(500), nrow = 100, ncol = 5)
#' DFanPC(data = data, m = 3, n1 = 20, K = 5)
#'
#' @export
DFanPC <- function(data, m, n1, K) {
  SigmahatF <- vector("list", K)
  AF <- vector("list", K)
  DF <- vector("list", K)
  n <- nrow(data)
  for (i in seq_len(K)) {
    L <- sample(seq_len(n), n1, replace = FALSE)
    R <- matrix(0, nrow = n1, ncol = n)
    R[cbind(seq_len(n1), L)] <- 1
    X1 <- R %*% as.matrix(data)
    X <- scale(X1)
    SigmahatF[[i]] <- cor(X)
    eig2 <- eigen(SigmahatF[[i]])
    
    lambdahat <- eig2$values[1:m]
    ind <- order(lambdahat, decreasing = TRUE)
    Q <- eig2$vectors[, ind, drop = FALSE]
    
    AF[[i]] <- Q[, 1:m, drop = FALSE]
    hF <- diag(AF[[i]] %*% t(AF[[i]]))
    DF[[i]] <- diag(SigmahatF[[i]] - hF)
  }
  
  return(list(AF = AF, DF = DF, SigmahatF = SigmahatF))
}