% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iface-class.R
\name{iface}
\alias{iface}
\title{Construct an interface specification}
\usage{
iface(..., .groups = NULL, .default = NULL)
}
\arguments{
\item{...}{The specification of the interface (see details), or an unnamed
\code{iface} object to extend, or both.}

\item{.groups}{either \code{FALSE} for no grouping allowed or a formula of the form
\code{~ var1 + var2 + ...} which defines what columns must be grouped in the dataframe
(and in which order). If \code{NULL} (the default) then any grouping is permitted.
If the formula contains a dot e.g. \code{~ . + var1 + var2} then the grouping
must include \code{var1} and \code{var2} but other groups are also allowed.}

\item{.default}{a default value to supply if there is nothing given in a
function parameter using the \code{iface} as a formal. This is either \code{NULL} in
which case there is no default, \code{TRUE} in which case the default is a zero
row dataframe conforming to the specification, or a provided dataframe,
which is checked to conform, and used as the default.}
}
\value{
the definition of an interface as a \code{iface} object
}
\description{
An \code{iface} specification defines the expected structure of a dataframe, in
terms of the column names, column types, grouping structure and uniqueness
constraints that the dataframe must conform to. A dataframe can be tested
for conformance to an \code{iface} specification using \code{\link[=ivalidate]{ivalidate()}}.
}
\details{
An \code{iface} specification is designed to be used to define the type of a
parameter in a function. This is done by using the \code{iface} specification as
the default value of the parameter in the function definition. The definition
can then be validated at runtime by a call to \code{\link[=ivalidate]{ivalidate()}} inside the
function.

When developing a function output an \code{iface} specification may also be used
in \code{\link[=ireturn]{ireturn()}} to enforce that the output of a function is correct.

\code{iface} definitions can be printed and included in \code{roxygen2} documentation
and help us to document input dataframe parameters and dataframe return
values in a standardised way by using the \verb{@iparam} \code{roxygen2} tag.

\code{iface} specifications are defined in the form of a named list of formulae with the
structure \code{column_name = type ~ "documentation"}.

\code{type} can be one of \code{anything}, \code{character}, \code{complete}, \code{date}, \code{default}, \code{double}, \code{enum}, \code{factor}, \code{finite}, \code{group_unique}, \code{in_range}, \code{integer}, \code{logical}, \code{not_missing}, \code{numeric}, \code{of_type}, \code{positive_double}, \code{positive_integer}, \code{proportion}, \code{unique_id} (e.g. \code{enum(level1,level2,...)},
\code{in_range(min,max)}) or alternatively anything that resolves to a function e.g.
\code{as.ordered}.

If \code{type} is a function name, then the function must take a single vector
parameter and return a single vector of the same size. The function must also
return a zero length vector of an appropriate type if passed \code{NULL}.

\code{type} can also be a concatenation of rules separated by \code{+}, e.g.
\code{integer + group_unique} for an integer that is unique within a group.
}
\examples{

test_df = tibble::tibble(
  grp = c(rep("a",10),rep("b",10)), 
  col1 = c(1:10,1:10)
) \%>\% dplyr::group_by(grp)

my_iface = iface( 
  col1 = integer + group_unique ~ "an integer column",
  .default = test_df
)

print(my_iface)

# the function x defines a formal `df` with default value of `my_iface`
# this default value is used to validate the structure of the user supplied
# value when the function is called.
x = function(df = my_iface, ...) {
  df = ivalidate(df,...)
  return(df)
}

# this works
x(tibble::tibble(col1 = c(1,2,3)))

# this fails as x is of the wrong type
try(x(tibble::tibble(col1 = c("a","b","c"))))

# this fails as x has duplicates
try(x(tibble::tibble(col1 = c(1,2,3,3))))

# this gives the default value
x()


my_iface2 = iface(
  first_col = numeric ~ "column order example",
  my_iface, 
  last_col = character ~ "another col", .groups = ~ first_col + col1
)
print(my_iface2)



my_iface_3 = iface( 
  col1 = integer + group_unique ~ "an integer column",
  .default = test_df_2
)
x = function(d = my_iface_3) {ivalidate(d)}

# Doesn't work as test_df_2 hasn't been defined
try(x())

test_df_2 = tibble::tibble(
  grp = c(rep("a",10),rep("b",10)), 
  col1 = c(1:10,1:10)
) \%>\% dplyr::group_by(grp)

# now it works as has been defined
x()

# it still works as default has been cached.
rm(test_df_2)
x()
}
\concept{interface}
