/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Zmodule.h" 

#include "H5private.h"   
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Oprivate.h"  
#include "H5Pprivate.h"  
#include "H5PLprivate.h" 
#include "H5Sprivate.h"  
#include "H5Zpkg.h"      

#ifdef H5_HAVE_SZLIB_H
#include "szlib.h"
#endif

#ifdef H5Z_DEBUG
typedef struct H5Z_stats_t {
    struct {
        hsize_t       total;  
        hsize_t       errors; 
        H5_timevals_t times;  
    } stats[2];               
} H5Z_stats_t;
#endif 

typedef struct H5Z_object_t {
    H5Z_filter_t filter_id; 
    htri_t       found;     
#ifdef H5_HAVE_PARALLEL
    bool sanity_checked; 
#endif                   
} H5Z_object_t;

typedef enum {
    H5Z_PRELUDE_CAN_APPLY, 
    H5Z_PRELUDE_SET_LOCAL  
} H5Z_prelude_type_t;

bool H5_PKG_INIT_VAR = false;

static size_t        H5Z_table_alloc_g = 0;
static size_t        H5Z_table_used_g  = 0;
static H5Z_class2_t *H5Z_table_g       = NULL;
#ifdef H5Z_DEBUG
static H5Z_stats_t *H5Z_stat_table_g = NULL;

static const bool DUMP_DEBUG_STATS_g = false;
#endif 

static int H5Z__find_idx(H5Z_filter_t id);
static int H5Z__check_unregister_dset_cb(void *obj_ptr, hid_t obj_id, void *key);
static int H5Z__check_unregister_group_cb(void *obj_ptr, hid_t obj_id, void *key);
static int H5Z__flush_file_cb(void *obj_ptr, hid_t obj_id, void *key);

herr_t
H5Z__init_package(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5Z_register(H5Z_SHUFFLE) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to register shuffle filter");
    if (H5Z_register(H5Z_FLETCHER32) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to register fletcher32 filter");
    if (H5Z_register(H5Z_NBIT) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to register nbit filter");
    if (H5Z_register(H5Z_SCALEOFFSET) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to register scaleoffset filter");

        
#ifdef H5_HAVE_FILTER_DEFLATE
    if (H5Z_register(H5Z_DEFLATE) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to register deflate filter");
#endif 
#ifdef H5_HAVE_FILTER_SZIP
    {
        int encoder_enabled = SZ_encoder_enabled();
        if (encoder_enabled < 0)
            HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "check for szip encoder failed");

        H5Z_SZIP->encoder_present = (unsigned)encoder_enabled;
        if (H5Z_register(H5Z_SZIP) < 0)
            HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to register szip filter");
    }
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5Z_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
#ifdef H5Z_DEBUG
        char   comment[16], bandwidth[32];
        int    dir, nprint = 0;
        size_t i;

        if (DUMP_DEBUG_STATS_g) {
            for (i = 0; i < H5Z_table_used_g; i++) {
                for (dir = 0; dir < 2; dir++) {
                    struct {
                        char *user;
                        char *system;
                        char *elapsed;
                    } timestrs = {H5_timer_get_time_string(H5Z_stat_table_g[i].stats[dir].times.user),
                                  H5_timer_get_time_string(H5Z_stat_table_g[i].stats[dir].times.system),
                                  H5_timer_get_time_string(H5Z_stat_table_g[i].stats[dir].times.elapsed)};
                    if (0 == H5Z_stat_table_g[i].stats[dir].total)
                        goto next;

                    if (0 == nprint++) {
                        
                        Rfprintf(Rstdout, "H5Z: filter statistics "
                                        "accumulated over life of library:\n");
                        Rfprintf(Rstdout, "   %-16s %10s %10s %8s %8s %8s %10s\n", "Filter", "Total", "Errors",
                                "User", "System", "Elapsed", "Bandwidth");
                        Rfprintf(Rstdout, "   %-16s %10s %10s %8s %8s %8s %10s\n", "------", "-----", "------",
                                "----", "------", "-------", "---------");
                    } 

                    
                    strncpy(comment, H5Z_table_g[i].name, sizeof comment);
                    comment[sizeof(comment) - 1] = '\0';

                    
                    H5_bandwidth(bandwidth, sizeof(bandwidth), (double)(H5Z_stat_table_g[i].stats[dir].total),
                                 H5Z_stat_table_g[i].stats[dir].times.elapsed);

                    
                    Rfprintf(Rstdout, "   %s%-15s %10" PRIdHSIZE " %10" PRIdHSIZE " %8s %8s %8s %10s\n",
                            (dir ? "<" : ">"), comment, H5Z_stat_table_g[i].stats[dir].total,
                            H5Z_stat_table_g[i].stats[dir].errors, timestrs.user, timestrs.system,
                            timestrs.elapsed, bandwidth);
next:
                    free(timestrs.user);
                    free(timestrs.system);
                    free(timestrs.elapsed);
                } 
            }     
        }         
#endif            

        
        if (H5Z_table_g) {
            H5Z_table_g = (H5Z_class2_t *)H5MM_xfree(H5Z_table_g);

#ifdef H5Z_DEBUG
            H5Z_stat_table_g = (H5Z_stats_t *)H5MM_xfree(H5Z_stat_table_g);
#endif 
            H5Z_table_used_g = H5Z_table_alloc_g = 0;

            n++;
        } 

        
        if (0 == n)
            H5_PKG_INIT_VAR = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

herr_t
H5Zregister(const void *cls)
{
    const H5Z_class2_t *cls_real  = (const H5Z_class2_t *)cls; 
    herr_t              ret_value = SUCCEED;                   
#ifndef H5_NO_DEPRECATED_SYMBOLS
    H5Z_class2_t cls_new; 
#endif

    FUNC_ENTER_API(FAIL)

    
    if (cls_real == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid filter class");

    
    if (cls_real->version != H5Z_CLASS_T_VERS) {
#ifndef H5_NO_DEPRECATED_SYMBOLS
        
        const H5Z_class1_t *cls_old = (const H5Z_class1_t *)cls;

        
        cls_new.version         = H5Z_CLASS_T_VERS;
        cls_new.id              = cls_old->id;
        cls_new.encoder_present = 1;
        cls_new.decoder_present = 1;
        cls_new.name            = cls_old->name;
        cls_new.can_apply       = cls_old->can_apply;
        cls_new.set_local       = cls_old->set_local;
        cls_new.filter          = cls_old->filter;

        
        cls_real = &cls_new;

#else  
        
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid H5Z_class_t version number");
#endif 
    }  

    if (cls_real->id < 0 || cls_real->id > H5Z_FILTER_MAX)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid filter identification number");
    if (cls_real->id < H5Z_FILTER_RESERVED)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to modify predefined filters");
    if (cls_real->filter == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no filter function specified");

    
    if (H5Z_register(cls_real) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to register filter");

done:
    FUNC_LEAVE_API(ret_value)
}

herr_t
H5Z_register(const H5Z_class2_t *cls)
{
    size_t i;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(cls);
    assert(cls->id >= 0 && cls->id <= H5Z_FILTER_MAX);

    
    for (i = 0; i < H5Z_table_used_g; i++)
        if (H5Z_table_g[i].id == cls->id)
            break;

    
    if (i >= H5Z_table_used_g) {
        if (H5Z_table_used_g >= H5Z_table_alloc_g) {
            size_t        n     = MAX(H5Z_MAX_NFILTERS, 2 * H5Z_table_alloc_g);
            H5Z_class2_t *table = (H5Z_class2_t *)H5MM_realloc(H5Z_table_g, n * sizeof(H5Z_class2_t));
#ifdef H5Z_DEBUG
            H5Z_stats_t *stat_table = (H5Z_stats_t *)H5MM_realloc(H5Z_stat_table_g, n * sizeof(H5Z_stats_t));
#endif 
            if (!table)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "unable to extend filter table");
            H5Z_table_g = table;
#ifdef H5Z_DEBUG
            if (!stat_table)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "unable to extend filter statistics table");
            H5Z_stat_table_g = stat_table;
#endif 
            H5Z_table_alloc_g = n;
        } 

        
        i = H5Z_table_used_g++;
        H5MM_memcpy(H5Z_table_g + i, cls, sizeof(H5Z_class2_t));
#ifdef H5Z_DEBUG
        memset(H5Z_stat_table_g + i, 0, sizeof(H5Z_stats_t));
#endif 
    }  
    
    else {
        
        H5MM_memcpy(H5Z_table_g + i, cls, sizeof(H5Z_class2_t));
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5Zunregister(H5Z_filter_t id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (id < 0 || id > H5Z_FILTER_MAX)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid filter identification number");
    if (id < H5Z_FILTER_RESERVED)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to modify predefined filters");

    
    if (H5Z__unregister(id) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to unregister filter");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Z__unregister(H5Z_filter_t filter_id)
{
    size_t       filter_index;        
    H5Z_object_t object;              
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(filter_id >= 0 && filter_id <= H5Z_FILTER_MAX);

    
    for (filter_index = 0; filter_index < H5Z_table_used_g; filter_index++)
        if (H5Z_table_g[filter_index].id == filter_id)
            break;

    
    if (filter_index >= H5Z_table_used_g)
        HGOTO_ERROR(H5E_PLINE, H5E_NOTFOUND, FAIL, "filter is not registered");

    
    object.filter_id = filter_id;
    object.found     = false;
#ifdef H5_HAVE_PARALLEL
    object.sanity_checked = false;
#endif 

    
    if (H5I_iterate(H5I_DATASET, H5Z__check_unregister_dset_cb, &object, false) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_BADITER, FAIL, "iteration failed");

    if (object.found)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTRELEASE, FAIL,
                    "can't unregister filter because a dataset is still using it");

    
    if (H5I_iterate(H5I_GROUP, H5Z__check_unregister_group_cb, &object, false) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_BADITER, FAIL, "iteration failed");

    if (object.found)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTRELEASE, FAIL,
                    "can't unregister filter because a group is still using it");

    
    if (H5I_iterate(H5I_FILE, H5Z__flush_file_cb, &object, false) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_BADITER, FAIL, "iteration failed");

    
    
    memmove(&H5Z_table_g[filter_index], &H5Z_table_g[filter_index + 1],
            sizeof(H5Z_class2_t) * ((H5Z_table_used_g - 1) - filter_index));
#ifdef H5Z_DEBUG
    memmove(&H5Z_stat_table_g[filter_index], &H5Z_stat_table_g[filter_index + 1],
            sizeof(H5Z_stats_t) * ((H5Z_table_used_g - 1) - filter_index));
#endif 
    H5Z_table_used_g--;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5Z__check_unregister(hid_t ocpl_id, H5Z_filter_t filter_id)
{
    H5P_genplist_t *plist;             
    htri_t          ret_value = false; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (plist = H5P_object_verify(ocpl_id, H5P_OBJECT_CREATE, true)))
        HGOTO_ERROR(H5E_PLINE, H5E_BADID, FAIL, "can't find object for ID");

    
    if ((ret_value = H5P_filter_in_pline(plist, filter_id)) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTGET, FAIL, "can't check filter in pipeline");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5Z__check_unregister_group_cb(void H5_ATTR_UNUSED *obj_ptr, hid_t obj_id, void *key)
{
    hid_t                 ocpl_id = -1;
    H5Z_object_t         *object  = (H5Z_object_t *)key;
    H5VL_object_t        *vol_obj;     
    H5VL_group_get_args_t vol_cb_args; 
    htri_t                filter_in_pline = false;
    int                   ret_value       = false; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (vol_obj = (H5VL_object_t *)H5I_object_verify(obj_id, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid group identifier");

    
    vol_cb_args.op_type               = H5VL_GROUP_GET_GCPL;
    vol_cb_args.args.get_gcpl.gcpl_id = H5I_INVALID_HID;

    
    if (H5VL_group_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTGET, H5I_INVALID_HID, "unable to get group creation properties");

    if ((ocpl_id = vol_cb_args.args.get_gcpl.gcpl_id) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTGET, FAIL, "can't get group creation property list");

    
    if ((filter_in_pline = H5Z__check_unregister(ocpl_id, object->filter_id)) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTGET, FAIL, "can't check filter in pipeline");

    
    if (filter_in_pline) {
        object->found = true;
        ret_value     = true;
    }

done:
    if (ocpl_id > 0)
        if (H5I_dec_app_ref(ocpl_id) < 0)
            HDONE_ERROR(H5E_PLINE, H5E_CANTDEC, FAIL, "can't release plist");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5Z__check_unregister_dset_cb(void H5_ATTR_UNUSED *obj_ptr, hid_t obj_id, void *key)
{
    hid_t                   ocpl_id = -1;
    H5Z_object_t           *object  = (H5Z_object_t *)key;
    H5VL_object_t          *vol_obj;     
    H5VL_dataset_get_args_t vol_cb_args; 
    htri_t                  filter_in_pline = false;
    int                     ret_value       = false; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (vol_obj = (H5VL_object_t *)H5I_object_verify(obj_id, H5I_DATASET)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid dataset identifier");

    
    vol_cb_args.op_type               = H5VL_DATASET_GET_DCPL;
    vol_cb_args.args.get_dcpl.dcpl_id = H5I_INVALID_HID;

    
    if (H5VL_dataset_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, H5I_INVALID_HID, "unable to get dataset creation properties");

    if ((ocpl_id = vol_cb_args.args.get_dcpl.dcpl_id) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTGET, FAIL, "can't get dataset creation property list");

    
    if ((filter_in_pline = H5Z__check_unregister(ocpl_id, object->filter_id)) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTGET, FAIL, "can't check filter in pipeline");

    
    if (filter_in_pline) {
        object->found = true;
        ret_value     = true;
    }

done:
    if (ocpl_id > 0)
        if (H5I_dec_app_ref(ocpl_id) < 0)
            HDONE_ERROR(H5E_PLINE, H5E_CANTDEC, FAIL, "can't release plist");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5Z__flush_file_cb(void H5_ATTR_UNUSED *obj_ptr, hid_t obj_id, void H5_ATTR_PARALLEL_USED *key)
{

#ifdef H5_HAVE_PARALLEL
    H5Z_object_t *object = (H5Z_object_t *)key;
#endif                                              
    int                       ret_value = false;    
    H5VL_file_specific_args_t vol_cb_args_specific; 
    H5VL_object_t            *vol_obj;              
    H5VL_file_get_args_t      vol_cb_args;          
    bool                      is_native_vol_obj = true;
    unsigned int              intent            = 0;

    FUNC_ENTER_PACKAGE

    
    assert(key);

    
    if (NULL == (vol_obj = (H5VL_object_t *)H5I_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid file identifier");

    
    vol_cb_args.op_type               = H5VL_FILE_GET_INTENT;
    vol_cb_args.args.get_intent.flags = &intent;

    
    if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to get file's intent flags");

    if (H5VL_object_is_native(vol_obj, &is_native_vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, H5I_INVALID_HID,
                    "can't determine if VOL object is native connector object");

    
    if (H5F_ACC_RDWR & intent) {

#ifdef H5_HAVE_PARALLEL
        
        if (is_native_vol_obj) {
            H5F_t *f = (H5F_t *)obj_ptr; 

            
            if (H5F_HAS_FEATURE(f, H5FD_FEAT_HAS_MPI)) {

                
                
                if (H5_coll_api_sanity_check_g && !object->sanity_checked) {
                    MPI_Comm mpi_comm; 

                    
                    if (H5F_mpi_retrieve_comm(obj_id, H5P_DEFAULT, &mpi_comm) < 0)
                        HGOTO_ERROR(H5E_PLINE, H5E_CANTGET, FAIL, "can't get MPI communicator");

                    
                    if (mpi_comm != MPI_COMM_NULL)
                        MPI_Barrier(mpi_comm);

                    
                    object->sanity_checked = true;
                } 
            }     
        }
#endif 

        
        vol_cb_args_specific.op_type             = H5VL_FILE_FLUSH;
        vol_cb_args_specific.args.flush.obj_type = H5I_FILE;
        vol_cb_args_specific.args.flush.scope    = H5F_SCOPE_GLOBAL;

        
        if (H5VL_file_specific(vol_obj, &vol_cb_args_specific, H5P_DATASET_XFER_DEFAULT, NULL) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTFLUSH, FAIL, "unable to flush file hierarchy");

    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5Zfilter_avail(H5Z_filter_t id)
{
    htri_t ret_value = false; 

    FUNC_ENTER_API(FAIL)

    
    if (id < 0 || id > H5Z_FILTER_MAX)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid filter identification number");

    if ((ret_value = H5Z_filter_avail(id)) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_NOTFOUND, FAIL, "unable to check the availability of the filter");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Z_filter_avail(H5Z_filter_t id)
{
    H5PL_key_t          key;               
    const H5Z_class2_t *filter_info;       
    size_t              i;                 
    htri_t              ret_value = false; 

    FUNC_ENTER_NOAPI(FAIL)

    
    for (i = 0; i < H5Z_table_used_g; i++)
        if (H5Z_table_g[i].id == id)
            HGOTO_DONE(true);

    key.id = (int)id;
    if (NULL != (filter_info = (const H5Z_class2_t *)H5PL_load(H5PL_TYPE_FILTER, &key))) {
        if (H5Z_register(filter_info) < 0)
            HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to register loaded filter");
        HGOTO_DONE(true);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5Z__prelude_callback(const H5O_pline_t *pline, hid_t dcpl_id, hid_t type_id, hid_t space_id,
                      H5Z_prelude_type_t prelude_type)
{
    H5Z_class2_t *fclass;           
    size_t        u;                
    htri_t        ret_value = true; 

    FUNC_ENTER_PACKAGE

    assert(pline->nused > 0);

    
    for (u = 0; u < pline->nused; u++) {
        
        if (H5Z_find(pline->filter[u].flags & H5Z_FLAG_OPTIONAL, pline->filter[u].id, &fclass) < 0)
            HGOTO_ERROR(H5E_PLINE, H5E_NOTFOUND, FAIL, "required filter was not located");
        if (fclass) {
            
            switch (prelude_type) {
                case H5Z_PRELUDE_CAN_APPLY:
                    
                    if (!fclass->encoder_present)
                        HGOTO_ERROR(H5E_PLINE, H5E_NOENCODER, FAIL,
                                    "Filter present but encoding is disabled.");

                    
                    if (fclass->can_apply) {
                        htri_t status;

                        
                        H5_BEFORE_USER_CB(FAIL)
                            {
                                
                                status = (fclass->can_apply)(dcpl_id, type_id, space_id);
                            }
                        H5_AFTER_USER_CB(FAIL)

                        
                        if (status < 0)
                            HGOTO_ERROR(H5E_PLINE, H5E_CANAPPLY, FAIL, "error during user callback");

                        
                        if (status == false && !(pline->filter[u].flags & H5Z_FLAG_OPTIONAL))
                            HGOTO_ERROR(H5E_PLINE, H5E_CANAPPLY, FAIL, "filter parameters not appropriate");
                    } 
                    break;

                case H5Z_PRELUDE_SET_LOCAL:
                    
                    if (fclass->set_local) {
                        herr_t status;

                        
                        H5_BEFORE_USER_CB(FAIL)
                            {
                                
                                status = (fclass->set_local)(dcpl_id, type_id, space_id);
                            }
                        H5_AFTER_USER_CB(FAIL)

                        
                        if (status < 0)
                            HGOTO_ERROR(H5E_PLINE, H5E_SETLOCAL, FAIL, "error during user callback");
                    } 
                    break;

                default:
                    assert("invalid prelude type" && 0);
            } 
        }     
    }         

done:

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5Z__prepare_prelude_callback_dcpl(hid_t dcpl_id, hid_t type_id, H5Z_prelude_type_t prelude_type)
{
    hid_t         space_id    = -1;      
    H5O_layout_t *dcpl_layout = NULL;    
    herr_t        ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(H5I_GENPROP_LST == H5I_get_type(dcpl_id));
    assert(H5I_DATATYPE == H5I_get_type(type_id));

    
    if (dcpl_id != H5P_DATASET_CREATE_DEFAULT) {
        H5P_genplist_t *dc_plist; 

        
        if (NULL == (dcpl_layout = (H5O_layout_t *)H5MM_calloc(sizeof(H5O_layout_t))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "unable to allocate dcpl layout buffer");

        
        if (NULL == (dc_plist = (H5P_genplist_t *)H5I_object(dcpl_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "can't get dataset creation property list");

        
        if (H5P_peek(dc_plist, H5D_CRT_LAYOUT_NAME, dcpl_layout) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't retrieve layout");

        
        if (H5D_CHUNKED == dcpl_layout->type) {
            H5O_pline_t dcpl_pline; 

            
            if (H5P_peek(dc_plist, H5O_CRT_PIPELINE_NAME, &dcpl_pline) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't retrieve pipeline filter");

            
            if (dcpl_pline.nused > 0) {
                hsize_t chunk_dims[H5O_LAYOUT_NDIMS]; 
                H5S_t  *space;                        
                size_t  u;                            

                
                for (u = 0; u < dcpl_layout->u.chunk.ndims; u++)
                    chunk_dims[u] = dcpl_layout->u.chunk.dim[u];
                if (NULL == (space = H5S_create_simple(dcpl_layout->u.chunk.ndims, chunk_dims, NULL)))
                    HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCREATE, FAIL, "can't create simple dataspace");

                
                if ((space_id = H5I_register(H5I_DATASPACE, space, false)) < 0) {
                    (void)H5S_close(space);
                    HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, FAIL, "unable to register dataspace ID");
                }

                
                if (H5Z__prelude_callback(&dcpl_pline, dcpl_id, type_id, space_id, prelude_type) < 0)
                    HGOTO_ERROR(H5E_PLINE, H5E_CANAPPLY, FAIL, "unable to apply filter");
            }
        }
    }

done:
    if (space_id > 0 && H5I_dec_ref(space_id) < 0)
        HDONE_ERROR(H5E_PLINE, H5E_CANTRELEASE, FAIL, "unable to close dataspace");

    if (dcpl_layout)
        dcpl_layout = (H5O_layout_t *)H5MM_xfree(dcpl_layout);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Z_can_apply(hid_t dcpl_id, hid_t type_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5Z__prepare_prelude_callback_dcpl(dcpl_id, type_id, H5Z_PRELUDE_CAN_APPLY) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANAPPLY, FAIL, "unable to apply filter");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Z_set_local(hid_t dcpl_id, hid_t type_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5Z__prepare_prelude_callback_dcpl(dcpl_id, type_id, H5Z_PRELUDE_SET_LOCAL) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_SETLOCAL, FAIL, "local filter parameters not set");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Z_can_apply_direct(const H5O_pline_t *pline)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(pline->nused > 0);

    
    if (H5Z__prelude_callback(pline, (hid_t)-1, (hid_t)-1, (hid_t)-1, H5Z_PRELUDE_CAN_APPLY) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANAPPLY, FAIL, "unable to apply filter");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Z_set_local_direct(const H5O_pline_t *pline)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(pline->nused > 0);

    
    if (H5Z__prelude_callback(pline, (hid_t)-1, (hid_t)-1, (hid_t)-1, H5Z_PRELUDE_SET_LOCAL) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_SETLOCAL, FAIL, "local filter parameters not set");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5Z_ignore_filters(hid_t dcpl_id, const H5T_t *type, const H5S_t *space)
{
    H5P_genplist_t *dc_plist;                
    H5O_pline_t     pline;                   
    H5S_class_t     space_class;             
    H5T_class_t     type_class;              
    bool            bad_for_filters = false; 
    htri_t          ret_value       = false; 

    FUNC_ENTER_NOAPI(FAIL)

    if (NULL == (dc_plist = (H5P_genplist_t *)H5I_object(dcpl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "can't get dataset creation property list");

    
    if (H5P_peek(dc_plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTGET, FAIL, "can't retrieve pipeline filter");

    
    space_class = H5S_GET_EXTENT_TYPE(space);
    type_class  = H5T_get_class(type, false);

    
    bad_for_filters = (H5S_NULL == space_class || H5S_SCALAR == space_class || H5T_VLEN == type_class ||
                       (H5T_STRING == type_class && true == H5T_is_variable_str(type)));

    
    if (bad_for_filters) {
        size_t ii;
        if (pline.nused > 0) {
            for (ii = 0; ii < pline.nused; ii++) {
                if (!(pline.filter[ii].flags & H5Z_FLAG_OPTIONAL))
                    HGOTO_ERROR(H5E_PLINE, H5E_CANTFILTER, FAIL, "not suitable for filters");
            }

            
            ret_value = true;
        }
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Z_modify(const H5O_pline_t *pline, H5Z_filter_t filter, unsigned flags, size_t cd_nelmts,
           const unsigned int cd_values[])
{
    size_t idx;                 
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(pline);
    assert(filter >= 0 && filter <= H5Z_FILTER_MAX);
    assert(0 == (flags & ~((unsigned)H5Z_FLAG_DEFMASK)));
    assert(0 == cd_nelmts || cd_values);

    
    for (idx = 0; idx < pline->nused; idx++)
        if (pline->filter[idx].id == filter)
            break;

    
    if (idx > pline->nused)
        HGOTO_ERROR(H5E_PLINE, H5E_NOTFOUND, FAIL, "filter not in pipeline");

    
    pline->filter[idx].flags     = flags;
    pline->filter[idx].cd_nelmts = cd_nelmts;

    
    if (pline->filter[idx].cd_values != NULL && pline->filter[idx].cd_values != pline->filter[idx]._cd_values)
        H5MM_xfree(pline->filter[idx].cd_values);

    
    if (cd_nelmts > 0) {
        size_t i; 

        
        if (cd_nelmts > H5Z_COMMON_CD_VALUES) {
            pline->filter[idx].cd_values = (unsigned *)H5MM_malloc(cd_nelmts * sizeof(unsigned));
            if (NULL == pline->filter[idx].cd_values)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                            "memory allocation failed for filter parameters");
        } 
        else
            pline->filter[idx].cd_values = pline->filter[idx]._cd_values;

        
        for (i = 0; i < cd_nelmts; i++)
            pline->filter[idx].cd_values[i] = cd_values[i];
    } 
    else
        pline->filter[idx].cd_values = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Z_append(H5O_pline_t *pline, H5Z_filter_t filter, unsigned flags, size_t cd_nelmts,
           const unsigned int cd_values[])
{
    size_t idx;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(pline);
    assert(filter >= 0 && filter <= H5Z_FILTER_MAX);
    assert(0 == (flags & ~((unsigned)H5Z_FLAG_DEFMASK)));
    assert(0 == cd_nelmts || cd_values);

    
    if (pline->nused >= H5Z_MAX_NFILTERS)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "too many filters in pipeline");

    
    if (pline->version == 0)
        pline->version = H5O_PLINE_VERSION_1;

    
    if (pline->nused >= pline->nalloc) {
        H5O_pline_t x;
        size_t      n;

        
        for (n = 0; n < pline->nalloc; ++n)
            if (pline->filter[n].cd_values == pline->filter[n]._cd_values)
                pline->filter[n].cd_values = (unsigned *)((void *)~((size_t)NULL));

        x.nalloc = MAX(H5Z_MAX_NFILTERS, 2 * pline->nalloc);
        x.filter = (H5Z_filter_info_t *)H5MM_realloc(pline->filter, x.nalloc * sizeof(x.filter[0]));
        if (NULL == x.filter)
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for filter pipeline");

        
        for (n = 0; n < pline->nalloc; ++n)
            if (x.filter[n].cd_values == (void *)~((size_t)NULL))
                x.filter[n].cd_values = x.filter[n]._cd_values;

        
        pline->nalloc = x.nalloc;
        pline->filter = x.filter;
    } 

    
    idx                          = pline->nused;
    pline->filter[idx].id        = filter;
    pline->filter[idx].flags     = flags;
    pline->filter[idx].name      = NULL; 
    pline->filter[idx].cd_nelmts = cd_nelmts;
    if (cd_nelmts > 0) {
        size_t i; 

        
        if (cd_nelmts > H5Z_COMMON_CD_VALUES) {
            pline->filter[idx].cd_values = (unsigned *)H5MM_malloc(cd_nelmts * sizeof(unsigned));
            if (NULL == pline->filter[idx].cd_values)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for filter");
        } 
        else
            pline->filter[idx].cd_values = pline->filter[idx]._cd_values;

        
        for (i = 0; i < cd_nelmts; i++)
            pline->filter[idx].cd_values[i] = cd_values[i];
    } 
    else
        pline->filter[idx].cd_values = NULL;

    pline->nused++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5Z__find_idx(H5Z_filter_t id)
{
    size_t i;                
    int    ret_value = FAIL; 

    FUNC_ENTER_PACKAGE_NOERR

    for (i = 0; i < H5Z_table_used_g; i++)
        if (H5Z_table_g[i].id == id)
            HGOTO_DONE((int)i);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Z_find(bool try, H5Z_filter_t id, H5Z_class2_t **cls)
{
    int    idx;                 
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if ((idx = H5Z__find_idx(id)) < 0) {
        *cls = NULL;

        
        if (!try)
            HGOTO_ERROR(H5E_PLINE, H5E_NOTFOUND, FAIL, "required filter %d is not registered", id);
    }
    else
        *cls = H5Z_table_g + idx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Z_pipeline(const H5O_pline_t *pline, unsigned flags, unsigned *filter_mask , H5Z_EDC_t edc_read,
             H5Z_cb_t cb_struct, size_t *nbytes , size_t *buf_size ,
             void **buf )
{
    size_t        idx;
    size_t        new_nbytes;
    int           fclass_idx;    
    H5Z_class2_t *fclass = NULL; 
#ifdef H5Z_DEBUG
    H5Z_stats_t  *fstats = NULL; 
    H5_timer_t    timer;         
    H5_timevals_t times;         
#endif
    unsigned failed = 0;
    unsigned tmp_flags;
    size_t   i;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(0 == (flags & ~((unsigned)H5Z_FLAG_INVMASK)));
    assert(filter_mask);
    assert(nbytes && *nbytes > 0);
    assert(buf_size && *buf_size > 0);
    assert(buf && *buf);
    assert(!pline || pline->nused < H5Z_MAX_NFILTERS);

    

#ifdef H5Z_DEBUG
    H5_timer_init(&timer);
#endif
    if (pline && (flags & H5Z_FLAG_REVERSE)) { 
        for (i = pline->nused; i > 0; --i) {
            idx = i - 1;
            if (*filter_mask & ((unsigned)1 << idx)) {
                failed |= (unsigned)1 << idx;
                continue; 
            }

            
            if ((fclass_idx = H5Z__find_idx(pline->filter[idx].id)) < 0) {
                H5PL_key_t          key;
                const H5Z_class2_t *filter_info;
                bool                issue_error = false;

                
                key.id = (int)(pline->filter[idx].id);
                if (NULL != (filter_info = (const H5Z_class2_t *)H5PL_load(H5PL_TYPE_FILTER, &key))) {
                    
                    if (H5Z_register(filter_info) < 0)
                        HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to register filter");

                    
                    if ((fclass_idx = H5Z__find_idx(pline->filter[idx].id)) < 0)
                        issue_error = true;
                }
                else
                    issue_error = true;

                
                if (issue_error) {
                    
                    if (pline->filter[idx].name)
                        HGOTO_ERROR(H5E_PLINE, H5E_READERROR, FAIL, "required filter '%s' is not registered",
                                    pline->filter[idx].name);
                    else
                        HGOTO_ERROR(H5E_PLINE, H5E_READERROR, FAIL,
                                    "required filter (name unavailable) is not registered");
                }
            } 

            fclass = &H5Z_table_g[fclass_idx];

#ifdef H5Z_DEBUG
            fstats = &H5Z_stat_table_g[fclass_idx];
            H5_timer_start(&timer);
#endif

            tmp_flags = flags | (pline->filter[idx].flags);
            tmp_flags |= (edc_read == H5Z_DISABLE_EDC) ? H5Z_FLAG_SKIP_EDC : 0;

            H5E_PAUSE_ERRORS
                {
                    H5_BEFORE_USER_CB(FAIL)
                        {
                            new_nbytes = (fclass->filter)(tmp_flags, pline->filter[idx].cd_nelmts,
                                                          pline->filter[idx].cd_values, *nbytes, buf_size, buf);
                        }
                    H5_AFTER_USER_CB(FAIL)
                }
            H5E_RESUME_ERRORS

#ifdef H5Z_DEBUG
            H5_timer_stop(&timer);
            H5_timer_get_times(timer, &times);
            fstats->stats[1].times.elapsed += times.elapsed;
            fstats->stats[1].times.system += times.system;
            fstats->stats[1].times.user += times.user;

            fstats->stats[1].total += MAX(*nbytes, new_nbytes);
            if (0 == new_nbytes)
                fstats->stats[1].errors += *nbytes;
#endif

            if (0 == new_nbytes) {
                if (cb_struct.func) {
                    H5Z_cb_return_t status;

                    
                    H5_BEFORE_USER_CB(FAIL)
                        {
                            status = cb_struct.func(pline->filter[idx].id, *buf, *buf_size, cb_struct.op_data);
                        }
                    H5_AFTER_USER_CB(FAIL)
                    if (H5Z_CB_FAIL == status)
                        HGOTO_ERROR(H5E_PLINE, H5E_READERROR, FAIL, "filter returned failure during read");
                }
                else
                    HGOTO_ERROR(H5E_PLINE, H5E_READERROR, FAIL, "filter returned failure during read");

                *nbytes = *buf_size;
                failed |= (unsigned)1 << idx;
            }
            else
                *nbytes = new_nbytes;
        }
    }
    else if (pline)
    { 
        for (idx = 0; idx < pline->nused; idx++) {
            if (*filter_mask & ((unsigned)1 << idx)) {
                failed |= (unsigned)1 << idx;
                continue; 
            }
            if ((fclass_idx = H5Z__find_idx(pline->filter[idx].id)) < 0) {
                
                if ((pline->filter[idx].flags & H5Z_FLAG_OPTIONAL) == 0)
                    HGOTO_ERROR(H5E_PLINE, H5E_WRITEERROR, FAIL, "required filter is not registered");
                failed |= (unsigned)1 << idx;
                continue; 
            }             

            fclass = &H5Z_table_g[fclass_idx];

#ifdef H5Z_DEBUG
            fstats = &H5Z_stat_table_g[fclass_idx];
            H5_timer_start(&timer);
#endif

            H5E_PAUSE_ERRORS
                {
                    H5_BEFORE_USER_CB(FAIL)
                        {
                            new_nbytes = (fclass->filter)(flags | (pline->filter[idx].flags), pline->filter[idx].cd_nelmts,
                                                          pline->filter[idx].cd_values, *nbytes, buf_size, buf);
                        }
                    H5_AFTER_USER_CB(FAIL)
                }
            H5E_RESUME_ERRORS

#ifdef H5Z_DEBUG
            H5_timer_stop(&timer);
            H5_timer_get_times(timer, &times);
            fstats->stats[0].times.elapsed += times.elapsed;
            fstats->stats[0].times.system += times.system;
            fstats->stats[0].times.user += times.user;

            fstats->stats[0].total += MAX(*nbytes, new_nbytes);
            if (0 == new_nbytes)
                fstats->stats[0].errors += *nbytes;
#endif

            if (0 == new_nbytes) {
                if (0 == (pline->filter[idx].flags & H5Z_FLAG_OPTIONAL)) {
                    if (cb_struct.func) {
                        H5Z_cb_return_t status;

                        
                        H5_BEFORE_USER_CB(FAIL)
                            {
                                status = cb_struct.func(pline->filter[idx].id, *buf, *nbytes, cb_struct.op_data);
                            }
                        H5_AFTER_USER_CB(FAIL)
                        if (H5Z_CB_FAIL == status)
                            HGOTO_ERROR(H5E_PLINE, H5E_WRITEERROR, FAIL, "filter returned failure");
                    }
                    else
                        HGOTO_ERROR(H5E_PLINE, H5E_WRITEERROR, FAIL, "filter returned failure");

                    *nbytes = *buf_size;
                }
                failed |= (unsigned)1 << idx;
            }
            else
                *nbytes = new_nbytes;
        } 
    }

    *filter_mask = failed;

done:
    FUNC_LEAVE_NOAPI(ret_value)

}

H5Z_filter_info_t *
H5Z_filter_info(const H5O_pline_t *pline, H5Z_filter_t filter)
{
    size_t             idx;              
    H5Z_filter_info_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    assert(pline);
    assert(filter >= 0 && filter <= H5Z_FILTER_MAX);

    
    for (idx = 0; idx < pline->nused; idx++)
        if (pline->filter[idx].id == filter)
            break;

    
    if (idx >= pline->nused)
        HGOTO_ERROR(H5E_PLINE, H5E_NOTFOUND, NULL, "filter not in pipeline");

    
    ret_value = &pline->filter[idx];

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5Z_filter_in_pline(const H5O_pline_t *pline, H5Z_filter_t filter)
{
    size_t idx;              
    htri_t ret_value = true; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(pline);
    assert(filter >= 0 && filter <= H5Z_FILTER_MAX);

    
    for (idx = 0; idx < pline->nused; idx++)
        if (pline->filter[idx].id == filter)
            break;

    
    if (idx >= pline->nused)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5Z_all_filters_avail(const H5O_pline_t *pline)
{
    size_t i, j;             
    htri_t ret_value = true; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(pline);

    
    for (i = 0; i < pline->nused; i++) {
        
        for (j = 0; j < H5Z_table_used_g; j++)
            if (H5Z_table_g[j].id == pline->filter[i].id)
                break;

        
        if (j == H5Z_table_used_g)
            HGOTO_DONE(false);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Z_delete(H5O_pline_t *pline, H5Z_filter_t filter)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(pline);
    assert(filter >= 0 && filter <= H5Z_FILTER_MAX);

    
    if (pline->nused == 0)
        HGOTO_DONE(SUCCEED);

    
    if (H5Z_FILTER_ALL == filter) {
        if (H5O_msg_reset(H5O_PLINE_ID, pline) < 0)
            HGOTO_ERROR(H5E_PLINE, H5E_CANTFREE, FAIL, "can't release pipeline info");
    }
    
    else {
        size_t idx;           
        bool   found = false; 

        
        for (idx = 0; idx < pline->nused; idx++)
            if (pline->filter[idx].id == filter) {
                found = true;
                break;
            }

        
        if (!found)
            HGOTO_ERROR(H5E_PLINE, H5E_NOTFOUND, FAIL, "filter not in pipeline");

        
        if (pline->filter[idx].name && pline->filter[idx].name != pline->filter[idx]._name)
            assert((strlen(pline->filter[idx].name) + 1) > H5Z_COMMON_NAME_LEN);
        if (pline->filter[idx].name != pline->filter[idx]._name)
            pline->filter[idx].name = (char *)H5MM_xfree(pline->filter[idx].name);
        if (pline->filter[idx].cd_values && pline->filter[idx].cd_values != pline->filter[idx]._cd_values)
            assert(pline->filter[idx].cd_nelmts > H5Z_COMMON_CD_VALUES);
        if (pline->filter[idx].cd_values != pline->filter[idx]._cd_values)
            pline->filter[idx].cd_values = (unsigned *)H5MM_xfree(pline->filter[idx].cd_values);

        
        if ((idx + 1) < pline->nused) {
            
            for (; (idx + 1) < pline->nused; idx++) {
                pline->filter[idx] = pline->filter[idx + 1];
                if (pline->filter[idx].name && (strlen(pline->filter[idx].name) + 1) <= H5Z_COMMON_NAME_LEN)
                    pline->filter[idx].name = pline->filter[idx]._name;
                if (pline->filter[idx].cd_nelmts <= H5Z_COMMON_CD_VALUES)
                    pline->filter[idx].cd_values = pline->filter[idx]._cd_values;
            }
        }

        
        pline->nused--;

        
        memset(&pline->filter[pline->nused], 0, sizeof(H5Z_filter_info_t));
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Zget_filter_info(H5Z_filter_t filter, unsigned *filter_config_flags )
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (H5Z_get_filter_info(filter, filter_config_flags) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTGET, FAIL, "Filter info not retrieved");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Z_get_filter_info(H5Z_filter_t filter, unsigned int *filter_config_flags)
{
    H5Z_class2_t *fclass    = NULL;
    herr_t        ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5Z_find(false, filter, &fclass) < 0)
        HGOTO_ERROR(H5E_PLINE, H5E_BADVALUE, FAIL, "Filter not defined");

    
    if (filter_config_flags != NULL) {
        *filter_config_flags = 0;

        if (fclass->encoder_present)
            *filter_config_flags |= H5Z_FILTER_CONFIG_ENCODE_ENABLED;
        if (fclass->decoder_present)
            *filter_config_flags |= H5Z_FILTER_CONFIG_DECODE_ENABLED;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
