/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Zmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 
#include "H5Zpkg.h"      

static size_t H5Z__filter_fletcher32(unsigned flags, size_t cd_nelmts, const unsigned cd_values[],
                                     size_t nbytes, size_t *buf_size, void **buf);

const H5Z_class2_t H5Z_FLETCHER32[1] = {{
    H5Z_CLASS_T_VERS,       
    H5Z_FILTER_FLETCHER32,  
    1,                      
    1,                      
    "fletcher32",           
    NULL,                   
    NULL,                   
    H5Z__filter_fletcher32, 
}};

#define FLETCHER_LEN 4

static size_t
H5Z__filter_fletcher32(unsigned flags, size_t H5_ATTR_UNUSED cd_nelmts,
                       const unsigned H5_ATTR_UNUSED cd_values[], size_t nbytes, size_t *buf_size, void **buf)
{
    void          *outbuf = NULL; 
    unsigned char *src    = (unsigned char *)(*buf);
    uint32_t       fletcher;          
    uint32_t       reversed_fletcher; 
    uint8_t        c[4];
    uint8_t        tmp;
    size_t         ret_value = 0; 

    FUNC_ENTER_PACKAGE

    assert(sizeof(uint32_t) >= 4);

    if (flags & H5Z_FLAG_REVERSE) { 
        
        if (!(flags & H5Z_FLAG_SKIP_EDC)) {
            unsigned char *tmp_src;             
            size_t         src_nbytes = nbytes; 
            uint32_t       stored_fletcher;     

            
            src_nbytes -= FLETCHER_LEN;
            tmp_src = src + src_nbytes;
            UINT32DECODE(tmp_src, stored_fletcher);

            
            fletcher = H5_checksum_fletcher32(src, src_nbytes);

            
            H5MM_memcpy(c, &fletcher, (size_t)4);

            tmp  = c[1];
            c[1] = c[0];
            c[0] = tmp;

            tmp  = c[3];
            c[3] = c[2];
            c[2] = tmp;

            H5MM_memcpy(&reversed_fletcher, c, (size_t)4);

            
            if (stored_fletcher != fletcher && stored_fletcher != reversed_fletcher)
                HGOTO_ERROR(H5E_STORAGE, H5E_READERROR, 0, "data error detected by Fletcher32 checksum");
        }

        
        
        ret_value = nbytes - FLETCHER_LEN;
    }
    else {                  
        unsigned char *dst; 

        
        fletcher = H5_checksum_fletcher32(src, nbytes);

        if (NULL == (outbuf = H5MM_malloc(nbytes + FLETCHER_LEN)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, 0,
                        "unable to allocate Fletcher32 checksum destination buffer");

        dst = (unsigned char *)outbuf;

        
        H5MM_memcpy((void *)dst, (void *)(*buf), nbytes);

        
        dst += nbytes;
        UINT32ENCODE(dst, fletcher);

        
        H5MM_xfree(*buf);

        
        *buf_size = nbytes + FLETCHER_LEN;
        *buf      = outbuf;
        outbuf    = NULL;
        ret_value = *buf_size;
    }

done:
    if (outbuf)
        H5MM_xfree(outbuf);
    FUNC_LEAVE_NOAPI(ret_value)
}
