% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{pbeta_}
\alias{pbeta_}
\title{CDF of the Beta Distribution (gamma, delta+1 Parameterization)}
\usage{
pbeta_(q, gamma, delta, lower_tail = TRUE, log_p = FALSE)
}
\arguments{
\item{q}{Vector of quantiles (values generally between 0 and 1).}

\item{gamma}{First shape parameter (\code{shape1}), \eqn{\gamma > 0}. Can be a
scalar or a vector. Default: 1.0.}

\item{delta}{Second shape parameter is \code{delta + 1} (\code{shape2}), requires
\eqn{\delta \ge 0} so that \code{shape2 >= 1}. Can be a scalar or a vector.
Default: 0.0 (leading to \code{shape2 = 1}).}

\item{lower_tail}{Logical; if \code{TRUE} (default), probabilities are
\eqn{P(X \le q)}, otherwise, \eqn{P(X > q)}.}

\item{log_p}{Logical; if \code{TRUE}, probabilities \eqn{p} are given as
\eqn{\log(p)}. Default: \code{FALSE}.}
}
\value{
A vector of probabilities, \eqn{F(q)}, or their logarithms/complements
depending on \code{lower_tail} and \code{log_p}. The length of the result
is determined by the recycling rule applied to the arguments (\code{q},
\code{gamma}, \code{delta}). Returns \code{0} (or \code{-Inf} if
\code{log_p = TRUE}) for \code{q <= 0} and \code{1} (or \code{0} if
\code{log_p = TRUE}) for \code{q >= 1}. Returns \code{NaN} for invalid
parameters.
}
\description{
Computes the cumulative distribution function (CDF), \eqn{F(q) = P(X \le q)},
for the standard Beta distribution, using a parameterization common in
generalized distribution families. The distribution is parameterized by
\code{gamma} (\eqn{\gamma}) and \code{delta} (\eqn{\delta}), corresponding to
the standard Beta distribution with shape parameters \code{shape1 = gamma}
and \code{shape2 = delta + 1}.
}
\details{
This function computes the CDF of a Beta distribution with parameters
\code{shape1 = gamma} and \code{shape2 = delta + 1}. It is equivalent to
calling \code{stats::pbeta(q, shape1 = gamma, shape2 = delta + 1,
lower.tail = lower_tail, log.p = log_p)}.

This distribution arises as a special case of the five-parameter
Generalized Kumaraswamy (GKw) distribution (\code{\link{pgkw}}) obtained
by setting \eqn{\alpha = 1}, \eqn{\beta = 1}, and \eqn{\lambda = 1}.
It is therefore also equivalent to the McDonald (Mc)/Beta Power distribution
(\code{\link{pmc}}) with \eqn{\lambda = 1}.

The function likely calls R's underlying \code{pbeta} function but ensures
consistent parameter recycling and handling within the C++ environment,
matching the style of other functions in the related families.
}
\examples{
\donttest{
# Example values
q_vals <- c(0.2, 0.5, 0.8)
gamma_par <- 2.0 # Corresponds to shape1
delta_par <- 3.0 # Corresponds to shape2 - 1
shape1 <- gamma_par
shape2 <- delta_par + 1

# Calculate CDF using pbeta_
probs <- pbeta_(q_vals, gamma_par, delta_par)
print(probs)

# Compare with stats::pbeta
probs_stats <- stats::pbeta(q_vals, shape1 = shape1, shape2 = shape2)
print(paste("Max difference vs stats::pbeta:", max(abs(probs - probs_stats))))

# Compare with pgkw setting alpha=1, beta=1, lambda=1
probs_gkw <- pgkw(q_vals, alpha = 1.0, beta = 1.0, gamma = gamma_par,
                  delta = delta_par, lambda = 1.0)
print(paste("Max difference vs pgkw:", max(abs(probs - probs_gkw))))

# Compare with pmc setting lambda=1
probs_mc <- pmc(q_vals, gamma = gamma_par, delta = delta_par, lambda = 1.0)
print(paste("Max difference vs pmc:", max(abs(probs - probs_mc))))

# Calculate upper tail P(X > q)
probs_upper <- pbeta_(q_vals, gamma_par, delta_par, lower_tail = FALSE)
print(probs_upper)
print(stats::pbeta(q_vals, shape1, shape2, lower.tail = FALSE))

# Calculate log CDF
log_probs <- pbeta_(q_vals, gamma_par, delta_par, log_p = TRUE)
print(log_probs)
print(stats::pbeta(q_vals, shape1, shape2, log.p = TRUE))

# Plot the CDF
curve_q <- seq(0.001, 0.999, length.out = 200)
curve_p <- pbeta_(curve_q, gamma = 2, delta = 3) # Beta(2, 4)
plot(curve_q, curve_p, type = "l", main = "Beta(2, 4) CDF via pbeta_",
     xlab = "q", ylab = "F(q)", col = "blue")
curve(stats::pbeta(x, 2, 4), add=TRUE, col="red", lty=2)
legend("bottomright", legend=c("pbeta_(gamma=2, delta=3)", "stats::pbeta(shape1=2, shape2=4)"),
       col=c("blue", "red"), lty=c(1,2), bty="n")

}

}
\references{
Johnson, N. L., Kotz, S., & Balakrishnan, N. (1995). \emph{Continuous Univariate
Distributions, Volume 2} (2nd ed.). Wiley.

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},
}
\seealso{
\code{\link[stats]{pbeta}} (standard R implementation),
\code{\link{pgkw}} (parent distribution CDF),
\code{\link{pmc}} (McDonald/Beta Power CDF),
\code{dbeta_}, \code{qbeta_}, \code{rbeta_} (other functions for this parameterization, if they exist).
}
\author{
Lopes, J. E.
}
\keyword{beta}
\keyword{cumulative}
\keyword{distribution}
