% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wMix.R
\name{wMix}
\alias{wMix}
\title{Construct a weighted mixture object}
\usage{
wMix(
  f1,
  f2,
  w,
  f1.precompute = function(x, ...) {     x },
  spec = "ff",
  level = 2,
  c.int = NULL,
  c.level = 2,
  c.init = NULL,
  c.link = rep("identity", length(c.init)),
  c.link.params = rep(list(NA), length(c.init)),
  c.optim.control = list(maxit = 5000, method = "BFGS"),
  ncores = 1,
  quadError = TRUE,
  ...
)
}
\arguments{
\item{f1}{evaluates \eqn{f(\theta_1 | X, \theta_2)}.  \code{f1} must be able
to be called via \code{f1(theta1, params, log, ...)}.
\describe{
  \item{\code{theta1}}{a matrix of parameters at which to evaluate
    \eqn{f(\theta_1 | X, \theta_2)}. each row should be one set of values
    at which the density should be evaluated}
  \item{params}{a vector of parameters needed to evaluate
    \eqn{f(\theta_1 | X, \theta_2)}.  In most cases \code{params} will
    equal \eqn{theta_2}, but in some cases, \eqn{f(\theta_1 | X, \theta_2)}
    depends on functions of \eqn{\theta_2}, which can be pre-evaluated
    as the weighted mixture approximation is constructed.}
  \item{log}{TRUE to return \eqn{ln(f(\theta_1 | X, \theta_2))}}
  \item{...}{additional arguments needed for function evaluation}
}}

\item{f2}{evaluates \eqn{f(theta_2 | X)}.  \code{f2} must be able to be
called via \code{f2(theta2, log, ...)}.}

\item{w}{\code{wBuild} object created by \code{wBuild} function.  \code{w} 
contains posterior mode of \eqn{f(\theta_2| X)} and wrapper functions to 
generate quadrature grid.}

\item{f1.precompute}{function that pre-computes parameters for evaluating
\eqn{f(\theta_1 | X, \theta_2)}.  \code{f1.precompute} must be able to
be called via \code{f1.precompute(theta2, ...)} and return the argument
\code{params} for the function \code{f1}.}

\item{spec}{Specification of whether \code{f1} and \code{f2} are known 
exactly, or need numerical approximation to determine integration constants.
\code{'ff'} if both functions are known, \code{'gg'} if \code{f1} is 
proportional to the full conditional distribution 
\eqn{f(\theta_1|\theta_2,X)}, but needs the integration constant computed,
and if the marginal posterior \eqn{f(theta_2|X)} is equal to \code{f2} times
the integration constant that needs to be numerically approximated.}

\item{level}{accuracy level of the numerical approximation
(typically number of grid points for the
underlying 1D quadrature rule) [description from mvQuad::createNIGrid]}

\item{c.int}{If \code{spec=='gg'}, then \code{c.int} specifies the function
that can be integrated in order to yield the missing integration constant.}

\item{c.level}{accuracy level of the numerical approximation for \code{c.int} 
(typically number of grid points for the
underlying 1D quadrature rule) [description from mvQuad::createNIGrid]}

\item{c.init}{initial guess for mode of \code{c.int}.}

\item{c.link}{character vector that specifies transformations used during
optimization and integration of \code{c.int}.  See corresponding 
documentation in \code{wBuild} function for more details.}

\item{c.link.params}{Optional list of additional parameters for link
functions used with \code{c.int}.   See corresponding 
documentation in \code{wBuild} function for more details.}

\item{c.optim.control}{Arguments used to find mode of \code{c.int}.   See 
corresponding  documentation in \code{wBuild} function for more details.}

\item{ncores}{number of cores used to parallelize computation of parameters
for \eqn{f(\theta_1 | \theta_2, X)}.}

\item{quadError}{TRUE if integration nodes and weight should be computed for
the \code{level-1} integration grid, so that quadrature approximation
error can be estimated.}

\item{...}{Additional arguments to pass to \code{f1}, \code{f1.precompute},
\code{f12}, and \code{f2}.}
}
\value{
A list with class \code{wMix}, which contains the following items.
  \describe{
    \item{\code{f}}{Function for evaluating the posterior density
     \eqn{f(\theta_1|X)}.  \code{f} is callable  via
     \code{f(theta1, log, ...)}.}
    \item{\code{mix}}{A matrix containing the pre-computed parameters for
      evaluating the mixture components \eqn{f(\theta_1 | \theta_2, X)}.
      Each row of the matrix contains parameters for one of the \eqn{K}
      mixture components.}
    \item{\code{wts}}{Integration weights for each of the mixture components.
      Some of the weights may be negative.}
    \item{\code{expectation}}{List containing additional tools for computing
      posterior expectations of \eqn{f(\theta_2|X)}.  However, posterior
      expectations of \eqn{f(\theta_1|X)} can also be computed when
      expectations of \eqn{f(\theta_1|\theta_2, X)} are known.  The elements
      of \code{expectation} are
      \describe{
        \item{\code{Eh}}{Function to compute \eqn{E[h(\theta_2)|X]}.
          \code{Eh} is callable via \code{Eh(h, ...)}, where \code{h} is a
          function callable via \code{h(theta2, ...)} and \code{...} are
          additional arguments to the function.  The function \code{h} is
          evaluated at the quadrature nodes \eqn{\theta_2^{(j)}}.}
        \item{\code{Eh.precompute}}{Exactly the same idea as \code{Eh}, but
          the function \code{h} is evalauted at the quadrature nodes after
          being passed through the function \code{f1.precompute}.}
        \item{\code{grid}}{The sparse-quadrature integration grid used.
          Helpful for seeing the quadrature nodes \eqn{\theta_2^{(j)}}.}
        \item{\code{wts}}{The integration weights for approximating the
          expectation \eqn{E[h]}.  Note that these integration weights may
          differ from the main integration weights for evaluating the
          posterior density \eqn{f(\theta_1|X)}.}
      }}
  }
}
\description{
For a Bayesian model
\deqn{ X ~ f(X | \theta_1, \theta_2)}
\deqn{ (\theta_1, \theta_2) ~ f(\theta_1, \theta_2),}
the marginal  posterior \eqn{f(\theta_1 | X)} distribution can be
approximated via weighted mixtures via
\deqn{ f(\theta_1 | X) \approx \sum_{j=1}^K f(\theta_1 | X, \theta_2) w_j }
where \eqn{w_j} is based on \eqn{f(\theta_2^{(j)} | X)} and weights
\eqn{\tilde w_j}, where \eqn{\theta_2^{(j)}} and \eqn{\tilde w_j} are
nodes and weights for a sparse-grid quadrature integration scheme.
The quadrature rule is developed by finding the posterior mode of
\eqn{f(\theta_2|X)}, after transforming \eqn{\theta_2} to an unconstrained
support.  For best results, \eqn{\theta_2} should be a continuous random
variable, or be able to be approximated by one.
}
\examples{
# Use BISQuE to approximate the marginal posterior distribution for unknown
# population f(N|c, r) for the fur seals capture-recapture data example in 
# Givens and Hoeting (2013), example 7.10.

data('furseals')

# define theta transformation and jacobian
tx.theta = function(theta) { 
  c(log(theta[1]/theta[2]), log(sum(theta[1:2]))) 
}
itx.theta = function(u) { 
  c(exp(sum(u[1:2])), exp(u[2])) / (1 + exp(u[1])) 
}
lJ.tx.theta = function(u) {
  log(exp(u[1] + 2*u[2]) + exp(2*sum(u[1:2]))) - 3 * log(1 + exp(u[1]))
}

# compute constants
r = sum(furseals$m)
nC = nrow(furseals)

# set basic initialization for parameters
init = list(U = c(-.7, 5.5))
init = c(init, list(
  alpha = rep(.5, nC),
  theta = itx.theta(init$U),
  N = r + 1
))


post.alpha_theta = function(theta2, log = TRUE, ...) {
  # Function proportional to f(alpha, U1, U2 | c, r) 
  
  alpha = theta2[1:nC]
  u = theta2[-(1:nC)]
  theta = itx.theta(u)
  p = 1 - prod(1-alpha)
  
  res = - sum(theta)/1e3 - r * log(p) + lJ.tx.theta(u) - 
    nC * lbeta(theta[1], theta[2])
  for(i in 1:nC) {
    res = res + (theta[1] + furseals$c[i] - 1)*log(alpha[i]) + 
      (theta[2] + r - furseals$c[i] - 1)*log(1-alpha[i])
  }
  
  if(log) { res } else { exp(res) }
}

post.N.mixtures = function(N, params, log = TRUE, ...) {
  # The mixture component of the weighted mixtures for f(N | c, r)
  dnbinom(x = N-r, size = r, prob = params, log = log)
}

mixparams.N = function(theta2, ...) {
  # compute parameters for post.N.mixtures
  1 - prod(1 - theta2[1:nC])
}


w.N = wBuild(f = post.alpha_theta, init = c(init$alpha, init$U), 
             approx = 'gauss', link = c(rep('logit', nC), rep('identity', 2)))

m.N = wMix(f1 = post.N.mixtures, f1.precompute = mixparams.N, 
           f2 = post.alpha_theta, w = w.N)



# compute posterior mean
m.N$expectation$Eh.precompute(h = function(p) ((1-p)*r/p + r), 
                                   quadError = TRUE)

# compute posterior density
post.N.dens = data.frame(N = r:105)
post.N.dens$d = m.N$f(post.N.dens$N)

# plot posterior density
plot(d~N, post.N.dens, ylab = expression(f(N~'|'~bold(c),r)))

}
