% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/triangulate_matrix.R
\name{triangulate_matrix}
\alias{triangulate_matrix}
\title{Triangulate a Height Map}
\usage{
triangulate_matrix(
  heightmap,
  maxError = 1e-04,
  maxTriangles = 0,
  y_up = TRUE,
  start_index = 1,
  verbose = FALSE
)
}
\arguments{
\item{heightmap}{A two-dimensional matrix, where each entry in the matrix is the elevation at that point.
All points are assumed to be evenly spaced.}

\item{maxError}{Default `0.0001`. Maximum error allowed in triangulating the height map.}

\item{maxTriangles}{Default `0`, which turns off this setting (and only uses the `max_error` arg).
Otherwise, specifies the maximum number of triangles when triangulating the height map.}

\item{y_up}{Default `TRUE`. Which axis is "upwards" in the return matrix. If `FALSE`, `z` is up.}

\item{start_index}{Default `1`. The offset to the first `x` and `z` indices.}

\item{verbose}{Default `FALSE`. Prints reduction in number of triangles/max error.}
}
\value{
Returns a matrix of vertices and IDs for each triangle.
}
\description{
Uses Delaney triangulation to approximate a rectangular height field (in
matrix form) with constraints (either maximum allowable error, or a maximum number of triangles).
Increasing the error limit will result in a courser approximation, but fewer triangles in the model.
For many models (particularly those with large, flat regions or smooth changes in height),
this can result in significant reductions in model size with no perceptual loss in
terrain surface quality.
}
\examples{
#Let's triangulate the built-in `volcano` dataset.

#Helper function to plot polygons over an `image()` plot.
plot_polys = function(tri_matrix) {
  #reverse orienation for `image`
  tri_matrix[,3] = max(tri_matrix[,3])-tri_matrix[,3]+1
  for(i in seq_len(nrow(tri_matrix)/3)) {
    polypath(tri_matrix[(3*(i-1)+1):(3*i), c(1,3)])
  }
}

#Here, we don't accept any error, but still triangulate
tris = triangulate_matrix(volcano, maxError = 0, verbose = TRUE)
image(x=1:nrow(volcano), y = 1:ncol(volcano), volcano)
plot_polys(tris)

#Let's increase the allowable error:
tris = triangulate_matrix(volcano, maxError = 1, verbose = TRUE)
image(x=1:nrow(volcano), y = 1:ncol(volcano), volcano)
plot_polys(tris)

#Increase it again
tris = triangulate_matrix(volcano, maxError = 10, verbose = TRUE)
image(x=1:nrow(volcano), y = 1:ncol(volcano), volcano)
plot_polys(tris)

#Here, we set an allowable number of triangles instead, using exactly 20 triangles:
tris = triangulate_matrix(volcano, maxTriangles = 20, verbose = TRUE)
image(x=1:nrow(volcano), y = 1:ncol(volcano), volcano)
plot_polys(tris)

#The output of this function can be passed directly to `rgl::triangles3d()` for plotting in 3D.
}
