/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Ppkg.h"      
#include "H5SLprivate.h" 

typedef struct {
    const H5P_genclass_t *parent;    
    const char           *name;      
    H5P_genclass_t       *new_class; 
} H5P_check_class_t;

typedef struct {
    H5P_iterate_int_t     cb_func;      
    void                 *udata;        
    const H5P_genplist_t *plist;        
    H5SL_t               *seen;         
    int                  *curr_idx_ptr; 
    int                   prev_idx;     
} H5P_iter_plist_ud_t;

typedef struct {
    H5P_iterate_int_t cb_func;      
    void             *udata;        
    int              *curr_idx_ptr; 
    int               prev_idx;     
} H5P_iter_pclass_ud_t;

typedef struct {
    const H5P_genplist_t *plist2;    
    int                   cmp_value; 
} H5P_plist_cmp_ud_t;

typedef struct {
    const void *value; 
} H5P_prop_set_ud_t;

typedef struct {
    void *value; 
} H5P_prop_get_ud_t;

typedef herr_t (*H5P_do_plist_op_t)(H5P_genplist_t *plist, const char *name, H5P_genprop_t *prop,
                                    void *udata);
typedef herr_t (*H5P_do_pclass_op_t)(H5P_genplist_t *plist, const char *name, H5P_genprop_t *prop,
                                     void *udata);

static herr_t H5P__close_class_cb(void *space, void **request);
static herr_t H5P__close_list_cb(void *space, void **request);

static H5P_genplist_t *H5P__create(H5P_genclass_t *pclass);
static H5P_genprop_t  *H5P__create_prop(const char *name, size_t size, H5P_prop_within_t type,
                                        const void *value, H5P_prp_create_func_t prp_create,
                                        H5P_prp_set_func_t prp_set, H5P_prp_get_func_t prp_get,
                                        H5P_prp_encode_func_t prp_encode, H5P_prp_decode_func_t prp_decode,
                                        H5P_prp_delete_func_t prp_delete, H5P_prp_copy_func_t prp_copy,
                                        H5P_prp_compare_func_t prp_cmp, H5P_prp_close_func_t prp_close);
static H5P_genprop_t  *H5P__dup_prop(H5P_genprop_t *oprop, H5P_prop_within_t type);
static herr_t          H5P__free_prop(H5P_genprop_t *prop);
static int             H5P__cmp_prop(const H5P_genprop_t *prop1, const H5P_genprop_t *prop2);
static herr_t          H5P__do_prop(H5P_genplist_t *plist, const char *name, H5P_do_plist_op_t plist_op,
                                    H5P_do_pclass_op_t pclass_op, void *udata);
static int             H5P__open_class_path_cb(void *_obj, hid_t H5_ATTR_UNUSED id, void *_key);
static H5P_genprop_t  *H5P__find_prop_pclass(H5P_genclass_t *pclass, const char *name);
static herr_t          H5P__free_prop_cb(void *item, void H5_ATTR_UNUSED *key, void *op_data);
static herr_t H5P__free_del_name_cb(void *item, void H5_ATTR_UNUSED *key, void H5_ATTR_UNUSED *op_data);

hid_t           H5P_CLS_ROOT_ID_g = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_ROOT_g    = NULL;

hid_t           H5P_CLS_ATTRIBUTE_ACCESS_ID_g = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_ATTRIBUTE_ACCESS_g    = NULL;
hid_t           H5P_CLS_ATTRIBUTE_CREATE_ID_g = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_ATTRIBUTE_CREATE_g    = NULL;
hid_t           H5P_CLS_DATASET_ACCESS_ID_g   = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_DATASET_ACCESS_g      = NULL;
hid_t           H5P_CLS_DATASET_CREATE_ID_g   = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_DATASET_CREATE_g      = NULL;
hid_t           H5P_CLS_DATASET_XFER_ID_g     = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_DATASET_XFER_g        = NULL;
hid_t           H5P_CLS_DATATYPE_ACCESS_ID_g  = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_DATATYPE_ACCESS_g     = NULL;
hid_t           H5P_CLS_DATATYPE_CREATE_ID_g  = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_DATATYPE_CREATE_g     = NULL;
hid_t           H5P_CLS_FILE_ACCESS_ID_g      = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_FILE_ACCESS_g         = NULL;
hid_t           H5P_CLS_FILE_CREATE_ID_g      = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_FILE_CREATE_g         = NULL;
hid_t           H5P_CLS_FILE_MOUNT_ID_g       = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_FILE_MOUNT_g          = NULL;
hid_t           H5P_CLS_GROUP_ACCESS_ID_g     = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_GROUP_ACCESS_g        = NULL;
hid_t           H5P_CLS_GROUP_CREATE_ID_g     = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_GROUP_CREATE_g        = NULL;
hid_t           H5P_CLS_LINK_ACCESS_ID_g      = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_LINK_ACCESS_g         = NULL;
hid_t           H5P_CLS_LINK_CREATE_ID_g      = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_LINK_CREATE_g         = NULL;
hid_t           H5P_CLS_MAP_ACCESS_ID_g       = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_MAP_ACCESS_g          = NULL;
hid_t           H5P_CLS_MAP_CREATE_ID_g       = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_MAP_CREATE_g          = NULL;
hid_t           H5P_CLS_OBJECT_COPY_ID_g      = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_OBJECT_COPY_g         = NULL;
hid_t           H5P_CLS_OBJECT_CREATE_ID_g    = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_OBJECT_CREATE_g       = NULL;
hid_t           H5P_CLS_REFERENCE_ACCESS_ID_g = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_REFERENCE_ACCESS_g    = NULL;
hid_t           H5P_CLS_STRING_CREATE_ID_g    = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_STRING_CREATE_g       = NULL;
hid_t           H5P_CLS_VOL_INITIALIZE_ID_g   = H5I_INVALID_HID;
H5P_genclass_t *H5P_CLS_VOL_INITIALIZE_g      = NULL;

hid_t H5P_LST_ATTRIBUTE_ACCESS_ID_g = H5I_INVALID_HID;
hid_t H5P_LST_ATTRIBUTE_CREATE_ID_g = H5I_INVALID_HID;
hid_t H5P_LST_DATASET_ACCESS_ID_g   = H5I_INVALID_HID;
hid_t H5P_LST_DATASET_CREATE_ID_g   = H5I_INVALID_HID;
hid_t H5P_LST_DATASET_XFER_ID_g     = H5I_INVALID_HID;
hid_t H5P_LST_DATATYPE_ACCESS_ID_g  = H5I_INVALID_HID;
hid_t H5P_LST_DATATYPE_CREATE_ID_g  = H5I_INVALID_HID;
hid_t H5P_LST_FILE_ACCESS_ID_g      = H5I_INVALID_HID;
hid_t H5P_LST_FILE_CREATE_ID_g      = H5I_INVALID_HID;
hid_t H5P_LST_FILE_MOUNT_ID_g       = H5I_INVALID_HID;
hid_t H5P_LST_GROUP_ACCESS_ID_g     = H5I_INVALID_HID;
hid_t H5P_LST_GROUP_CREATE_ID_g     = H5I_INVALID_HID;
hid_t H5P_LST_LINK_ACCESS_ID_g      = H5I_INVALID_HID;
hid_t H5P_LST_LINK_CREATE_ID_g      = H5I_INVALID_HID;
hid_t H5P_LST_MAP_ACCESS_ID_g       = H5I_INVALID_HID;
hid_t H5P_LST_MAP_CREATE_ID_g       = H5I_INVALID_HID;
hid_t H5P_LST_OBJECT_COPY_ID_g      = H5I_INVALID_HID;
hid_t H5P_LST_REFERENCE_ACCESS_ID_g = H5I_INVALID_HID;
hid_t H5P_LST_VOL_INITIALIZE_ID_g   = H5I_INVALID_HID;

const H5P_libclass_t H5P_CLS_ROOT[1] = {{
    "root",        
    H5P_TYPE_ROOT, 

    NULL,               
    &H5P_CLS_ROOT_g,    
    &H5P_CLS_ROOT_ID_g, 
    NULL,               
    NULL,               

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

const H5P_libclass_t H5P_CLS_AACC[1] = {{
    "attribute access",        
    H5P_TYPE_ATTRIBUTE_ACCESS, 

    &H5P_CLS_LINK_ACCESS_g,         
    &H5P_CLS_ATTRIBUTE_ACCESS_g,    
    &H5P_CLS_ATTRIBUTE_ACCESS_ID_g, 
    &H5P_LST_ATTRIBUTE_ACCESS_ID_g, 
    NULL,                           

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

const H5P_libclass_t H5P_CLS_GACC[1] = {{
    "group access",        
    H5P_TYPE_GROUP_ACCESS, 

    &H5P_CLS_LINK_ACCESS_g,     
    &H5P_CLS_GROUP_ACCESS_g,    
    &H5P_CLS_GROUP_ACCESS_ID_g, 
    &H5P_LST_GROUP_ACCESS_ID_g, 
    NULL,                       

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

const H5P_libclass_t H5P_CLS_TCRT[1] = {{
    "datatype create",        
    H5P_TYPE_DATATYPE_CREATE, 

    &H5P_CLS_OBJECT_CREATE_g,      
    &H5P_CLS_DATATYPE_CREATE_g,    
    &H5P_CLS_DATATYPE_CREATE_ID_g, 
    &H5P_LST_DATATYPE_CREATE_ID_g, 
    NULL,                          

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

const H5P_libclass_t H5P_CLS_TACC[1] = {{
    "datatype access",        
    H5P_TYPE_DATATYPE_ACCESS, 

    &H5P_CLS_LINK_ACCESS_g,        
    &H5P_CLS_DATATYPE_ACCESS_g,    
    &H5P_CLS_DATATYPE_ACCESS_ID_g, 
    &H5P_LST_DATATYPE_ACCESS_ID_g, 
    NULL,                          

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

const H5P_libclass_t H5P_CLS_VINI[1] = {{
    "VOL initialization",    
    H5P_TYPE_VOL_INITIALIZE, 

    &H5P_CLS_ROOT_g,              
    &H5P_CLS_VOL_INITIALIZE_g,    
    &H5P_CLS_VOL_INITIALIZE_ID_g, 
    &H5P_LST_VOL_INITIALIZE_ID_g, 
    NULL,                         

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

const H5P_libclass_t H5P_CLS_RACC[1] = {{
    "reference access",        
    H5P_TYPE_REFERENCE_ACCESS, 

    &H5P_CLS_FILE_ACCESS_g,         
    &H5P_CLS_REFERENCE_ACCESS_g,    
    &H5P_CLS_REFERENCE_ACCESS_ID_g, 
    &H5P_LST_REFERENCE_ACCESS_ID_g, 
    NULL,                           

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

H5_DLLVAR const H5P_libclass_t H5P_CLS_OCRT[1];   
H5_DLLVAR const H5P_libclass_t H5P_CLS_STRCRT[1]; 
H5_DLLVAR const H5P_libclass_t H5P_CLS_GCRT[1];   
H5_DLLVAR const H5P_libclass_t H5P_CLS_FCRT[1];   
H5_DLLVAR const H5P_libclass_t H5P_CLS_DCRT[1];   
H5_DLLVAR const H5P_libclass_t H5P_CLS_MCRT[1];   
H5_DLLVAR const H5P_libclass_t H5P_CLS_DXFR[1];   
H5_DLLVAR const H5P_libclass_t H5P_CLS_FMNT[1];   
H5_DLLVAR const H5P_libclass_t H5P_CLS_ACRT[1];   

static unsigned H5P_next_rev = 0;
#define H5P_GET_NEXT_REV (H5P_next_rev++)

static H5P_libclass_t const *const init_class[] = {
    H5P_CLS_ROOT,   
    H5P_CLS_OCRT,   
    H5P_CLS_STRCRT, 
    H5P_CLS_LACC,   
    H5P_CLS_GCRT,   
    H5P_CLS_OCPY,   
    H5P_CLS_GACC,   
    H5P_CLS_FCRT,   
    H5P_CLS_FACC,   
    H5P_CLS_DCRT,   
    H5P_CLS_DACC,   
    H5P_CLS_DXFR,   
    H5P_CLS_FMNT,   
    H5P_CLS_TCRT,   
    H5P_CLS_TACC,   
    H5P_CLS_MCRT,   
    H5P_CLS_MACC,   
    H5P_CLS_ACRT,   
    H5P_CLS_AACC,   
    H5P_CLS_LCRT,   
    H5P_CLS_VINI,   
    H5P_CLS_RACC    
};

H5FL_DEFINE_STATIC(H5P_genclass_t);

H5FL_DEFINE_STATIC(H5P_genprop_t);

H5FL_DEFINE_STATIC(H5P_genplist_t);

static const H5I_class_t H5I_GENPROPCLS_CLS[1] = {{
    H5I_GENPROP_CLS,                
    0,                              
    0,                              
    (H5I_free_t)H5P__close_class_cb 
}};

static const H5I_class_t H5I_GENPROPLST_CLS[1] = {{
    H5I_GENPROP_LST,               
    0,                             
    0,                             
    (H5I_free_t)H5P__close_list_cb 
}};

herr_t
H5P_init_phase1(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)
    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P_init_phase2(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5P__facc_set_def_driver() < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set default VFL driver");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__init_package(void)
{
    size_t tot_init = 0; 
    size_t pass_init;    
    size_t u;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    HDcompile_assert(H5P_TYPE_REFERENCE_ACCESS == (H5P_TYPE_MAX_TYPE - 1));

    
    if (H5I_register_type(H5I_GENPROPCLS_CLS) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTINIT, FAIL, "unable to initialize ID group");
    if (H5I_register_type(H5I_GENPROPLST_CLS) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTINIT, FAIL, "unable to initialize ID group");

    
    tot_init = 0;
    do {
        
        pass_init = 0;

        
        for (u = 0; u < NELMTS(init_class); u++) {
            H5P_libclass_t const *lib_class = init_class[u]; 

            
            assert(lib_class->class_id);
            if (*lib_class->class_id == (-1) &&
                (lib_class->par_pclass == NULL || *lib_class->par_pclass != NULL)) {
                
                assert(lib_class->par_pclass || lib_class == H5P_CLS_ROOT);

                
                if (NULL == (*lib_class->pclass = H5P__create_class(
                                 lib_class->par_pclass ? *lib_class->par_pclass : NULL, lib_class->name,
                                 lib_class->type, lib_class->create_func, lib_class->create_data,
                                 lib_class->copy_func, lib_class->copy_data, lib_class->close_func,
                                 lib_class->close_data)))
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "class initialization failed");

                
                if (lib_class->reg_prop_func && (*lib_class->reg_prop_func)(*lib_class->pclass) < 0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "can't register properties");

                
                if ((*lib_class->class_id = H5I_register(H5I_GENPROP_CLS, *lib_class->pclass, false)) < 0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "can't register property list class");

                
                if (lib_class->def_plist_id && *lib_class->def_plist_id == (-1)) {
                    
                    if ((*lib_class->def_plist_id = H5P_create_id(*lib_class->pclass, false)) < 0)
                        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL,
                                    "can't register default property list for class");
                } 

                
                pass_init++;
                tot_init++;
            } 
        }     
    } while (pass_init > 0);

    
    assert(tot_init == NELMTS(init_class));

done:
    if (ret_value < 0 && tot_init > 0) {
        
        H5I_clear_type(H5I_GENPROP_LST, false, false);

        
        for (u = 0; u < NELMTS(init_class); u++) {
            H5P_libclass_t const *lib_class = init_class[u]; 

            assert(lib_class->class_id);
            if (*lib_class->class_id >= 0) {
                
                if (H5I_dec_ref(*lib_class->class_id) < 0)
                    HDONE_ERROR(H5E_PLIST, H5E_CLOSEERROR, FAIL, "unable to close property list class ID");
            }
            else if (lib_class->pclass && *lib_class->pclass) {
                
                if (H5P__close_class(*lib_class->pclass) < 0)
                    HDONE_ERROR(H5E_PLIST, H5E_CLOSEERROR, FAIL, "unable to close property list class");
            }
        }
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5P_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        int64_t nlist, nclass;

        

        
        nclass = H5I_nmembers(H5I_GENPROP_CLS);
        nlist  = H5I_nmembers(H5I_GENPROP_LST);

        
        if ((nclass + nlist) > 0) {
            
            if (nlist > 0) {
                (void)H5I_clear_type(H5I_GENPROP_LST, false, false);

                
                if (H5I_nmembers(H5I_GENPROP_LST) == 0) {
                    H5P_LST_ATTRIBUTE_ACCESS_ID_g = H5I_INVALID_HID;
                    H5P_LST_ATTRIBUTE_CREATE_ID_g = H5I_INVALID_HID;
                    H5P_LST_DATASET_ACCESS_ID_g   = H5I_INVALID_HID;
                    H5P_LST_DATASET_CREATE_ID_g   = H5I_INVALID_HID;
                    H5P_LST_DATASET_XFER_ID_g     = H5I_INVALID_HID;
                    H5P_LST_DATATYPE_ACCESS_ID_g  = H5I_INVALID_HID;
                    H5P_LST_DATATYPE_CREATE_ID_g  = H5I_INVALID_HID;
                    H5P_LST_FILE_ACCESS_ID_g      = H5I_INVALID_HID;
                    H5P_LST_FILE_CREATE_ID_g      = H5I_INVALID_HID;
                    H5P_LST_FILE_MOUNT_ID_g       = H5I_INVALID_HID;
                    H5P_LST_GROUP_ACCESS_ID_g     = H5I_INVALID_HID;
                    H5P_LST_GROUP_CREATE_ID_g     = H5I_INVALID_HID;
                    H5P_LST_LINK_ACCESS_ID_g      = H5I_INVALID_HID;
                    H5P_LST_LINK_CREATE_ID_g      = H5I_INVALID_HID;
                    H5P_LST_MAP_ACCESS_ID_g       = H5I_INVALID_HID;
                    H5P_LST_MAP_CREATE_ID_g       = H5I_INVALID_HID;
                    H5P_LST_OBJECT_COPY_ID_g      = H5I_INVALID_HID;
                    H5P_LST_REFERENCE_ACCESS_ID_g = H5I_INVALID_HID;
                    H5P_LST_VOL_INITIALIZE_ID_g   = H5I_INVALID_HID;
                }
            }

            
            if (nlist == 0 && nclass > 0) {
                (void)H5I_clear_type(H5I_GENPROP_CLS, false, false);

                
                if (H5I_nmembers(H5I_GENPROP_CLS) == 0) {
                    H5P_CLS_ROOT_g = NULL;

                    H5P_CLS_ATTRIBUTE_ACCESS_g = NULL;
                    H5P_CLS_ATTRIBUTE_CREATE_g = NULL;
                    H5P_CLS_DATASET_ACCESS_g   = NULL;
                    H5P_CLS_DATASET_CREATE_g   = NULL;
                    H5P_CLS_DATASET_XFER_g     = NULL;
                    H5P_CLS_DATATYPE_ACCESS_g  = NULL;
                    H5P_CLS_DATATYPE_CREATE_g  = NULL;
                    H5P_CLS_FILE_ACCESS_g      = NULL;
                    H5P_CLS_FILE_CREATE_g      = NULL;
                    H5P_CLS_FILE_MOUNT_g       = NULL;
                    H5P_CLS_GROUP_ACCESS_g     = NULL;
                    H5P_CLS_GROUP_CREATE_g     = NULL;
                    H5P_CLS_LINK_ACCESS_g      = NULL;
                    H5P_CLS_LINK_CREATE_g      = NULL;
                    H5P_CLS_MAP_ACCESS_g       = NULL;
                    H5P_CLS_MAP_CREATE_g       = NULL;
                    H5P_CLS_OBJECT_COPY_g      = NULL;
                    H5P_CLS_OBJECT_CREATE_g    = NULL;
                    H5P_CLS_REFERENCE_ACCESS_g = NULL;
                    H5P_CLS_STRING_CREATE_g    = NULL;
                    H5P_CLS_VOL_INITIALIZE_g   = NULL;

                    H5P_CLS_ROOT_ID_g = H5I_INVALID_HID;

                    H5P_CLS_ATTRIBUTE_ACCESS_ID_g = H5I_INVALID_HID;
                    H5P_CLS_ATTRIBUTE_CREATE_ID_g = H5I_INVALID_HID;
                    H5P_CLS_DATASET_ACCESS_ID_g   = H5I_INVALID_HID;
                    H5P_CLS_DATASET_CREATE_ID_g   = H5I_INVALID_HID;
                    H5P_CLS_DATASET_XFER_ID_g     = H5I_INVALID_HID;
                    H5P_CLS_DATATYPE_ACCESS_ID_g  = H5I_INVALID_HID;
                    H5P_CLS_DATATYPE_CREATE_ID_g  = H5I_INVALID_HID;
                    H5P_CLS_FILE_ACCESS_ID_g      = H5I_INVALID_HID;
                    H5P_CLS_FILE_CREATE_ID_g      = H5I_INVALID_HID;
                    H5P_CLS_FILE_MOUNT_ID_g       = H5I_INVALID_HID;
                    H5P_CLS_GROUP_ACCESS_ID_g     = H5I_INVALID_HID;
                    H5P_CLS_GROUP_CREATE_ID_g     = H5I_INVALID_HID;
                    H5P_CLS_LINK_ACCESS_ID_g      = H5I_INVALID_HID;
                    H5P_CLS_LINK_CREATE_ID_g      = H5I_INVALID_HID;
                    H5P_CLS_MAP_ACCESS_ID_g       = H5I_INVALID_HID;
                    H5P_CLS_MAP_CREATE_ID_g       = H5I_INVALID_HID;
                    H5P_CLS_OBJECT_COPY_ID_g      = H5I_INVALID_HID;
                    H5P_CLS_OBJECT_CREATE_ID_g    = H5I_INVALID_HID;
                    H5P_CLS_REFERENCE_ACCESS_ID_g = H5I_INVALID_HID;
                    H5P_CLS_STRING_CREATE_ID_g    = H5I_INVALID_HID;
                    H5P_CLS_VOL_INITIALIZE_ID_g   = H5I_INVALID_HID;
                }
            }

            n++; 
        }
        else {
            
            n += (H5I_dec_type_ref(H5I_GENPROP_LST) > 0);
            n += (H5I_dec_type_ref(H5I_GENPROP_CLS) > 0);

            
            if (0 == n)
                H5_PKG_INIT_VAR = false;
        } 
    }     

    FUNC_LEAVE_NOAPI(n)
} 

static herr_t
H5P__close_class_cb(void *_pclass, void H5_ATTR_UNUSED **request)
{
    H5P_genclass_t *pclass    = (H5P_genclass_t *)_pclass; 
    herr_t          ret_value = SUCCEED;                   

    FUNC_ENTER_PACKAGE

    
    assert(pclass);

    
    if (H5P__close_class(pclass) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CLOSEERROR, FAIL, "unable to close property list class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__close_list_cb(void *_plist, void H5_ATTR_UNUSED **request)
{
    H5P_genplist_t *plist     = (H5P_genplist_t *)_plist; 
    herr_t          ret_value = SUCCEED;                  

    FUNC_ENTER_PACKAGE

    
    assert(plist);

    
    if (H5P_close(plist) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CLOSEERROR, FAIL, "unable to close property list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__do_prop_cb1(H5SL_t *slist, H5P_genprop_t *prop, H5P_prp_cb1_t cb)
{
    void          *tmp_value = NULL;    
    H5P_genprop_t *pcopy     = NULL;    
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(slist);
    assert(prop);
    assert(prop->cmp);
    assert(cb);

    
    if (NULL == (tmp_value = H5MM_malloc(prop->size)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed for temporary property value");
    H5MM_memcpy(tmp_value, prop->value, prop->size);

    
    H5_BEFORE_USER_CB(FAIL)
        {
            
            ret_value = cb(prop->name, prop->size, tmp_value);
        }
    H5_AFTER_USER_CB(FAIL)
    if (ret_value < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "Property callback failed");

    
    if (NULL == (pcopy = H5P__dup_prop(prop, H5P_PROP_WITHIN_LIST)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "Can't copy property");

    
    H5MM_memcpy(pcopy->value, tmp_value, prop->size);

    
    if (H5P__add_prop(slist, pcopy) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "Can't insert property into skip list");

done:
    
    if (tmp_value)
        H5MM_xfree(tmp_value);

    
    if (ret_value < 0)
        if (pcopy)
            H5P__free_prop(pcopy);

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5P_genclass_t *
H5P__copy_pclass(H5P_genclass_t *pclass)
{
    H5P_genclass_t *new_pclass = NULL; 
    H5P_genprop_t  *pcopy;             
    H5P_genclass_t *ret_value = NULL;  

    FUNC_ENTER_PACKAGE

    assert(pclass);

    

    
    if (NULL == (new_pclass = H5P__create_class(pclass->parent, pclass->name, pclass->type,
                                                pclass->create_func, pclass->create_data, pclass->copy_func,
                                                pclass->copy_data, pclass->close_func, pclass->close_data)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, NULL, "unable to create property list class");

    
    if (pclass->nprops > 0) {
        H5SL_node_t *curr_node; 

        
        curr_node = H5SL_first(pclass->props);
        while (curr_node != NULL) {
            
            if (NULL == (pcopy = H5P__dup_prop((H5P_genprop_t *)H5SL_item(curr_node), H5P_PROP_WITHIN_CLASS)))
                HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, NULL, "Can't copy property");

            
            if (H5P__add_prop(new_pclass->props, pcopy) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, NULL, "Can't insert property into class");

            
            new_pclass->nprops++;

            
            curr_node = H5SL_next(curr_node);
        } 
    }     

    
    ret_value = new_pclass;

done:
    if (NULL == ret_value && new_pclass)
        H5P__close_class(new_pclass);

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5P_copy_plist(const H5P_genplist_t *old_plist, bool app_ref)
{
    H5P_genclass_t *tclass;           
    H5P_genplist_t *new_plist = NULL; 
    H5P_genprop_t  *tmp;              
    H5P_genprop_t  *new_prop;         
    hid_t           new_plist_id;     
    H5SL_node_t    *curr_node;        
    H5SL_t         *seen = NULL;      
    size_t          nseen;            
    bool            has_parent_class; 
    hid_t           ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    assert(old_plist);

    

    
    if (NULL == (new_plist = H5FL_CALLOC(H5P_genplist_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, H5I_INVALID_HID, "memory allocation failed");

    
    new_plist->pclass     = old_plist->pclass;
    new_plist->nprops     = 0;     
    new_plist->class_init = false; 

    
    if ((new_plist->props = H5SL_create(H5SL_TYPE_STR, NULL)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, H5I_INVALID_HID,
                    "can't create skip list for changed properties");

    
    if ((new_plist->del = H5SL_create(H5SL_TYPE_STR, NULL)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, H5I_INVALID_HID,
                    "can't create skip list for deleted properties");

    
    if ((seen = H5SL_create(H5SL_TYPE_STR, NULL)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, H5I_INVALID_HID, "can't create skip list for seen properties");
    nseen = 0;

    
    if (H5SL_count(old_plist->del) > 0) {
        curr_node = H5SL_first(old_plist->del);
        while (curr_node) {
            char *new_name; 

            
            if ((new_name = H5MM_xstrdup((char *)H5SL_item(curr_node))) == NULL)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, H5I_INVALID_HID, "memory allocation failed");

            
            if (H5SL_insert(new_plist->del, new_name, new_name) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, H5I_INVALID_HID,
                            "can't insert property into deleted skip list");

            
            if (H5SL_insert(seen, new_name, new_name) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, H5I_INVALID_HID,
                            "can't insert property into seen skip list");
            nseen++;

            
            curr_node = H5SL_next(curr_node);
        } 
    }     

    
    if (H5SL_count(old_plist->props) > 0) {
        curr_node = H5SL_first(old_plist->props);
        while (curr_node) {
            
            tmp = (H5P_genprop_t *)H5SL_item(curr_node);

            
            if (NULL == (new_prop = H5P__dup_prop(tmp, H5P_PROP_WITHIN_LIST)))
                HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, H5I_INVALID_HID, "Can't copy property");

            
            if (new_prop->copy) {
                herr_t status;

                
                H5_BEFORE_USER_CB(H5I_INVALID_HID)
                    {
                        status = (new_prop->copy)(new_prop->name, new_prop->size, new_prop->value);
                    }
                H5_AFTER_USER_CB(H5I_INVALID_HID)
                if (status < 0) {
                    H5P__free_prop(new_prop);
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, H5I_INVALID_HID, "Can't copy property");
                } 
            }     

            
            if (H5P__add_prop(new_plist->props, new_prop) < 0) {
                H5P__free_prop(new_prop);
                HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, H5I_INVALID_HID, "Can't insert property into list");
            } 

            
            if (H5SL_insert(seen, new_prop->name, new_prop->name) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, H5I_INVALID_HID,
                            "can't insert property into seen skip list");
            nseen++;

            
            new_plist->nprops++;

            
            curr_node = H5SL_next(curr_node);
        } 
    }     

    
    tclass           = old_plist->pclass;
    has_parent_class = (bool)(tclass != NULL && tclass->parent != NULL && tclass->parent->nprops > 0);
    while (tclass != NULL) {
        if (tclass->nprops > 0) {
            
            curr_node = H5SL_first(tclass->props);
            while (curr_node != NULL) {
                
                tmp = (H5P_genprop_t *)H5SL_item(curr_node);

                
                if (nseen == 0 || H5SL_search(seen, tmp->name) == NULL) {
                    
                    if (tmp->copy) {
                        
                        if (H5P__do_prop_cb1(new_plist->props, tmp, tmp->copy) < 0)
                            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, H5I_INVALID_HID, "Can't create property");
                    } 

                    
                    if (has_parent_class) {
                        if (H5SL_insert(seen, tmp->name, tmp->name) < 0)
                            HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, H5I_INVALID_HID,
                                        "can't insert property into seen skip list");
                        nseen++;
                    } 

                    
                    new_plist->nprops++;
                } 

                
                curr_node = H5SL_next(curr_node);
            } 
        }     

        
        tclass = tclass->parent;
    } 

    
    if (H5P__access_class(new_plist->pclass, H5P_MOD_INC_LST) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, H5I_INVALID_HID, "Can't increment class ref count");

    
    if ((new_plist_id = H5I_register(H5I_GENPROP_LST, new_plist, app_ref)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register property list");

    
    new_plist->plist_id = new_plist_id;

    
    tclass = new_plist->pclass;
    while (NULL != tclass) {
        if (NULL != tclass->copy_func) {
            herr_t status;

            
            H5_BEFORE_USER_CB(H5I_INVALID_HID)
                {
                    status =
                        (tclass->copy_func)(new_plist_id, old_plist->plist_id, old_plist->pclass->copy_data);
                }
            H5_AFTER_USER_CB(H5I_INVALID_HID)
            if (status < 0) {
                
                H5I_remove(new_plist_id);
                HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, H5I_INVALID_HID, "Can't initialize property");
            } 
        }     

        
        tclass = tclass->parent;
    } 

    
    new_plist->class_init = true;

    
    ret_value = new_plist_id;

done:
    
    if (seen != NULL)
        H5SL_close(seen);

    if (H5I_INVALID_HID == ret_value && new_plist)
        H5P_close(new_plist);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5P_genprop_t *
H5P__dup_prop(H5P_genprop_t *oprop, H5P_prop_within_t type)
{
    H5P_genprop_t *prop      = NULL; 
    H5P_genprop_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(oprop);
    assert(type != H5P_PROP_WITHIN_UNKNOWN);

    
    if (NULL == (prop = H5FL_MALLOC(H5P_genprop_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    H5MM_memcpy(prop, oprop, sizeof(H5P_genprop_t));

    

    
    if (type == H5P_PROP_WITHIN_CLASS) {
        assert(oprop->type == H5P_PROP_WITHIN_CLASS);
        assert(oprop->shared_name == false);

        
        prop->name = H5MM_xstrdup(oprop->name);
    } 
    
    else {
        

        
        if (oprop->type == H5P_PROP_WITHIN_LIST) {
            
            if (!oprop->shared_name)
                prop->name = H5MM_xstrdup(oprop->name);
        } 
        
        else {
            assert(oprop->type == H5P_PROP_WITHIN_CLASS);
            assert(oprop->shared_name == false);

            
            prop->shared_name = true;

            
            prop->type = type;
        } 
    }     

    
    if (oprop->value != NULL) {
        assert(prop->size > 0);
        if (NULL == (prop->value = H5MM_malloc(prop->size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
        H5MM_memcpy(prop->value, oprop->value, prop->size);
    } 

    
    ret_value = prop;

done:
    
    if (ret_value == NULL) {
        if (prop != NULL) {
            if (prop->name != NULL)
                H5MM_xfree(prop->name);
            if (prop->value != NULL)
                H5MM_xfree(prop->value);
            prop = H5FL_FREE(H5P_genprop_t, prop);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5P_genprop_t *
H5P__create_prop(const char *name, size_t size, H5P_prop_within_t type, const void *value,
                 H5P_prp_create_func_t prp_create, H5P_prp_set_func_t prp_set, H5P_prp_get_func_t prp_get,
                 H5P_prp_encode_func_t prp_encode, H5P_prp_decode_func_t prp_decode,
                 H5P_prp_delete_func_t prp_delete, H5P_prp_copy_func_t prp_copy,
                 H5P_prp_compare_func_t prp_cmp, H5P_prp_close_func_t prp_close)
{
    H5P_genprop_t *prop      = NULL; 
    H5P_genprop_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(name);
    assert((size > 0 && value != NULL) || (size == 0));
    assert(type != H5P_PROP_WITHIN_UNKNOWN);

    
    if (NULL == (prop = H5FL_MALLOC(H5P_genprop_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    prop->name        = H5MM_xstrdup(name); 
    prop->shared_name = false;
    prop->size        = size;
    prop->type        = type;

    
    if (value != NULL) {
        if (NULL == (prop->value = H5MM_malloc(prop->size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
        H5MM_memcpy(prop->value, value, prop->size);
    } 
    else
        prop->value = NULL;

    
    prop->create = prp_create;
    prop->set    = prp_set;
    prop->get    = prp_get;
    prop->encode = prp_encode;
    prop->decode = prp_decode;
    prop->del    = prp_delete;
    prop->copy   = prp_copy;
    
    if (prp_cmp != NULL)
        prop->cmp = prp_cmp;
    else
        prop->cmp = &memcmp;
    prop->close = prp_close;

    
    ret_value = prop;

done:
    
    if (ret_value == NULL) {
        if (prop != NULL) {
            if (prop->name != NULL)
                H5MM_xfree(prop->name);
            if (prop->value != NULL)
                H5MM_xfree(prop->value);
            prop = H5FL_FREE(H5P_genprop_t, prop);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__add_prop(H5SL_t *slist, H5P_genprop_t *prop)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(slist);
    assert(prop);
    assert(prop->type != H5P_PROP_WITHIN_UNKNOWN);

    
    if (H5SL_insert(slist, prop, prop->name) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into skip list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5P_genprop_t *
H5P__find_prop_plist(const H5P_genplist_t *plist, const char *name)
{
    H5P_genprop_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(plist);
    assert(name);

    
    if (H5SL_search(plist->del, name) != NULL) {
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, NULL, "property deleted from skip list");
    } 
    else {
        
        if (NULL == (ret_value = (H5P_genprop_t *)H5SL_search(plist->props, name))) {
            H5P_genclass_t *tclass; 

            
            tclass = plist->pclass;
            while (tclass != NULL) {
                
                if (NULL != (ret_value = (H5P_genprop_t *)H5SL_search(tclass->props, name)))
                    
                    break;

                
                tclass = tclass->parent;
            } 

            
            if (ret_value == NULL)
                HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, NULL, "can't find property in skip list");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5P_genprop_t *
H5P__find_prop_pclass(H5P_genclass_t *pclass, const char *name)
{
    H5P_genprop_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(pclass);
    assert(name);

    
    if (NULL == (ret_value = (H5P_genprop_t *)H5SL_search(pclass->props, name)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, NULL, "can't find property in skip list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__free_prop(H5P_genprop_t *prop)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(prop);

    
    if (prop->value)
        H5MM_xfree(prop->value);

    
    if (!prop->shared_name)
        H5MM_xfree(prop->name);

    prop = H5FL_FREE(H5P_genprop_t, prop);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__free_prop_cb(void *item, void H5_ATTR_UNUSED *key, void *op_data)
{
    H5P_genprop_t *tprop   = (H5P_genprop_t *)item; 
    bool           make_cb = *(bool *)op_data;      

    FUNC_ENTER_PACKAGE_NOERR

    assert(tprop);

    
    if (make_cb && tprop->close != NULL) {
        
        H5_BEFORE_USER_CB_NOCHECK
            {
                
                (tprop->close)(tprop->name, tprop->size, tprop->value);
            }
        H5_AFTER_USER_CB_NOCHECK
    }

    
    H5P__free_prop(tprop);

    FUNC_LEAVE_NOAPI(0)
} 

static herr_t
H5P__free_del_name_cb(void *item, void H5_ATTR_UNUSED *key, void H5_ATTR_UNUSED *op_data)
{
    char *del_name = (char *)item; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(del_name);

    
    H5MM_xfree(del_name);

    FUNC_LEAVE_NOAPI(0)
} 

herr_t
H5P__access_class(H5P_genclass_t *pclass, H5P_class_mod_t mod)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(pclass);
    assert(mod > H5P_MOD_ERR && mod < H5P_MOD_MAX);

    switch (mod) {
        case H5P_MOD_INC_CLS: 
            pclass->classes++;
            break;

        case H5P_MOD_DEC_CLS: 
            pclass->classes--;
            break;

        case H5P_MOD_INC_LST: 
            pclass->plists++;
            break;

        case H5P_MOD_DEC_LST: 
            pclass->plists--;
            break;

        case H5P_MOD_INC_REF: 
            
            if (pclass->deleted)
                pclass->deleted = false;
            pclass->ref_count++;
            break;

        case H5P_MOD_DEC_REF: 
            pclass->ref_count--;

            
            if (pclass->ref_count == 0)
                pclass->deleted = true;
            break;

        case H5P_MOD_ERR:
        case H5P_MOD_MAX:
        default:
            assert(0 && "Invalid H5P class modification");
    } 

    
    if (pclass->deleted && pclass->plists == 0 && pclass->classes == 0) {
        H5P_genclass_t *par_class = pclass->parent; 

        assert(pclass->name);
        H5MM_xfree(pclass->name);

        
        if (pclass->props) {
            bool make_cb = false;

            H5SL_destroy(pclass->props, H5P__free_prop_cb, &make_cb);
        } 

        pclass = H5FL_FREE(H5P_genclass_t, pclass);

        
        if (par_class != NULL)
            H5P__access_class(par_class, H5P_MOD_DEC_CLS);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static int
H5P__open_class_path_cb(void *_obj, hid_t H5_ATTR_UNUSED id, void *_key)
{
    H5P_genclass_t    *obj       = (H5P_genclass_t *)_obj;    
    H5P_check_class_t *key       = (H5P_check_class_t *)_key; 
    int                ret_value = 0;                         

    FUNC_ENTER_PACKAGE_NOERR

    assert(obj);
    assert(H5I_GENPROP_CLS == H5I_get_type(id));
    assert(key);

    
    if (obj->parent == key->parent) {
        
        if (strcmp(obj->name, key->name) == 0) {
            key->new_class = obj;
            ret_value      = 1; 
        }                       
    }                           

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5P_genclass_t *
H5P__create_class(H5P_genclass_t *par_class, const char *name, H5P_plist_type_t type,
                  H5P_cls_create_func_t cls_create, void *create_data, H5P_cls_copy_func_t cls_copy,
                  void *copy_data, H5P_cls_close_func_t cls_close, void *close_data)
{
    H5P_genclass_t *pclass    = NULL; 
    H5P_genclass_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(name);
    
    
    if (type == H5P_TYPE_USER)
        assert(par_class);

    
    if (NULL == (pclass = H5FL_CALLOC(H5P_genclass_t)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, NULL, "property list class allocation failed");

    
    pclass->parent = par_class;
    if (NULL == (pclass->name = H5MM_xstrdup(name)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, NULL, "property list class name allocation failed");
    pclass->type      = type;
    pclass->nprops    = 0;                
    pclass->plists    = 0;                
    pclass->classes   = 0;                
    pclass->ref_count = 1;                
    pclass->deleted   = false;            
    pclass->revision  = H5P_GET_NEXT_REV; 

    
    if (NULL == (pclass->props = H5SL_create(H5SL_TYPE_STR, NULL)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, NULL, "can't create skip list for properties");

    
    pclass->create_func = cls_create;
    pclass->create_data = create_data;
    pclass->copy_func   = cls_copy;
    pclass->copy_data   = copy_data;
    pclass->close_func  = cls_close;
    pclass->close_data  = close_data;

    
    if (par_class != NULL) {
        if (H5P__access_class(par_class, H5P_MOD_INC_CLS) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, NULL, "Can't increment parent class ref count");
    } 

    
    ret_value = pclass;

done:
    
    if (ret_value == NULL)
        if (pclass) {
            if (pclass->name)
                H5MM_xfree(pclass->name);
            if (pclass->props) {
                bool make_cb = false;

                H5SL_destroy(pclass->props, H5P__free_prop_cb, &make_cb);
            } 
            pclass = H5FL_FREE(H5P_genclass_t, pclass);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5P_genplist_t *
H5P__create(H5P_genclass_t *pclass)
{
    H5P_genclass_t *tclass;           
    H5P_genplist_t *plist = NULL;     
    H5P_genprop_t  *tmp;              
    H5SL_t         *seen      = NULL; 
    H5P_genplist_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(pclass);

    

    
    if (NULL == (plist = H5FL_CALLOC(H5P_genplist_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    plist->pclass     = pclass;
    plist->nprops     = 0;     
    plist->class_init = false; 

    
    if ((plist->props = H5SL_create(H5SL_TYPE_STR, NULL)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, NULL, "can't create skip list for changed properties");

    
    if ((plist->del = H5SL_create(H5SL_TYPE_STR, NULL)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, NULL, "can't create skip list for deleted properties");

    
    if ((seen = H5SL_create(H5SL_TYPE_STR, NULL)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, NULL, "can't create skip list for seen properties");

    
    tclass = pclass;
    while (tclass != NULL) {
        if (tclass->nprops > 0) {
            H5SL_node_t *curr_node; 

            
            curr_node = H5SL_first(tclass->props);
            while (curr_node != NULL) {
                
                tmp = (H5P_genprop_t *)H5SL_item(curr_node);

                
                if (H5SL_search(seen, tmp->name) == NULL) {
                    
                    if (tmp->create) {
                        
                        if (H5P__do_prop_cb1(plist->props, tmp, tmp->create) < 0)
                            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, NULL, "Can't create property");
                    } 

                    
                    if (H5SL_insert(seen, tmp->name, tmp->name) < 0)
                        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, NULL,
                                    "can't insert property into seen skip list");

                    
                    plist->nprops++;
                } 

                
                curr_node = H5SL_next(curr_node);
            } 
        }     

        
        tclass = tclass->parent;
    } 

    
    if (H5P__access_class(plist->pclass, H5P_MOD_INC_LST) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, NULL, "Can't increment class ref count");

    
    ret_value = plist;

done:
    
    if (seen != NULL)
        H5SL_close(seen);

    
    if (ret_value == NULL) {
        if (plist != NULL) {
            
            if (plist->props) {
                unsigned make_cb = 1;

                H5SL_destroy(plist->props, H5P__free_prop_cb, &make_cb);
            } 

            
            if (plist->del)
                H5SL_close(plist->del);

            
            plist = H5FL_FREE(H5P_genplist_t, plist);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5P_create_id(H5P_genclass_t *pclass, bool app_ref)
{
    H5P_genclass_t *tclass;                      
    H5P_genplist_t *plist     = NULL;            
    hid_t           plist_id  = FAIL;            
    hid_t           ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    assert(pclass);

    
    if ((plist = H5P__create(pclass)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, H5I_INVALID_HID, "unable to create property list");

    
    if ((plist_id = H5I_register(H5I_GENPROP_LST, plist, app_ref)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register property list");

    
    plist->plist_id = plist_id;

    
    tclass = plist->pclass;
    while (NULL != tclass) {
        if (NULL != tclass->create_func) {
            herr_t status;

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    status = (tclass->create_func)(plist_id, tclass->create_data);
                }
            H5_AFTER_USER_CB(FAIL)
            if (status < 0) {
                
                H5I_remove(plist_id);
                HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, H5I_INVALID_HID, "Can't initialize property");
            } 
        }     

        
        tclass = tclass->parent;
    } 

    
    plist->class_init = true;

    
    ret_value = plist_id;

done:
    if (H5I_INVALID_HID == ret_value && plist)
        H5P_close(plist);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__register_real(H5P_genclass_t *pclass, const char *name, size_t size, const void *def_value,
                   H5P_prp_create_func_t prp_create, H5P_prp_set_func_t prp_set, H5P_prp_get_func_t prp_get,
                   H5P_prp_encode_func_t prp_encode, H5P_prp_decode_func_t prp_decode,
                   H5P_prp_delete_func_t prp_delete, H5P_prp_copy_func_t prp_copy,
                   H5P_prp_compare_func_t prp_cmp, H5P_prp_close_func_t prp_close)
{
    H5P_genprop_t *new_prop  = NULL;    
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(pclass);
    assert(0 == pclass->plists);
    assert(0 == pclass->classes);
    assert(name);
    assert((size > 0 && def_value != NULL) || (size == 0));

    
    if (NULL != H5SL_search(pclass->props, name))
        HGOTO_ERROR(H5E_PLIST, H5E_EXISTS, FAIL, "property already exists");

    
    if (NULL == (new_prop = H5P__create_prop(name, size, H5P_PROP_WITHIN_CLASS, def_value, prp_create,
                                             prp_set, prp_get, prp_encode, prp_decode, prp_delete, prp_copy,
                                             prp_cmp, prp_close)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, FAIL, "Can't create property");

    
    if (H5P__add_prop(pclass->props, new_prop) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "Can't insert property into class");

    
    pclass->nprops++;

    
    pclass->revision = H5P_GET_NEXT_REV;

done:
    if (ret_value < 0)
        if (new_prop && H5P__free_prop(new_prop) < 0)
            HDONE_ERROR(H5E_PLIST, H5E_CANTRELEASE, FAIL, "unable to close property");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__register(H5P_genclass_t **ppclass, const char *name, size_t size, const void *def_value,
              H5P_prp_create_func_t prp_create, H5P_prp_set_func_t prp_set, H5P_prp_get_func_t prp_get,
              H5P_prp_encode_func_t prp_encode, H5P_prp_decode_func_t prp_decode,
              H5P_prp_delete_func_t prp_delete, H5P_prp_copy_func_t prp_copy, H5P_prp_compare_func_t prp_cmp,
              H5P_prp_close_func_t prp_close)
{
    H5P_genclass_t *pclass    = *ppclass; 
    H5P_genclass_t *new_class = NULL;     
    herr_t          ret_value = SUCCEED;  

    FUNC_ENTER_PACKAGE

    
    assert(ppclass);
    assert(pclass);

    
    if (pclass->plists > 0 || pclass->classes > 0) {
        if (NULL == (new_class = H5P__create_class(
                         pclass->parent, pclass->name, pclass->type, pclass->create_func, pclass->create_data,
                         pclass->copy_func, pclass->copy_data, pclass->close_func, pclass->close_data)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "can't copy class");

        
        if (pclass->nprops > 0) {
            H5SL_node_t *curr_node; 

            
            curr_node = H5SL_first(pclass->props);
            while (curr_node != NULL) {
                H5P_genprop_t *pcopy; 

                
                if (NULL ==
                    (pcopy = H5P__dup_prop((H5P_genprop_t *)H5SL_item(curr_node), H5P_PROP_WITHIN_CLASS)))
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "Can't copy property");

                
                if (H5P__add_prop(new_class->props, pcopy) < 0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "Can't insert property into class");

                
                new_class->nprops++;

                
                curr_node = H5SL_next(curr_node);
            } 
        }     

        
        pclass = new_class;
    } 

    
    if (H5P__register_real(pclass, name, size, def_value, prp_create, prp_set, prp_get, prp_encode,
                           prp_decode, prp_delete, prp_copy, prp_cmp, prp_close) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, FAIL, "can't register property");

    
    if (new_class)
        *ppclass = pclass;

done:
    if (ret_value < 0)
        if (new_class && H5P__close_class(new_class) < 0)
            HDONE_ERROR(H5E_PLIST, H5E_CANTRELEASE, FAIL, "unable to close new property class");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P_insert(H5P_genplist_t *plist, const char *name, size_t size, void *value, H5P_prp_set_func_t prp_set,
           H5P_prp_get_func_t prp_get, H5P_prp_encode_func_t prp_encode, H5P_prp_decode_func_t prp_decode,
           H5P_prp_delete_func_t prp_delete, H5P_prp_copy_func_t prp_copy, H5P_prp_compare_func_t prp_cmp,
           H5P_prp_close_func_t prp_close)
{
    H5P_genprop_t *new_prop  = NULL;    
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    assert(plist);
    assert(name);
    assert((size > 0 && value != NULL) || (size == 0));

    
    if (NULL != H5SL_search(plist->props, name))
        HGOTO_ERROR(H5E_PLIST, H5E_EXISTS, FAIL, "property already exists");

    
    if (NULL != H5SL_search(plist->del, name)) {
        char *temp_name = NULL;

        
        if (NULL == (temp_name = (char *)H5SL_remove(plist->del, name)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTDELETE, FAIL, "can't remove property from deleted skip list");

        
        H5MM_xfree(temp_name);
    } 
    else {
        H5P_genclass_t *tclass; 

        
        tclass = plist->pclass;
        while (tclass) {
            if (tclass->nprops > 0) {
                
                if (NULL != H5SL_search(tclass->props, name))
                    HGOTO_ERROR(H5E_PLIST, H5E_EXISTS, FAIL, "property already exists");
            } 

            
            tclass = tclass->parent;
        } 
    }     

    

    
    if (NULL ==
        (new_prop = H5P__create_prop(name, size, H5P_PROP_WITHIN_LIST, value, NULL, prp_set, prp_get,
                                     prp_encode, prp_decode, prp_delete, prp_copy, prp_cmp, prp_close)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, FAIL, "Can't create property");

    
    if (H5P__add_prop(plist->props, new_prop) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "Can't insert property into class");

    
    plist->nprops++;

done:
    if (ret_value < 0)
        if (new_prop && H5P__free_prop(new_prop) < 0)
            HDONE_ERROR(H5E_PLIST, H5E_CANTRELEASE, FAIL, "unable to close property");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__do_prop(H5P_genplist_t *plist, const char *name, H5P_do_plist_op_t plist_op,
             H5P_do_pclass_op_t pclass_op, void *udata)
{
    H5P_genclass_t *tclass;              
    H5P_genprop_t  *prop;                
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(plist);
    assert(name);
    assert(plist_op);
    assert(pclass_op);

    
    if (NULL != H5SL_search(plist->del, name))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "property doesn't exist");

    
    if (NULL != (prop = (H5P_genprop_t *)H5SL_search(plist->props, name))) {
        
        if ((*plist_op)(plist, name, prop, udata) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTOPERATE, FAIL, "can't operate on property");
    } 
    else {
        
        tclass = plist->pclass;
        while (NULL != tclass) {
            if (tclass->nprops > 0) {
                
                if (NULL != (prop = (H5P_genprop_t *)H5SL_search(tclass->props, name))) {
                    
                    if ((*pclass_op)(plist, name, prop, udata) < 0)
                        HGOTO_ERROR(H5E_PLIST, H5E_CANTOPERATE, FAIL, "can't operate on property");

                    
                    break;
                } 
            }     

            
            tclass = tclass->parent;
        } 

        
        if (NULL == tclass)
            HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "can't find property in skip list");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__poke_plist_cb(H5P_genplist_t H5_ATTR_NDEBUG_UNUSED *plist, const char H5_ATTR_NDEBUG_UNUSED *name,
                   H5P_genprop_t *prop, void *_udata)
{
    H5P_prop_set_ud_t *udata     = (H5P_prop_set_ud_t *)_udata; 
    herr_t             ret_value = SUCCEED;                     

    FUNC_ENTER_PACKAGE

    
    assert(plist);
    assert(name);
    assert(prop);

    
    if (0 == prop->size)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "property has zero size");

    
    H5MM_memcpy(prop->value, udata->value, prop->size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__poke_pclass_cb(H5P_genplist_t *plist, const char H5_ATTR_NDEBUG_UNUSED *name, H5P_genprop_t *prop,
                    void *_udata)
{
    H5P_prop_set_ud_t *udata     = (H5P_prop_set_ud_t *)_udata; 
    H5P_genprop_t     *pcopy     = NULL;    
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(plist);
    assert(name);
    assert(prop);
    assert(prop->cmp);

    
    if (0 == prop->size)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "property has zero size");

    
    if (NULL == (pcopy = H5P__dup_prop(prop, H5P_PROP_WITHIN_LIST)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "Can't copy property");

    H5MM_memcpy(pcopy->value, udata->value, pcopy->size);

    
    if (H5P__add_prop(plist->props, pcopy) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "Can't insert changed property into skip list");

done:
    
    if (ret_value < 0)
        if (pcopy)
            H5P__free_prop(pcopy);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P_poke(H5P_genplist_t *plist, const char *name, const void *value)
{
    H5P_prop_set_ud_t udata;               
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(plist);
    assert(name);
    assert(value);

    
    udata.value = value;
    if (H5P__do_prop(plist, name, H5P__poke_plist_cb, H5P__poke_pclass_cb, &udata) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTOPERATE, FAIL, "can't operate on plist to overwrite value");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__set_plist_cb(H5P_genplist_t *plist, const char *name, H5P_genprop_t *prop, void *_udata)
{
    H5P_prop_set_ud_t *udata     = (H5P_prop_set_ud_t *)_udata; 
    void              *tmp_value = NULL;                        
    const void        *prp_value = NULL;                        
    herr_t             ret_value = SUCCEED;                     

    FUNC_ENTER_PACKAGE

    
    assert(plist);
    assert(name);
    assert(prop);

    
    if (0 == prop->size)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "property has zero size");

    
    if (NULL != prop->set) {
        
        if (NULL == (tmp_value = H5MM_malloc(prop->size)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed temporary property value");
        H5MM_memcpy(tmp_value, udata->value, prop->size);

        
        H5_BEFORE_USER_CB(FAIL)
            {
                
                ret_value = (*(prop->set))(plist->plist_id, name, prop->size, tmp_value);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't set property value");

        
        prp_value = tmp_value;
    } 
    
    else
        prp_value = udata->value;

    
    if (NULL != prop->del) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                
                ret_value = (*(prop->del))(plist->plist_id, name, prop->size, prop->value);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTFREE, FAIL, "can't release property value");
    } 

    
    H5MM_memcpy(prop->value, prp_value, prop->size);

done:
    
    if (tmp_value != NULL)
        H5MM_xfree(tmp_value);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__set_pclass_cb(H5P_genplist_t *plist, const char *name, H5P_genprop_t *prop, void *_udata)
{
    H5P_prop_set_ud_t *udata     = (H5P_prop_set_ud_t *)_udata; 
    H5P_genprop_t     *pcopy     = NULL;    
    void              *tmp_value = NULL;    
    const void        *prp_value = NULL;    
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(plist);
    assert(name);
    assert(prop);
    assert(prop->cmp);

    
    if (0 == prop->size)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "property has zero size");

    
    if (NULL != prop->set) {
        
        if (NULL == (tmp_value = H5MM_malloc(prop->size)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed temporary property value");
        H5MM_memcpy(tmp_value, udata->value, prop->size);

        
        H5_BEFORE_USER_CB(FAIL)
            {
                
                ret_value = (*(prop->set))(plist->plist_id, name, prop->size, tmp_value);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't set property value");

        
        prp_value = tmp_value;
    } 
    
    else
        prp_value = udata->value;

    
    if (NULL == (pcopy = H5P__dup_prop(prop, H5P_PROP_WITHIN_LIST)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "Can't copy property");

    H5MM_memcpy(pcopy->value, prp_value, pcopy->size);

    
    if (H5P__add_prop(plist->props, pcopy) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "Can't insert changed property into skip list");

done:
    
    if (tmp_value != NULL)
        H5MM_xfree(tmp_value);

    
    if (ret_value < 0)
        if (pcopy)
            H5P__free_prop(pcopy);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P_set(H5P_genplist_t *plist, const char *name, const void *value)
{
    H5P_prop_set_ud_t udata;               
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(plist);
    assert(name);
    assert(value);

    
    udata.value = value;
    if (H5P__do_prop(plist, name, H5P__set_plist_cb, H5P__set_pclass_cb, &udata) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTOPERATE, FAIL, "can't operate on plist to set value");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__class_get(const H5P_genclass_t *pclass, const char *name, void *value)
{
    H5P_genprop_t *prop;                
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pclass);
    assert(name);
    assert(value);

    
    if (NULL == (prop = (H5P_genprop_t *)H5SL_search(pclass->props, name)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "property doesn't exist");

    
    if (0 == prop->size)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "property has zero size");

    
    H5MM_memcpy(value, prop->value, prop->size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__class_set(const H5P_genclass_t *pclass, const char *name, const void *value)
{
    H5P_genprop_t *prop;                
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pclass);
    assert(name);
    assert(value);

    
    if (NULL == (prop = (H5P_genprop_t *)H5SL_search(pclass->props, name)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "property doesn't exist");

    
    if (0 == prop->size)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "property has zero size");

    
    H5MM_memcpy(prop->value, value, prop->size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5P_exist_plist(const H5P_genplist_t *plist, const char *name)
{
    htri_t ret_value = FAIL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(plist);
    assert(name);

    
    if (H5SL_search(plist->del, name) != NULL)
        ret_value = false;
    else {
        
        if (H5SL_search(plist->props, name) != NULL)
            ret_value = true;
        else {
            H5P_genclass_t *tclass; 

            tclass = plist->pclass;
            while (tclass != NULL) {
                if (H5SL_search(tclass->props, name) != NULL)
                    HGOTO_DONE(true);

                
                tclass = tclass->parent;
            } 

            
            ret_value = false;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5P__exist_pclass(H5P_genclass_t *pclass, const char *name)
{
    htri_t ret_value = FAIL; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(pclass);
    assert(name);

    
    if (H5SL_search(pclass->props, name) != NULL)
        ret_value = true;
    else {
        H5P_genclass_t *tclass; 

        tclass = pclass->parent;
        while (tclass != NULL) {
            if (H5SL_search(tclass->props, name) != NULL)
                HGOTO_DONE(true);

            
            tclass = tclass->parent;
        } 

        
        ret_value = false;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__get_size_plist(const H5P_genplist_t *plist, const char *name, size_t *size)
{
    H5P_genprop_t *prop;                
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(plist);
    assert(name);
    assert(size);

    
    if (NULL == (prop = H5P__find_prop_plist(plist, name)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "property doesn't exist");

    
    *size = prop->size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__get_size_pclass(H5P_genclass_t *pclass, const char *name, size_t *size)
{
    H5P_genprop_t *prop;                
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(pclass);
    assert(name);
    assert(size);

    
    if ((prop = H5P__find_prop_pclass(pclass, name)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "property doesn't exist");

    
    *size = prop->size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__get_nprops_plist(const H5P_genplist_t *plist, size_t *nprops)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(plist);
    assert(nprops);

    
    *nprops = plist->nprops;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5P_get_nprops_pclass(const H5P_genclass_t *pclass, size_t *nprops, bool recurse)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(pclass);
    assert(nprops);

    
    *nprops = pclass->nprops;

    
    if (recurse)
        while (pclass->parent != NULL) {
            pclass = pclass->parent;
            *nprops += pclass->nprops;
        } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__cmp_prop(const H5P_genprop_t *prop1, const H5P_genprop_t *prop2)
{
    int cmp_value;     
    int ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(prop1);
    assert(prop2);

    
    if ((cmp_value = strcmp(prop1->name, prop2->name)) != 0)
        HGOTO_DONE(cmp_value);

    
    if (prop1->size < prop2->size)
        HGOTO_DONE(-1);
    if (prop1->size > prop2->size)
        HGOTO_DONE(1);

    
    if (prop1->create == NULL && prop2->create != NULL)
        HGOTO_DONE(-1);
    if (prop1->create != NULL && prop2->create == NULL)
        HGOTO_DONE(1);
    if (prop1->create != prop2->create)
        HGOTO_DONE(-1);

    
    if (prop1->set == NULL && prop2->set != NULL)
        HGOTO_DONE(-1);
    if (prop1->set != NULL && prop2->set == NULL)
        HGOTO_DONE(1);
    if (prop1->set != prop2->set)
        HGOTO_DONE(-1);

    
    if (prop1->get == NULL && prop2->get != NULL)
        HGOTO_DONE(-1);
    if (prop1->get != NULL && prop2->get == NULL)
        HGOTO_DONE(1);
    if (prop1->get != prop2->get)
        HGOTO_DONE(-1);

    
    if (prop1->encode == NULL && prop2->encode != NULL)
        HGOTO_DONE(-1);
    if (prop1->encode != NULL && prop2->encode == NULL)
        HGOTO_DONE(1);
    if (prop1->encode != prop2->encode)
        HGOTO_DONE(-1);

    
    if (prop1->decode == NULL && prop2->decode != NULL)
        HGOTO_DONE(-1);
    if (prop1->decode != NULL && prop2->decode == NULL)
        HGOTO_DONE(1);
    if (prop1->decode != prop2->decode)
        HGOTO_DONE(-1);

    
    if (prop1->del == NULL && prop2->del != NULL)
        HGOTO_DONE(-1);
    if (prop1->del != NULL && prop2->del == NULL)
        HGOTO_DONE(1);
    if (prop1->del != prop2->del)
        HGOTO_DONE(-1);

    
    if (prop1->copy == NULL && prop2->copy != NULL)
        HGOTO_DONE(-1);
    if (prop1->copy != NULL && prop2->copy == NULL)
        HGOTO_DONE(1);
    if (prop1->copy != prop2->copy)
        HGOTO_DONE(-1);

    
    if (prop1->cmp == NULL && prop2->cmp != NULL)
        HGOTO_DONE(-1);
    if (prop1->cmp != NULL && prop2->cmp == NULL)
        HGOTO_DONE(1);
    if (prop1->cmp != prop2->cmp)
        HGOTO_DONE(-1);

    
    if (prop1->close == NULL && prop2->close != NULL)
        HGOTO_DONE(-1);
    if (prop1->close != NULL && prop2->close == NULL)
        HGOTO_DONE(1);
    if (prop1->close != prop2->close)
        HGOTO_DONE(-1);

    
    if (prop1->value == NULL && prop2->value != NULL)
        HGOTO_DONE(-1);
    if (prop1->value != NULL && prop2->value == NULL)
        HGOTO_DONE(1);
    if (prop1->value != NULL) {
        
        H5_BEFORE_USER_CB_NOCHECK
            {
                
                cmp_value = prop1->cmp(prop1->value, prop2->value, prop1->size);
            }
        H5_AFTER_USER_CB_NOCHECK
        
        if (0 != cmp_value)
            HGOTO_DONE(cmp_value);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5P__cmp_class(const H5P_genclass_t *pclass1, const H5P_genclass_t *pclass2)
{
    H5SL_node_t *tnode1, *tnode2; 
    int          cmp_value;       
    int          ret_value = 0;   

    FUNC_ENTER_PACKAGE_NOERR

    assert(pclass1);
    assert(pclass2);

    
    if (pclass1->revision == pclass2->revision)
        HGOTO_DONE(0);

    
    if ((cmp_value = strcmp(pclass1->name, pclass2->name)) != 0)
        HGOTO_DONE(cmp_value);

    
    if (pclass1->nprops < pclass2->nprops)
        HGOTO_DONE(-1);
    if (pclass1->nprops > pclass2->nprops)
        HGOTO_DONE(1);

    
    if (pclass1->plists < pclass2->plists)
        HGOTO_DONE(-1);
    if (pclass1->plists > pclass2->plists)
        HGOTO_DONE(1);

    
    if (pclass1->classes < pclass2->classes)
        HGOTO_DONE(-1);
    if (pclass1->classes > pclass2->classes)
        HGOTO_DONE(1);

    
    if (pclass1->ref_count < pclass2->ref_count)
        HGOTO_DONE(-1);
    if (pclass1->ref_count > pclass2->ref_count)
        HGOTO_DONE(1);

    
    if (pclass1->type < pclass2->type)
        HGOTO_DONE(-1);
    if (pclass1->type > pclass2->type)
        HGOTO_DONE(1);

    
    if (pclass1->deleted < pclass2->deleted)
        HGOTO_DONE(-1);
    if (pclass1->deleted > pclass2->deleted)
        HGOTO_DONE(1);

    
    if (pclass1->create_func == NULL && pclass2->create_func != NULL)
        HGOTO_DONE(-1);
    if (pclass1->create_func != NULL && pclass2->create_func == NULL)
        HGOTO_DONE(1);
    if (pclass1->create_func != pclass2->create_func)
        HGOTO_DONE(-1);
    if (pclass1->create_data < pclass2->create_data)
        HGOTO_DONE(-1);
    if (pclass1->create_data > pclass2->create_data)
        HGOTO_DONE(1);

    
    if (pclass1->close_func == NULL && pclass2->close_func != NULL)
        HGOTO_DONE(-1);
    if (pclass1->close_func != NULL && pclass2->close_func == NULL)
        HGOTO_DONE(1);
    if (pclass1->close_func != pclass2->close_func)
        HGOTO_DONE(-1);
    if (pclass1->close_data < pclass2->close_data)
        HGOTO_DONE(-1);
    if (pclass1->close_data > pclass2->close_data)
        HGOTO_DONE(1);

    
    tnode1 = H5SL_first(pclass1->props);
    tnode2 = H5SL_first(pclass2->props);
    while (tnode1 || tnode2) {
        H5P_genprop_t *prop1, *prop2; 

        
        if (tnode1 == NULL && tnode2 != NULL)
            HGOTO_DONE(-1);
        if (tnode1 != NULL && tnode2 == NULL)
            HGOTO_DONE(1);

        
        prop1 = (H5P_genprop_t *)H5SL_item(tnode1);
        prop2 = (H5P_genprop_t *)H5SL_item(tnode2);
        if ((cmp_value = H5P__cmp_prop(prop1, prop2)) != 0)
            HGOTO_DONE(cmp_value);

        
        tnode1 = H5SL_next(tnode1);
        tnode2 = H5SL_next(tnode2);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__cmp_plist_cb(H5P_genprop_t *prop, void *_udata)
{
    H5P_plist_cmp_ud_t *udata = (H5P_plist_cmp_ud_t *)_udata; 
    htri_t              prop2_exist; 
    int                 ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE

    
    assert(prop);
    assert(udata);

    
    if ((prop2_exist = H5P_exist_plist(udata->plist2, prop->name)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, H5_ITER_ERROR, "can't lookup existence of property?");
    if (prop2_exist) {
        const H5P_genprop_t *prop2; 

        
        if (NULL == (prop2 = H5P__find_prop_plist(udata->plist2, prop->name)))
            HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, H5_ITER_ERROR, "property doesn't exist");

        
        if ((udata->cmp_value = H5P__cmp_prop(prop, prop2)) != 0)
            HGOTO_DONE(H5_ITER_STOP);
    } 
    else {
        
        udata->cmp_value = 1;
        HGOTO_DONE(H5_ITER_STOP);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__cmp_plist(const H5P_genplist_t *plist1, const H5P_genplist_t *plist2, int *cmp_ret)
{
    H5P_plist_cmp_ud_t udata;               
    int                idx       = 0;       
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(plist1);
    assert(plist2);
    assert(cmp_ret);

    
    if (plist1->nprops < plist2->nprops) {
        *cmp_ret = -1;
        HGOTO_DONE(SUCCEED);
    } 
    if (plist1->nprops > plist2->nprops) {
        *cmp_ret = 1;
        HGOTO_DONE(SUCCEED);
    } 

    
    if (plist1->class_init < plist2->class_init) {
        *cmp_ret = -1;
        HGOTO_DONE(SUCCEED);
    } 
    if (plist1->class_init > plist2->class_init) {
        *cmp_ret = 1;
        HGOTO_DONE(SUCCEED);
    } 

    
    udata.cmp_value = 0;
    udata.plist2    = plist2;

    
    if ((ret_value = H5P__iterate_plist(plist1, true, &idx, H5P__cmp_plist_cb, &udata)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to iterate over list");
    if (ret_value != 0) {
        *cmp_ret = udata.cmp_value;
        HGOTO_DONE(SUCCEED);
    } 

    
    if ((*cmp_ret = H5P__cmp_class(plist1->pclass, plist2->pclass)) != 0)
        HGOTO_DONE(SUCCEED);

    
    *cmp_ret = 0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5P_class_isa(const H5P_genclass_t *pclass1, const H5P_genclass_t *pclass2)
{
    htri_t ret_value = FAIL; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(pclass1);
    assert(pclass2);

    
    if (H5P__cmp_class(pclass1, pclass2) == 0) {
        HGOTO_DONE(true);
    }
    else {
        
        if (pclass1->parent != NULL)
            ret_value = H5P_class_isa(pclass1->parent, pclass2);
        else
            HGOTO_DONE(false);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5P_isa_class(hid_t plist_id, hid_t pclass_id)
{
    H5P_genplist_t *plist;            
    H5P_genclass_t *pclass;           
    htri_t          ret_value = FAIL; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object_verify(plist_id, H5I_GENPROP_LST)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");
    if (NULL == (pclass = (H5P_genclass_t *)H5I_object_verify(pclass_id, H5I_GENPROP_CLS)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property class");

    
    if ((ret_value = H5P_class_isa(plist->pclass, pclass)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTREGISTER, FAIL, "unable to compare property list classes");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5P_is_default_plist(hid_t plist_id)
{
    hid_t H5I_def_plists[] = {
        H5P_LST_FILE_CREATE_ID_g,      H5P_LST_FILE_ACCESS_ID_g,      H5P_LST_DATASET_CREATE_ID_g,
        H5P_LST_DATASET_ACCESS_ID_g,   H5P_LST_DATASET_XFER_ID_g,     H5P_LST_FILE_MOUNT_ID_g,
        H5P_LST_GROUP_CREATE_ID_g,     H5P_LST_GROUP_ACCESS_ID_g,     H5P_LST_DATATYPE_CREATE_ID_g,
        H5P_LST_DATATYPE_ACCESS_ID_g,  H5P_LST_MAP_CREATE_ID_g,       H5P_LST_MAP_ACCESS_ID_g,
        H5P_LST_ATTRIBUTE_CREATE_ID_g, H5P_LST_ATTRIBUTE_ACCESS_ID_g, H5P_LST_OBJECT_COPY_ID_g,
        H5P_LST_LINK_CREATE_ID_g,      H5P_LST_LINK_ACCESS_ID_g,      H5P_LST_VOL_INITIALIZE_ID_g,
        H5P_LST_REFERENCE_ACCESS_ID_g};

    size_t num_default_plists = (size_t)(sizeof(H5I_def_plists) / sizeof(H5I_def_plists[0]));

    if (plist_id == H5P_DEFAULT)
        return true;

    for (size_t i = 0; i < num_default_plists; i++) {
        if (plist_id == H5I_def_plists[i])
            return true;
    }

    return false;
}

H5P_genplist_t *
H5P_object_verify(hid_t plist_id, hid_t pclass_id, bool allow_default)
{
    H5P_genplist_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    if (H5P_isa_class(plist_id, pclass_id) != true)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOMPARE, NULL, "property list is not a member of the class");

    if (!allow_default && H5P_is_default_plist(plist_id)) {
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOMPARE, NULL, "property list is a default list");
    }

    
    if (NULL == (ret_value = (H5P_genplist_t *)H5I_object(plist_id)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, NULL, "can't find object for ID");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__iterate_plist_cb(void *_item, void *_key, void *_udata)
{
    H5P_genprop_t       *item      = (H5P_genprop_t *)_item;        
    char                *key       = (char *)_key;                  
    H5P_iter_plist_ud_t *udata     = (H5P_iter_plist_ud_t *)_udata; 
    int                  ret_value = H5_ITER_CONT;                  

    FUNC_ENTER_PACKAGE

    
    assert(item);
    assert(key);

    
    if (*udata->curr_idx_ptr >= udata->prev_idx) {
        
        H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
            {
                
                ret_value = (*udata->cb_func)(item, udata->udata);
            }
        H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)
        if (ret_value != 0)
            HGOTO_DONE(ret_value);
    } 

    
    (*udata->curr_idx_ptr)++;

    
    if (H5SL_insert(udata->seen, key, key) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, H5_ITER_ERROR, "can't insert property into 'seen' skip list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__iterate_plist_pclass_cb(void *_item, void *_key, void *_udata)
{
    H5P_genprop_t       *item      = (H5P_genprop_t *)_item;        
    char                *key       = (char *)_key;                  
    H5P_iter_plist_ud_t *udata     = (H5P_iter_plist_ud_t *)_udata; 
    int                  ret_value = H5_ITER_CONT;                  

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(item);
    assert(key);

    
    if (NULL == H5SL_search(udata->seen, key) && NULL == H5SL_search(udata->plist->del, key))
        ret_value = H5P__iterate_plist_cb(item, key, udata);

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5P__iterate_plist(const H5P_genplist_t *plist, bool iter_all_prop, int *idx, H5P_iterate_int_t cb_func,
                   void *udata)
{
    H5P_genclass_t     *tclass;           
    H5P_iter_plist_ud_t udata_int;        
    H5SL_t             *seen      = NULL; 
    int                 curr_idx  = 0;    
    int                 ret_value = 0;    

    FUNC_ENTER_PACKAGE

    
    assert(plist);
    assert(idx);
    assert(cb_func);

    
    if (NULL == (seen = H5SL_create(H5SL_TYPE_STR, NULL)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, FAIL, "can't create skip list for seen properties");

    
    udata_int.plist        = plist;
    udata_int.cb_func      = cb_func;
    udata_int.udata        = udata;
    udata_int.seen         = seen;
    udata_int.curr_idx_ptr = &curr_idx;
    udata_int.prev_idx     = *idx;

    
    
    ret_value = H5SL_iterate(plist->props, H5P__iterate_plist_cb, &udata_int);
    if (ret_value != 0)
        HGOTO_DONE(ret_value);

    
    if (iter_all_prop) {
        
        tclass = plist->pclass;
        while (tclass != NULL) {
            
            ret_value = H5SL_iterate(tclass->props, H5P__iterate_plist_pclass_cb, &udata_int);
            if (ret_value != 0)
                HGOTO_DONE(ret_value);

            
            tclass = tclass->parent;
        } 
    }     

done:
    
    *idx = curr_idx;

    
    if (seen != NULL)
        H5SL_close(seen);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__iterate_pclass_cb(void *_item, void H5_ATTR_NDEBUG_UNUSED *_key, void *_udata)
{
    H5P_genprop_t        *item      = (H5P_genprop_t *)_item;         
    H5P_iter_pclass_ud_t *udata     = (H5P_iter_pclass_ud_t *)_udata; 
    int                   ret_value = 0;                              

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(item);
    assert((char *)_key);

    
    if (*udata->curr_idx_ptr >= udata->prev_idx) {
        
        H5_BEFORE_USER_CB_NOERR(H5_ITER_ERROR)
            {
                
                ret_value = (*udata->cb_func)(item, udata->udata);
            }
        H5_AFTER_USER_CB_NOERR(H5_ITER_ERROR)
        if (ret_value != 0)
            HGOTO_DONE(ret_value);
    } 

    
    (*udata->curr_idx_ptr)++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5P__iterate_pclass(const H5P_genclass_t *pclass, int *idx, H5P_iterate_int_t cb_func, void *udata)
{
    H5P_iter_pclass_ud_t udata_int;     
    int                  curr_idx  = 0; 
    int                  ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pclass);
    assert(idx);
    assert(cb_func);

    
    udata_int.cb_func      = cb_func;
    udata_int.udata        = udata;
    udata_int.curr_idx_ptr = &curr_idx;
    udata_int.prev_idx     = *idx;

    
    ret_value = H5SL_iterate(pclass->props, H5P__iterate_pclass_cb, &udata_int);
    if (ret_value != 0)
        HGOTO_DONE(ret_value);

done:
    
    *idx = curr_idx;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__peek_cb(H5P_genplist_t H5_ATTR_NDEBUG_UNUSED *plist, const char H5_ATTR_NDEBUG_UNUSED *name,
             H5P_genprop_t *prop, void *_udata)
{
    H5P_prop_get_ud_t *udata     = (H5P_prop_get_ud_t *)_udata; 
    herr_t             ret_value = SUCCEED;                     

    FUNC_ENTER_PACKAGE

    
    assert(plist);
    assert(name);
    assert(prop);

    
    if (0 == prop->size)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "property has zero size");

    
    H5MM_memcpy(udata->value, prop->value, prop->size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P_peek(H5P_genplist_t *plist, const char *name, void *value)
{
    H5P_prop_get_ud_t udata;               
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(plist);
    assert(name);
    assert(value);

    
    udata.value = value;
    if (H5P__do_prop(plist, name, H5P__peek_cb, H5P__peek_cb, &udata) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTOPERATE, FAIL, "can't operate on plist to peek at value");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__get_cb(H5P_genplist_t *plist, const char *name, H5P_genprop_t *prop, void *_udata)
{
    H5P_prop_get_ud_t *udata     = (H5P_prop_get_ud_t *)_udata; 
    void              *tmp_value = NULL;                        
    herr_t             ret_value = SUCCEED;                     

    FUNC_ENTER_PACKAGE

    
    assert(plist);
    assert(name);
    assert(prop);

    
    if (0 == prop->size)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "property has zero size");

    
    if (NULL != prop->get) {
        
        if (NULL == (tmp_value = H5MM_malloc(prop->size)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed temporary property value");
        H5MM_memcpy(tmp_value, prop->value, prop->size);

        
        H5_BEFORE_USER_CB(FAIL)
            {
                
                ret_value = (*(prop->get))(plist->plist_id, name, prop->size, tmp_value);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't set property value");

        
        H5MM_memcpy(udata->value, tmp_value, prop->size);
    } 
    
    else
        H5MM_memcpy(udata->value, prop->value, prop->size);

done:
    
    if (tmp_value)
        H5MM_xfree(tmp_value);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P_get(H5P_genplist_t *plist, const char *name, void *value)
{
    H5P_prop_get_ud_t udata;               
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(plist);
    assert(name);
    assert(value);

    
    udata.value = value;
    if (H5P__do_prop(plist, name, H5P__get_cb, H5P__get_cb, &udata) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTOPERATE, FAIL, "can't operate on plist to get value");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__del_plist_cb(H5P_genplist_t *plist, const char *name, H5P_genprop_t *prop, void H5_ATTR_UNUSED *_udata)
{
    char  *del_name  = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(plist);
    assert(name);
    assert(prop);

    
    if (NULL != prop->del) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                
                ret_value = (*(prop->del))(plist->plist_id, name, prop->size, prop->value);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTFREE, FAIL, "can't release property value");
    } 

    
    if (NULL == (del_name = H5MM_xstrdup(name)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed");

    
    if (H5SL_insert(plist->del, del_name, del_name) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into deleted skip list");

    
    if (NULL == H5SL_remove(plist->props, prop->name))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTDELETE, FAIL, "can't remove property from skip list");

    
    H5P__free_prop(prop);

    
    plist->nprops--;

done:
    
    if (ret_value < 0)
        if (del_name)
            H5MM_xfree(del_name);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__del_pclass_cb(H5P_genplist_t *plist, const char *name, H5P_genprop_t *prop, void H5_ATTR_UNUSED *_udata)
{
    char  *del_name  = NULL;    
    void  *tmp_value = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(plist);
    assert(name);
    assert(prop);

    
    if (NULL != prop->del) {
        
        if (NULL == (tmp_value = H5MM_malloc(prop->size)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL,
                        "memory allocation failed for temporary property value");
        H5MM_memcpy(tmp_value, prop->value, prop->size);

        
        H5_BEFORE_USER_CB(FAIL)
            {
                
                ret_value = (*(prop->del))(plist->plist_id, name, prop->size, tmp_value);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't close property value");
    } 

    
    if (NULL == (del_name = H5MM_xstrdup(name)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "memory allocation failed");

    
    if (H5SL_insert(plist->del, del_name, del_name) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into deleted skip list");

    
    plist->nprops--;

done:
    
    if (tmp_value)
        H5MM_xfree(tmp_value);

    
    if (ret_value < 0)
        if (del_name)
            H5MM_xfree(del_name);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P_remove(H5P_genplist_t *plist, const char *name)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(plist);
    assert(name);

    
    if (H5P__do_prop(plist, name, H5P__del_plist_cb, H5P__del_pclass_cb, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTOPERATE, FAIL, "can't operate on plist to remove value");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__copy_prop_plist(hid_t dst_id, hid_t src_id, const char *name)
{
    H5P_genplist_t *dst_plist;           
    H5P_genplist_t *src_plist;           
    H5P_genprop_t  *prop;                
    H5P_genprop_t  *new_prop  = NULL;    
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(name);

    
    if (NULL == (src_plist = (H5P_genplist_t *)H5I_object(src_id)) ||
        NULL == (dst_plist = (H5P_genplist_t *)H5I_object(dst_id)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "property object doesn't exist");

    
    if (NULL == (prop = H5P__find_prop_plist(src_plist, name)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "property doesn't exist");

    
    if (NULL != H5P__find_prop_plist(dst_plist, name)) {
        
        if (H5P_remove(dst_plist, name) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTDELETE, FAIL, "unable to remove property");

        
        if ((new_prop = H5P__dup_prop(prop, H5P_PROP_WITHIN_LIST)) == NULL)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "Can't copy property");

        
        if (new_prop->copy) {
            
            H5_BEFORE_USER_CB(FAIL)
                {
                    
                    ret_value = (new_prop->copy)(new_prop->name, new_prop->size, new_prop->value);
                }
            H5_AFTER_USER_CB(FAIL)
            if (ret_value < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "Can't copy property");
        } 
    }     
    
    else {
        
        if (NULL ==
            (new_prop = H5P__create_prop(prop->name, prop->size, H5P_PROP_WITHIN_LIST, prop->value,
                                         prop->create, prop->set, prop->get, prop->encode, prop->decode,
                                         prop->del, prop->copy, prop->cmp, prop->close)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, FAIL, "Can't create property");

        
        if (new_prop->create) {
            
            H5_BEFORE_USER_CB(FAIL)
                {
                    ret_value = (new_prop->create)(new_prop->name, new_prop->size, new_prop->value);
                }
            H5_AFTER_USER_CB(FAIL)
            if (ret_value < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "Can't initialize property");
        } 
    }     

    
    if (H5P__add_prop(dst_plist->props, new_prop) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "Can't insert property into list");

    
    dst_plist->nprops++;

done:
    
    if (ret_value < 0) {
        if (new_prop != NULL)
            H5P__free_prop(new_prop);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__copy_prop_pclass(hid_t dst_id, hid_t src_id, const char *name)
{
    H5P_genclass_t *src_pclass;          
    H5P_genclass_t *dst_pclass;          
    H5P_genclass_t *orig_dst_pclass;     
    H5P_genprop_t  *prop;                
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(name);

    
    if (NULL == (src_pclass = (H5P_genclass_t *)H5I_object(src_id)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "source property class object doesn't exist");
    if (NULL == (dst_pclass = (H5P_genclass_t *)H5I_object(dst_id)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "destination property class object doesn't exist");

    
    if (NULL == (prop = H5P__find_prop_pclass(src_pclass, name)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "unable to locate property");

    
    if (H5P__exist_pclass(dst_pclass, name)) {
        
        if (H5P__unregister(dst_pclass, name) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTDELETE, FAIL, "unable to remove property");
    } 

    
    orig_dst_pclass = dst_pclass;
    if (H5P__register(&dst_pclass, name, prop->size, prop->value, prop->create, prop->set, prop->get,
                      prop->encode, prop->decode, prop->del, prop->copy, prop->cmp, prop->close) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTDELETE, FAIL, "unable to remove property");

    
    if (dst_pclass != orig_dst_pclass) {
        H5P_genclass_t *old_dst_pclass; 

        
        if (NULL == (old_dst_pclass = (H5P_genclass_t *)H5I_subst(dst_id, dst_pclass)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to substitute property class in ID");
        assert(old_dst_pclass == orig_dst_pclass);

        
        if (H5P__close_class(old_dst_pclass) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCLOSEOBJ, FAIL,
                        "unable to close original property class after substitution");
    } 

done:
    

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__unregister(H5P_genclass_t *pclass, const char *name)
{
    H5P_genprop_t *prop;                
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(pclass);
    assert(name);

    
    if ((prop = (H5P_genprop_t *)H5SL_search(pclass->props, name)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "can't find property in skip list");

    
    if (H5SL_remove(pclass->props, prop->name) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTDELETE, FAIL, "can't remove property from skip list");

    
    H5P__free_prop(prop);

    
    pclass->nprops--;

    
    pclass->revision = H5P_GET_NEXT_REV;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P_close(H5P_genplist_t *plist)
{
    H5P_genclass_t *tclass;              
    H5SL_t         *seen = NULL;         
    size_t          nseen;               
    bool            has_parent_class;    
    size_t          ndel;                
    H5SL_node_t    *curr_node;           
    H5P_genprop_t  *tmp;                 
    unsigned        make_cb   = 0;       
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    assert(plist);

    
    if (plist->class_init) {
        tclass = plist->pclass;
        while (NULL != tclass) {
            if (NULL != tclass->close_func) {
                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        
                        (tclass->close_func)(plist->plist_id, tclass->close_data);
                    }
                H5_AFTER_USER_CB(FAIL)
            } 

            
            tclass = tclass->parent;
        } 
    }     

    
    if ((seen = H5SL_create(H5SL_TYPE_STR, NULL)) == NULL)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, FAIL, "can't create skip list for seen properties");
    nseen = 0;

    
    if (H5SL_count(plist->props) > 0) {
        curr_node = H5SL_first(plist->props);
        while (curr_node != NULL) {
            
            tmp = (H5P_genprop_t *)H5SL_item(curr_node);

            
            if (tmp->close) {
                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        
                        (tmp->close)(tmp->name, tmp->size, tmp->value);
                    }
                H5_AFTER_USER_CB(FAIL)
            } 

            
            if (H5SL_insert(seen, tmp->name, tmp->name) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into seen skip list");
            nseen++;

            
            curr_node = H5SL_next(curr_node);
        } 
    }     

    
    ndel = H5SL_count(plist->del);

    
    tclass           = plist->pclass;
    has_parent_class = (bool)(tclass != NULL && tclass->parent != NULL && tclass->parent->nprops > 0);
    while (tclass != NULL) {
        if (tclass->nprops > 0) {
            
            curr_node = H5SL_first(tclass->props);
            while (curr_node != NULL) {
                
                tmp = (H5P_genprop_t *)H5SL_item(curr_node);

                
                if ((nseen == 0 || H5SL_search(seen, tmp->name) == NULL) &&
                    (ndel == 0 || H5SL_search(plist->del, tmp->name) == NULL)) {

                    
                    if (tmp->close) {
                        void *tmp_value; 

                        
                        if (NULL == (tmp_value = H5MM_malloc(tmp->size)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                        "memory allocation failed for temporary property value");
                        H5MM_memcpy(tmp_value, tmp->value, tmp->size);

                        
                        H5_BEFORE_USER_CB(FAIL)
                            {
                                
                                (tmp->close)(tmp->name, tmp->size, tmp_value);
                            }
                        H5_AFTER_USER_CB(FAIL)

                        
                        H5MM_xfree(tmp_value);
                    } 

                    
                    if (has_parent_class) {
                        if (H5SL_insert(seen, tmp->name, tmp->name) < 0)
                            HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL,
                                        "can't insert property into seen skip list");
                        nseen++;
                    } 
                }     

                
                curr_node = H5SL_next(curr_node);
            } 
        }     

        
        tclass = tclass->parent;
    } 

    
    if (H5P__access_class(plist->pclass, H5P_MOD_DEC_LST) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "Can't decrement class ref count");

    
    H5SL_close(seen);
    seen = NULL;

    
    H5SL_destroy(plist->del, H5P__free_del_name_cb, NULL);

    
    H5SL_destroy(plist->props, H5P__free_prop_cb, &make_cb);

    
    plist = H5FL_FREE(H5P_genplist_t, plist);

done:
    
    if (seen != NULL)
        H5SL_close(seen);

    FUNC_LEAVE_NOAPI(ret_value)
} 

char *
H5P_get_class_name(H5P_genclass_t *pclass)
{
    char *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    assert(pclass);

    
    ret_value = H5MM_xstrdup(pclass->name);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

char *
H5P__get_class_path(H5P_genclass_t *pclass)
{
    char *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(pclass);

    
    if (pclass->parent != NULL) {
        char *par_path; 

        
        par_path = H5P__get_class_path(pclass->parent);
        if (par_path != NULL) {
            size_t ret_str_len;

            
            ret_str_len = strlen(par_path) + strlen(pclass->name) + 1 +
                          3; 
            if (NULL == (ret_value = (char *)H5MM_malloc(ret_str_len)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for class name");

            
            snprintf(ret_value, ret_str_len, "%s/%s", par_path, pclass->name);

            
            H5MM_xfree(par_path);
        } 
        else
            ret_value = H5MM_xstrdup(pclass->name);
    } 
    else
        ret_value = H5MM_xstrdup(pclass->name);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5P_genclass_t *
H5P__open_class_path(const char *path)
{
    char             *tmp_path = NULL;  
    char             *curr_name;        
    char             *delimit;          
    H5P_genclass_t   *curr_class;       
    H5P_check_class_t check_info;       
    H5P_genclass_t   *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(path);

    
    tmp_path = H5MM_xstrdup(path);
    assert(tmp_path);

    
    curr_name  = tmp_path;
    curr_class = NULL;
    while (NULL != (delimit = strchr(curr_name, '/'))) {
        
        *delimit = '\0';

        
        check_info.parent    = curr_class;
        check_info.name      = curr_name;
        check_info.new_class = NULL;

        
        if (H5I_iterate(H5I_GENPROP_CLS, H5P__open_class_path_cb, &check_info, false) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_BADITER, NULL, "can't iterate over classes");
        else if (NULL == check_info.new_class)
            HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, NULL, "can't locate class");

        
        curr_class = check_info.new_class;
        curr_name  = delimit + 1;
    } 

    

    
    check_info.parent    = curr_class;
    check_info.name      = curr_name;
    check_info.new_class = NULL;

    
    if (H5I_iterate(H5I_GENPROP_CLS, H5P__open_class_path_cb, &check_info, false) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADITER, NULL, "can't iterate over classes");
    else if (NULL == check_info.new_class)
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, NULL, "can't locate class");

    
    if (NULL == (ret_value = H5P__copy_pclass(check_info.new_class)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, NULL, "can't copy property class");

done:
    
    H5MM_xfree(tmp_path);

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5P_genclass_t *
H5P__get_class_parent(const H5P_genclass_t *pclass)
{
    H5P_genclass_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(pclass);

    
    ret_value = pclass->parent;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__close_class(H5P_genclass_t *pclass)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(pclass);

    
    if (H5P__access_class(pclass, H5P_MOD_DEC_REF) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "can't decrement ID ref count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5P__new_plist_of_type(H5P_plist_type_t type)
{
    H5P_genclass_t *pclass;                      
    hid_t           class_id;                    
    hid_t           ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_PACKAGE

    
    HDcompile_assert(H5P_TYPE_REFERENCE_ACCESS == (H5P_TYPE_MAX_TYPE - 1));
    assert(type >= H5P_TYPE_USER && type <= H5P_TYPE_REFERENCE_ACCESS);

    
    if (type == H5P_TYPE_USER)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, H5I_INVALID_HID, "can't create user property list");
    if (type == H5P_TYPE_ROOT)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, H5I_INVALID_HID,
                    "shouldn't be creating root class property list");

    
    switch (type) {
        case H5P_TYPE_OBJECT_CREATE:
            class_id = H5P_CLS_OBJECT_CREATE_ID_g;
            break;

        case H5P_TYPE_FILE_CREATE:
            class_id = H5P_CLS_FILE_CREATE_ID_g;
            break;

        case H5P_TYPE_FILE_ACCESS:
            class_id = H5P_CLS_FILE_ACCESS_ID_g;
            break;

        case H5P_TYPE_DATASET_CREATE:
            class_id = H5P_CLS_DATASET_CREATE_ID_g;
            break;

        case H5P_TYPE_DATASET_ACCESS:
            class_id = H5P_CLS_DATASET_ACCESS_ID_g;
            break;

        case H5P_TYPE_DATASET_XFER:
            class_id = H5P_CLS_DATASET_XFER_ID_g;
            break;

        case H5P_TYPE_FILE_MOUNT:
            class_id = H5P_CLS_FILE_MOUNT_ID_g;
            break;

        case H5P_TYPE_GROUP_CREATE:
            class_id = H5P_CLS_GROUP_CREATE_ID_g;
            break;

        case H5P_TYPE_GROUP_ACCESS:
            class_id = H5P_CLS_GROUP_ACCESS_ID_g;
            break;

        case H5P_TYPE_DATATYPE_CREATE:
            class_id = H5P_CLS_DATATYPE_CREATE_ID_g;
            break;

        case H5P_TYPE_DATATYPE_ACCESS:
            class_id = H5P_CLS_DATATYPE_ACCESS_ID_g;
            break;

        case H5P_TYPE_MAP_CREATE:
            class_id = H5P_CLS_MAP_CREATE_ID_g;
            break;

        case H5P_TYPE_MAP_ACCESS:
            class_id = H5P_CLS_MAP_ACCESS_ID_g;
            break;

        case H5P_TYPE_STRING_CREATE:
            class_id = H5P_CLS_STRING_CREATE_ID_g;
            break;

        case H5P_TYPE_ATTRIBUTE_CREATE:
            class_id = H5P_CLS_ATTRIBUTE_CREATE_ID_g;
            break;

        case H5P_TYPE_ATTRIBUTE_ACCESS:
            class_id = H5P_CLS_ATTRIBUTE_ACCESS_ID_g;
            break;

        case H5P_TYPE_OBJECT_COPY:
            class_id = H5P_CLS_OBJECT_COPY_ID_g;
            break;

        case H5P_TYPE_LINK_CREATE:
            class_id = H5P_CLS_LINK_CREATE_ID_g;
            break;

        case H5P_TYPE_LINK_ACCESS:
            class_id = H5P_CLS_LINK_ACCESS_ID_g;
            break;

        case H5P_TYPE_VOL_INITIALIZE:
            class_id = H5P_CLS_VOL_INITIALIZE_ID_g;
            break;

        case H5P_TYPE_REFERENCE_ACCESS:
            class_id = H5P_CLS_REFERENCE_ACCESS_ID_g;
            break;

        case H5P_TYPE_USER: 
        case H5P_TYPE_ROOT:
        case H5P_TYPE_MAX_TYPE:
        default:
            HGOTO_ERROR(H5E_PLIST, H5E_BADRANGE, FAIL, "invalid property list type: %u\n", (unsigned)type);
    } 

    
    if (NULL == (pclass = (H5P_genclass_t *)H5I_object(class_id)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, H5I_INVALID_HID, "not a property class");

    
    if ((ret_value = H5P_create_id(pclass, true)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, H5I_INVALID_HID, "unable to create property list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5P_get_plist_id(const H5P_genplist_t *plist)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(plist);

    FUNC_LEAVE_NOAPI(plist->plist_id)
} 

H5P_genclass_t *
H5P_get_class(const H5P_genplist_t *plist)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(plist);

    FUNC_LEAVE_NOAPI(plist->pclass)
} 

int
H5P_ignore_cmp(const void H5_ATTR_UNUSED *val1, const void H5_ATTR_UNUSED *val2, size_t H5_ATTR_UNUSED size)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    FUNC_LEAVE_NOAPI(0)
} 
