/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HLmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5HLpkg.h"     
#include "H5MFprivate.h" 

H5FL_DEFINE_STATIC(H5HL_dblk_t);

H5HL_dblk_t *
H5HL__dblk_new(H5HL_t *heap)
{
    H5HL_dblk_t *dblk      = NULL; 
    H5HL_dblk_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(heap);

    
    if (NULL == (dblk = H5FL_CALLOC(H5HL_dblk_t)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "memory allocation failed for local heap data block");

    
    if (FAIL == H5HL__inc_rc(heap))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, NULL, "can't increment heap ref. count");

    
    dblk->heap = heap;
    heap->dblk = dblk;

    
    ret_value = dblk;

done:
    
    if (!ret_value && dblk != NULL)
        
        dblk = H5FL_FREE(H5HL_dblk_t, dblk);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HL__dblk_dest(H5HL_dblk_t *dblk)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblk);

    
    if (dblk->heap) {
        
        dblk->heap->dblk = NULL;

        
        if (FAIL == H5HL__dec_rc(dblk->heap))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't decrement heap ref. count");

        
        dblk->heap = NULL;
    }

done:
    
    
    dblk = H5FL_FREE(H5HL_dblk_t, dblk);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HL__dblk_realloc(H5F_t *f, H5HL_t *heap, size_t new_heap_size)
{
    H5HL_dblk_t *dblk;          
    haddr_t      old_addr;      
    haddr_t      new_addr;      
    size_t       old_heap_size; 
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(heap);
    assert(new_heap_size > 0);

    
    old_addr      = heap->dblk_addr;
    old_heap_size = heap->dblk_size;
    H5_CHECK_OVERFLOW(old_heap_size, size_t, hsize_t);
    if (FAIL == H5MF_xfree(f, H5FD_MEM_LHEAP, old_addr, (hsize_t)old_heap_size))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "can't free old local heap data");

    
    H5_CHECK_OVERFLOW(new_heap_size, size_t, hsize_t);
    if (HADDR_UNDEF == (new_addr = H5MF_alloc(f, H5FD_MEM_LHEAP, (hsize_t)new_heap_size)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "unable to allocate file space for local heap");

    
    heap->dblk_addr = new_addr;
    heap->dblk_size = new_heap_size;

    
    if (H5_addr_eq(old_addr, new_addr)) {
        
        if (heap->single_cache_obj) {
            
            assert(H5_addr_eq(heap->prfx_addr + heap->prfx_size, old_addr));
            assert(heap->prfx);

            
            if (FAIL == H5AC_resize_entry(heap->prfx, (size_t)(heap->prfx_size + new_heap_size)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "unable to resize heap in cache");
        }
        else {
            
            assert(H5_addr_ne(heap->prfx_addr + heap->prfx_size, old_addr));
            assert(heap->dblk);

            
            if (H5AC_resize_entry(heap->dblk, (size_t)new_heap_size) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "unable to resize heap (data block) in cache");
        }
    }
    else {
        
        if (heap->single_cache_obj) {
            
            if (NULL == (dblk = H5HL__dblk_new(heap)))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "unable to allocate local heap data block");

            
            heap->prfx_size = H5HL_SIZEOF_HDR(f);
            if (FAIL == H5AC_resize_entry(heap->prfx, (size_t)heap->prfx_size))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "unable to resize heap prefix in cache");

            
            if (FAIL == H5AC_insert_entry(f, H5AC_LHEAP_DBLK, new_addr, dblk, H5AC__PIN_ENTRY_FLAG))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "unable to cache local heap data block");

            dblk = NULL;

            
            heap->single_cache_obj = false;
        }
        else {
            
            
            if (FAIL == H5AC_resize_entry(heap->dblk, (size_t)new_heap_size))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "unable to resize heap data block in cache");

            
            if (FAIL == H5AC_move_entry(f, H5AC_LHEAP_DBLK, old_addr, new_addr))
                HGOTO_ERROR(H5E_HEAP, H5E_CANTMOVE, FAIL, "unable to move heap data block in cache");
        }
    }

done:
    
    if (FAIL == ret_value) {
        heap->dblk_addr = old_addr;
        heap->dblk_size = old_heap_size;
    }
    FUNC_LEAVE_NOAPI(ret_value)
} 
