/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FApkg.h"     
#include "H5VMprivate.h" 

herr_t
H5FA__hdr_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth, const H5FA_class_t *cls,
                haddr_t obj_addr)
{
    
    H5FA_hdr_t *hdr       = NULL;    
    void       *dbg_ctx   = NULL;    
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(H5_addr_defined(obj_addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);
    assert(cls);

    
    if (cls->crt_dbg_ctx)
        
        if (NULL == (dbg_ctx = cls->crt_dbg_ctx(f, obj_addr)))
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTGET, FAIL, "unable to create fixed array debugging context");

    
    if (NULL == (hdr = H5FA__hdr_protect(f, addr, dbg_ctx, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, FAIL, "unable to load fixed array header");

    
    Rfprintf(stream, "%*sFixed Array Header...\n", indent, "");

    
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Array class ID:", hdr->cparam.cls->name);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, "Header size:", (unsigned long long)hdr->size);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Raw Element Size:", (unsigned)hdr->cparam.raw_elmt_size);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
            "Native Element Size (on this platform):", (unsigned long long)hdr->cparam.cls->nat_elmt_size);

    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Max. # of elements in data block page:",
            (unsigned)((size_t)1 << hdr->cparam.max_dblk_page_nelmts_bits));

    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Number of elements in Fixed Array:", hdr->stats.nelmts);

    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "Fixed Array Data Block Address:", hdr->dblk_addr);

done:
    if (dbg_ctx && cls->dst_dbg_ctx(dbg_ctx) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTRELEASE, FAIL, "unable to release fixed array debugging context");
    if (hdr && H5FA__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release fixed array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__dblock_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth, const H5FA_class_t *cls,
                   haddr_t hdr_addr, haddr_t obj_addr)
{
    
    H5FA_hdr_t    *hdr     = NULL;      
    H5FA_dblock_t *dblock  = NULL;      
    void          *dbg_ctx = NULL;      
    size_t         u;                   
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);
    assert(cls);
    assert(H5_addr_defined(hdr_addr));
    assert(H5_addr_defined(obj_addr));

    
    if (cls->crt_dbg_ctx)
        
        if (NULL == (dbg_ctx = cls->crt_dbg_ctx(f, obj_addr)))
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTGET, FAIL, "unable to create fixed array debugging context");

    
    if (NULL == (hdr = H5FA__hdr_protect(f, hdr_addr, dbg_ctx, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, FAIL, "unable to load fixed array header");

    
    if (NULL == (dblock = H5FA__dblock_protect(hdr, addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect fixed array data block, address = %llu", (unsigned long long)addr);

    
    Rfprintf(stream, "%*sFixed Array data Block...\n", indent, "");

    
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Array class ID:", hdr->cparam.cls->name);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth, "Address of Data Block:", dblock->addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth, "Data Block size:", dblock->size);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Number of elements in Data Block:", hdr->cparam.nelmts);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, "Number of pages in Data Block:", (unsigned long long)dblock->npages);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
            "Number of elements per Data Block page:", (unsigned long long)dblock->dblk_page_nelmts);

    if (dblock->npages) {         
        size_t  dblk_page_nelmts; 
        haddr_t dblk_page_addr;   
        size_t  page_idx;         

        Rfprintf(stream, "%*sPaging:\n", indent, "");

        
        dblk_page_addr   = dblock->addr + H5FA_DBLOCK_PREFIX_SIZE(dblock);
        dblk_page_nelmts = dblock->dblk_page_nelmts;

        
        for (page_idx = 0; page_idx < dblock->npages; page_idx++) {
            if (!H5VM_bit_get(dblock->dblk_page_init, page_idx)) {
                Rfprintf(stream, "%*s%-*s %llu %s\n", indent, "", fwidth, "Page %llu:", (unsigned long long)page_idx, "empty");

            }                                  
            else {                             
                H5FA_dblk_page_t *dblk_page;   
                hsize_t           nelmts_left; 

                
                if (((page_idx + 1) == dblock->npages) &&
                    (nelmts_left = hdr->cparam.nelmts % dblock->dblk_page_nelmts))
                    dblk_page_nelmts = (size_t)nelmts_left;

                if (NULL == (dblk_page = H5FA__dblk_page_protect(hdr, dblk_page_addr, dblk_page_nelmts,
                                                                 H5AC__READ_ONLY_FLAG)))
                    HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, FAIL,
                                "unable to protect fixed array data block page, address = %llu",
                                (unsigned long long)dblk_page_addr);

                Rfprintf(stream, "%*sElements in page %llu:\n", indent, "", (unsigned long long)page_idx);
                for (u = 0; u < dblk_page_nelmts; u++) {
                    
                    if ((hdr->cparam.cls->debug)(stream, (indent + 3), MAX(0, (fwidth - 3)), (hsize_t)u,
                                                 ((uint8_t *)dblk_page->elmts) +
                                                     (hdr->cparam.cls->nat_elmt_size * u)) < 0)
                        HGOTO_ERROR(H5E_FARRAY, H5E_CANTGET, FAIL, "can't get element for debugging");
                } 
                if (H5FA__dblk_page_unprotect(dblk_page, H5AC__NO_FLAGS_SET) < 0)
                    HGOTO_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL,
                                "unable to release fixed array data block page");

                
                dblk_page_addr += dblock->dblk_page_size;
            } 
        }     
    }         
    else {    
        
        Rfprintf(stream, "%*sElements:\n", indent, "");
        for (u = 0; u < hdr->cparam.nelmts; u++) {
            
            if ((hdr->cparam.cls->debug)(stream, (indent + 3), MAX(0, (fwidth - 3)), (hsize_t)u,
                                         ((uint8_t *)dblock->elmts) + (hdr->cparam.cls->nat_elmt_size * u)) <
                0)
                HGOTO_ERROR(H5E_FARRAY, H5E_CANTGET, FAIL, "can't get element for debugging");
        } 
    }     

done:
    if (dbg_ctx && cls->dst_dbg_ctx(dbg_ctx) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTRELEASE, FAIL, "unable to release fixed array debugging context");
    if (dblock && H5FA__dblock_unprotect(dblock, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release fixed array data block");
    if (hdr && H5FA__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release fixed array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 
