% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Tlasso.fit.R
\name{Tlasso.fit}
\alias{Tlasso.fit}
\title{Non-Convex Optimization for Sparse Tensor Graphical Models}
\usage{
Tlasso.fit(data, T = 1, lambda.vec = NULL, norm.type = 2, thres = 1e-05)
}
\arguments{
\item{data}{tensor object stored in a m1 * m2 * ... * mK * n array, where n is sample size
and mk is dimension of the kth tensor mode.}

\item{T}{number of maximal iteration, default is 1. Each iteration involves update on all modes. 
If output change less than \code{thres} after certain iteration, in terms of summation on Frobenius norm, this function will be terminated (before Tth iteration).}

\item{lambda.vec}{vector of tuning parameters (\eqn{\lambda_1}{\lambda_1},...,\eqn{\lambda_K}{\lambda_K}). Defalut is NULL, s.t. it is tuned via \code{HUGE} package directly.}

\item{norm.type}{normalization method of precision matrix, i.e., \eqn{\Omega_{11} = 1}{\Omega_{11}=1} 
if norm.type = 1 and \eqn{\|\Omega\|_{F}=1}{||\Omega||_F =1 } if norm.type = 2. Default value is 2.}

\item{thres}{thresholding value that terminates algorithm before Tth iteration if output change less than \code{thres} after certain iteration, in terms of summation over Frobenius norm. 
If \code{thres} is negative or zero, this algorithm will iterate T times.}
}
\value{
A length-K list of estimation of precision matrices.
}
\description{
An alternating optimization algorithm for estimation of precision matrices of sparse tensor graphical models. See \href{https://arxiv.org/abs/1609.04522}{Lyu et al. (2019)} for details.
}
\details{
This function conducts an alternating optimization algorithm to sparse tensor graphical model. The output is optimal consistent even when \code{T=1}, see \href{https://arxiv.org/abs/1609.04522}{Lyu et al. (2019)} for details.
There are two ternimation criteria, \code{T} and \code{thres}. Algorithm will be terminated if output in certain iteration change less than \code{thres}. Otherwise, T iterations will be fully operated.
}
\examples{

m.vec = c(5,5,5)  # dimensionality of a tensor 
n = 5   # sample size 
lambda.thm = 20*c( sqrt(log(m.vec[1])/(n*prod(m.vec))), 
                  sqrt(log(m.vec[2])/(n*prod(m.vec))), 
                  sqrt(log(m.vec[3])/(n*prod(m.vec))))
DATA=Trnorm(n,m.vec,type='Chain') 
# obersavations from tensor normal distribution
out.tlasso = Tlasso.fit(DATA,T=10,lambda.vec = lambda.thm,thres=10)   
# terminate by thres
out.tlasso = Tlasso.fit(DATA,T=3,lambda.vec = lambda.thm,thres=0)   
# thres=0, iterate 10 times


}
\seealso{
\code{\link{varcor}}, \code{\link{biascor}}, \code{\link{huge}}
}
\author{
Xiang Lyu, Will Wei Sun, Zhaoran Wang, Han Liu, Jian Yang, Guang Cheng.
}
