## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## -----------------------------------------------------------------------------
library(dplyr)
library(ViroReportR)
library(ggplot2)
library(purrr)
library(tidyr)

## -----------------------------------------------------------------------------
data <- simulate_data(days=365, #days spanning simulation
                      peaks = c("flu_a"=90,"rsv"=110,"sars_cov2"=160), #peak day for each disease
                      amplitudes=c("flu_a"=50,"rsv"=40,"sars_cov2"=20), #amplitude of peak for each disease
                      scales = c("flu_a"=-0.004,"rsv"=-0.005,"sars_cov2"=-0.001), # spread of peak for each disease
                      time_offset = 0, #number of days to offset start of simulation
                      noise_sd = 5, #noise term
                      start_date = "2024-01-07" #starting day (Sunday)
                      )

## -----------------------------------------------------------------------------
data$date <- lubridate::ymd(data$date)

vri_data <- data %>% 
            pivot_longer(cols = -date,  # all columns except 'date'
                          names_to = "disease_type",
                          values_to = "confirm") # <-- need ot be called confirm

# VRI data set-up
vri_name_list <- vri_data %>% 
    dplyr::group_by(disease_type) %>% 
  dplyr::group_keys() %>% dplyr::pull()


vri_data_list <- vri_data %>% 
  dplyr::group_by(disease_type) %>% 
  dplyr::group_map(~.x)

names(vri_data_list) <- vri_name_list


## -----------------------------------------------------------------------------
#parameters set-up
start_date <- min(vri_data$date) + 13
forecast_horizon <- 7
smooth <- FALSE
validate_window_size <- 7

forecasts_results <- tibble(
  vri_data_list,
  forecasts = map2(
    vri_data_list,
    vri_name_list,
    ~ generate_forecast(
      data = .x,
      smooth_data = smooth,
      type = .y,
      n_days = forecast_horizon,
      start_date = start_date
    )
  )
)

names(forecasts_results$forecasts) <- vri_name_list
names(forecasts_results$vri_data_list) <- vri_name_list

## ----validation-plots, echo=FALSE, results='asis'-----------------------------
validation_results <- tibble(
  vri_data_list,
  validation = map2(
    vri_data_list,
    vri_name_list,
    ~ generate_validation(
      data = .x,
      type = .y,
      n_days = forecast_horizon,
      start_date = min(.x$date, na.rm = TRUE),
      validate_window_size = validate_window_size,
      window_size = 7,
      smooth_data = smooth,
      smoothing_cutoff = 10
    )
  )
)
names(validation_results$validation) <- vri_name_list
names(validation_results$vri_data_list) <- vri_name_list

for (vri in vri_name_list) {
  cat("\n\n")                               # spacing
  cat(glue::glue("#### {vri}\n\n"))         # subheading for each disease
  print(
    plot_validation(
      data = vri_data_list[[vri]],
      validation_results$validation[[vri]],
      pred_plot = "ribbon"
    )
  )
  cat("\n\n")
}


## ----eval = FALSE-------------------------------------------------------------
# tmp_dir <- tempdir()
# 
# # Save the simulated data
# data_path <- file.path(tmp_dir, "simulated_data.csv")
# write.csv(vri_data, data_path, row.names = FALSE)
# 
# output_path <- tempdir()
# generate_forecast_report(input_dir = data_path,
#                             output_dir = output_path,
#                             n_days = 7,
#                             validate_window_size = 7,
#                             smooth = TRUE)
# 
# cat("Report saved to:", output_path, "\n")
# 
# if (interactive() && file.exists(output_path)) {
#   utils::browseURL(output_path)
# }
# 

