#### Accompanying Code for:
## Partial Least Squares Structural Equation Modeling (PLS-SEM) Using R (Second Edition) - A Workbook (2026)
## Hair, J.F. (Jr), Hult, T.M., Ringle, C.M., Sarstedt, M., Danks, N.P., and Adler, S.

## Chapter 3: Introduction to SEMinR

# Download and install the SEMinR package
# You only need to do this once to equip RStudio with SEMinR and seminrExtras
#install.packages("seminr") # remove the # to execute this line of code
#install.packages("seminrExtras") # remove the # to execute this line of code

# Make the SEMinR library ready to use
# You must do this every time you restart RStudio and wish to use packages
library(seminr)
library(seminrExtras)

# Load the corporate reputation data
corp_rep_data <- corp_rep_data

# Show the first several rows of the corporate reputation data
head(corp_rep_data)

# Create measurement model ----
simple_mm <- constructs(
  composite("COMP", multi_items("comp_", 1:3)),
  composite("LIKE", multi_items("like_", 1:3)),
  composite("CUSA", single_item("cusa")),
  composite("CUSL", multi_items("cusl_", 1:3)))

# Create structural model ----
simple_sm <- relationships(
  paths(from = c("COMP", "LIKE"), to = c("CUSA", "CUSL")),
  paths(from = c("CUSA"), to = c("CUSL")))

# Estimate the model
corp_rep_simple_model <- estimate_pls(data = corp_rep_data,
  measurement_model = simple_mm,
  structural_model  = simple_sm,
  inner_weights = path_weighting,
  missing = mean_replacement,
  missing_value = "-99")

# Estimate the model with default settings
corp_rep_simple <- estimate_pls(data = corp_rep_data,
  measurement_model = simple_mm,
  structural_model  = simple_sm)

## Using the `assess_syntax` argument in `estimate_pls()`
# Specify the measurement model with a bug in the spelling of the construct
# `COMP`. Here it is `COP` and in the SM it is `COMP`.
error_mm <- constructs(
  composite("COP", multi_items("comp_", 1:3)),
  composite("LIKE", multi_items("like_", 1:3)),
  composite("CUSL", multi_items("cusl_", 1:3)),
  composite("CUSA", single_item("cusa")))

# Specify the structural model.
simple_sm <- relationships(
  paths(from = c("COMP", "LIKE"), to = c("CUSA", "CUSL")),
  paths(from = c("CUSA"), to = c("CUSL")))

# Note that I have specified the argument `assess_syntax` as TRUE
# Now estimating the model should output an error
estimate_pls(data = corp_rep_data,
             measurement_model = error_mm,
             structural_model = simple_sm,
             assess_syntax = TRUE)

## Plotting a model for debugging
# Plot the measurement model
plot(error_mm, theme = seminr_theme_academic())

# Plot the structural model
plot(simple_sm, theme = seminr_theme_academic())

# Summarize the model results
summary_simple_corp_rep <- summary(corp_rep_simple_model)

# Inspect the structural paths
summary_simple_corp_rep$paths

# Inspect the construct reliability metrics
summary_simple_corp_rep$reliability

# Bootstrap the model
boot_simple_corp_rep <- bootstrap_model(
  seminr_model = corp_rep_simple_model,
  nboot = 1000,
  cores = NULL,
  seed = 123)

# Store the summary of the bootstrapped model
sum_boot_simple_corp_rep <- summary(boot_simple_corp_rep)

# Inspect the bootstrapped structural paths
sum_boot_simple_corp_rep$bootstrapped_paths

# Inspect the bootstrapped indicator loadings
sum_boot_simple_corp_rep$bootstrapped_loadings

# Write the bootstrapped paths object to csv file
# and you will have to delete the '#' and then execute the below line of code:
# write.csv(x = sum_boot_simple_corp_rep$bootstrapped_loadings, file = "boot_loadings.csv")

# Generate the plot for exporting
plot(summary_simple_corp_rep$reliability)
